# -*- coding: utf-8 -*-
import os

from google.protobuf.json_format import MessageToDict
from kikimr.public.sdk.python.persqueue.auth import (
    CredentialsProvider,
    OAuthTokenCredentialsProvider,
)
from passport.backend.core.logging_utils.helpers import escape_text_for_log
from passport.backend.core.tvm import get_tvm_credentials_manager
from passport.backend.core.ydb_client import YDB_AUTH_TICKET_HEADER
from passport.backend.utils.string import smart_bytes


class TvmManagerCredentialsProvider(CredentialsProvider):
    def __init__(self, tvm_credentials_manager, tvm_alias=None, tvm_client_id=None):
        super(TvmManagerCredentialsProvider, self).__init__()
        self._tvm_alias = tvm_alias
        self._tvm_client_id = tvm_client_id
        if (self._tvm_alias is None) == (self._tvm_client_id is None):
            raise ValueError('Either tvm_alias or tvm_client_id must be provided, but not both')
        self.tvm_credentials_manager = tvm_credentials_manager

    def _get_service_ticket(self):
        if self._tvm_alias is not None:
            ticket = self.tvm_credentials_manager.get_ticket_by_alias(self._tvm_alias)
        else:
            ticket = self.tvm_credentials_manager.get_ticket_by_client_id(self._tvm_client_id)
        return smart_bytes(ticket)

    def auth_metadata(self):
        return [
            (
                YDB_AUTH_TICKET_HEADER, self._get_service_ticket(),
            ),
        ]

    def _build_protobuf(self):
        self._cached_proto.tvm_service_ticket = self._get_service_ticket()


def create_credentials(config):
    cred_type = config.get('credentials_type')
    if cred_type == 'oauth_token':
        if 'LB_TOKEN' not in os.environ:
            raise ValueError('Logbroker OAuth token in env "LB_TOKEN" is missing')
        cred = OAuthTokenCredentialsProvider(os.environ['LB_TOKEN'])
    elif cred_type == 'tvm':
        cred = TvmManagerCredentialsProvider(
            tvm_credentials_manager=get_tvm_credentials_manager(),
            tvm_alias=config.get('tvm_alias'),
            tvm_client_id=config.get('tvm_client_id'),
        )
    else:
        raise ValueError('Unsupported credentials type {}'.format(cred_type))
    return cred


def format_protobuf(protobuf):
    return str(MessageToDict(protobuf))


def format_protobuf_safe(protobuf):
    try:
        return format_protobuf(protobuf)
    except Exception as err:
        return u'{} <!PB format error {}!>'.format(escape_text_for_log(protobuf), err)
