# -*- coding: utf-8 -*-
"""Модули для тестирования программ, использующих TskvLogger"""

from copy import deepcopy
from datetime import datetime
from functools import partial
import json
from pprint import pformat

import mock
from nose.tools import (
    assert_true,
    eq_,
)
from passport.backend.core.conf import settings
from passport.backend.core.test.test_utils import (
    iterdiff,
    pseudo_diff,
    single_entrant_patch,
)
from passport.backend.core.test.time_utils.time_utils import (
    DatetimeNow,
    TimeNow,
    TimeSpan,
)
from passport.backend.utils.common import merge_dicts
from passport.backend.core.logging_utils.helpers import mask_sessionid
from six import iteritems


BASE_STATBOX_TEMPLATES = {
    'base': {
        'tskv_format': 'passport-log',
        'unixtime': TimeNow,
        'timestamp': lambda: DatetimeNow(convert_to_datetime=True),
        'timezone': '+0300',
        'py': '1',
    },
    'account_created': {
        'action': 'account_created',
        'uid': '1',
        'login': 'test-login',
        'is_suggested_login': '0',
        'country': 'ru',
        'password_quality': '80',
        'karma': '0',
        'suggest_generation_number': '0',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
        'track_id': '1234567890abcdef',
    },
    'frodo_karma': {
        'destination': 'frodo',
        'event': 'account_modification',
        'entity': 'karma',
        'action': 'karma',
        'registration_datetime': None,
        'login': '-',
        'old': '-',
        'new': '0',
        'ip': '127.0.0.1',
        'uid': '1',
        'suid': '-',
    },
    'account_modification': {
        'event': 'account_modification',
        'uid': '1',
        'ip': '127.0.0.1',
        'user_agent': '-',
    },
    'subscriptions': {
        'event': 'account_modification',
        'operation': 'created',
        'uid': '1',
        'ip': '127.0.0.1',
        'entity': 'subscriptions',
        'sid': '1',
        'user_agent': '-',
    },
    'aliases': {
        'event': 'account_modification',
        'operation': 'created',
        'uid': '1',
        'ip': '127.0.0.1',
        'user_agent': '-',
        'entity': 'aliases',
        'type': None,
    },
    'check_cookies': {
        'mode': 'check_cookies',
        'host': 'passport-test.yandex.ru',
        'consumer': 'dev',
        'have_sessguard': '0',
        'sessionid': mask_sessionid('sessionid'),
    },
    'global_logout': {
        'event': 'account_modification',
        'uid': '1',
        'entity': 'account.global_logout_datetime',
        'operation': 'updated',
        'old': partial(datetime.fromtimestamp(1).strftime, '%Y-%m-%d %H:%M:%S'),
        'new': partial(DatetimeNow, convert_to_datetime=True),
        'ip': '127.0.0.1',
        'user_agent': 'curl',
        'consumer': 'dev',
    },
    'cookie_set': {
        'authid': '123:1422501443:126',
        'uid': '1',
        'input_login': 'test',
        'mode': 'any_auth',
        'action': 'cookie_set',
        'cookie_version': str(settings.BLACKBOX_SESSION_VERSION),
        'ttl': '5',
        'ip': '127.0.0.1',
        'captcha_passed': '0',
        'session_method': 'create',
        'uids_count': '1',
    },
    'simple_bind_operation_created': {
        'action': 'phone_operation_created',
        'operation_type': 'simple_bind',
        'operation_id': '1',
        'phone_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'secure_bind_operation_created': {
        'action': 'phone_operation_created',
        'operation_type': 'secure_bind',
        'operation_id': '1',
        'phone_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'replace_secure_phone_with_bound_phone_operation_created': {
        'action': 'phone_operation_created',
        'operation_type': 'replace_secure_phone_with_bound_phone',
        'operation_id': '1',
        'simple_phone_id': '1',
        'simple_number': '+79123******',
        'secure_phone_id': '2',
        'secure_number': '+79231******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'aliasify_secure_operation_created': {
        'action': 'phone_operation_created',
        'operation_type': 'aliasify_secure',
        'operation_id': '1',
        'phone_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'dealiasify_secure_operation_created': {
        'action': 'phone_operation_created',
        'operation_type': 'dealiasify_secure',
        'operation_id': '1',
        'phone_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'securify_operation_created': {
        'action': 'phone_operation_created',
        'operation_type': 'securify',
        'operation_id': '1',
        'phone_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'mark_operation_created': {
        'action': 'phone_operation_created',
        'operation_type': 'mark',
        'operation_id': '1',
        'phone_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'remove_secure_operation_created': {
        'action': 'phone_operation_created',
        'operation_type': 'remove_secure',
        'operation_id': '1',
        'phone_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'simple_phone_bound': {
        'action': 'simple_phone_bound',
        'phone_id': '1',
        'operation_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '1.2.3.4',
        'user_agent': 'curl',
    },
    'secure_phone_bound': {
        'action': 'secure_phone_bound',
        'phone_id': '1',
        'operation_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phone_secured': {
        'action': 'phone_secured',
        'phone_id': '1',
        'operation_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'secure_phone_replaced': {
        'action': 'secure_phone_replaced',
        'operation_id': '1',
        'new_secure_number': '+79123******',
        'new_secure_phone_id': '1',
        'old_secure_number': '+79231******',
        'old_secure_phone_id': '2',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'secure_phone_removed': {
        'action': 'secure_phone_removed',
        'phone_id': '1',
        'operation_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phone_confirmed': {
        'action': 'phone_confirmed',
        'phone_id': '1',
        'operation_id': '1',
        'number': '+79123******',
        'confirmation_time': partial(DatetimeNow, convert_to_datetime=True),
        'code_checks_count': '1',
        'uid': '1',
        'ip': '127.0.0.1',
        'consumer': 'dev',
        'user_agent': 'curl',
    },
    'phone_unbound': {
        'action': 'phone_unbound',
        'number': '+79123******',
        'phone_id': '1',
        'uid': '1',
        'ip': '127.0.0.1',
        'consumer': 'dev',
        'user_agent': 'curl',
    },
    'code_sent': {
        'action': 'code_sent',
        'operation_id': '1',
        'number': '+79123******',
        'sms_count': '1',
        'sms_id': '1',
        'uid': '1',
        'ip': '127.0.0.1',
        'consumer': 'dev',
        'user_agent': 'curl',
    },
    'phone_operation_replaced': {
        'action': 'phone_operation_replaced',
        'phone_id': '1',
        'old_operation_id': '2',
        'old_operation_type': 'simple_bind',
        'operation_id': '1',
        'number': '+79123******',
        'operation_type': 'secure_bind',
        'uid': '1',
        'ip': '127.0.0.1',
        'consumer': 'dev',
        'user_agent': 'curl',
    },
    'phone_operation_applied': {
        'action': 'phone_operation_applied',
        'operation_id': '1',
        'phone_id': '1',
        'number': '+79123******',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phone_operation_cancelled': {
        'action': 'phone_operation_cancelled',
        'operation_type': 'simple_bind',
        'operation_id': '1',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phonenumber_alias_given_out': {
        'operation': 'aliasify',
        'number': '+79123******',
        'is_owner_changed': '0',
        'validation_period': '0',
        'uid': '1',
        'login': 'test-login',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phonenumber_alias_subscription_added': {
        'event': 'account_modification',
        'operation': 'added',
        'entity': 'subscriptions',
        'uid': '1',
        'sid': '65',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phonenumber_alias_taken_away': {
        'operation': 'dealiasify',
        'number': '+79123******',
        'reason': 'owner_change',
        'uid': '1',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phonenumber_alias_subscription_removed': {
        'event': 'account_modification',
        'operation': 'removed',
        'entity': 'subscriptions',
        'uid': '1',
        'sid': '65',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phonenumber_alias_search_enabled': {
        'event': 'account_modification',
        'operation': 'created',
        'entity': 'phonenumber_alias.enable_search',
        'uid': '1',
        'new': '1',
        'old': '-',
        'consumer': 'dev',
        'ip': '127.0.0.1',
        'user_agent': 'curl',
    },
    'phonenumber_alias_added': {
        'event': 'account_modification',
        'operation': 'added',
        'uid': '1',
        'ip': '127.0.0.1',
        'user_agent': '-',
        'entity': 'aliases',
        'type': '11',
    },
    'phonenumber_alias_removed': {
        'event': 'account_modification',
        'operation': 'removed',
        'uid': '1',
        'ip': '127.0.0.1',
        'user_agent': '-',
        'entity': 'aliases',
        'type': '11',
    },
    'phonenumber_alias_updated': {
        'event': 'account_modification',
        'operation': 'updated',
        'uid': '1',
        'ip': '127.0.0.1',
        'user_agent': '-',
        'entity': 'aliases',
        'type': '11',
    },
}


BASE_CRYPTASTAT_TEMPLATES = {
    'base': {
        'tskv_format': 'passport-log',
        'unixtime': TimeNow,
        'py': '1',
    },
    'account_modification': {
        'event': 'account_modification',
        'uid': '1',
        'ip': '127.0.0.1',
        'user_agent': '-',
    },
}

BASE_GRAPHITE_TEMPLATES = {
    'base': {
        'tskv_format': 'passport-log',
        'unixtime': TimeNow,
        # Определим методом позже
        'timestamp': None,
        'duration': partial(TimeSpan, 0),
    },
}

BASE_ACCESS_LOG_TEMPLATES = {
    'base': {
        'tskv_format': 'passport-api-access-log',
    },
}


BASE_AVATARS_LOG_TEMPLATES = {
    'base': {
        'tskv_format': 'avatars-log',
    },
}

BASE_FAMILY_LOG_TEMPLATES = {
    'base': {
        'tskv_format': 'passport-family-log',
        'unixtime': TimeNow,
    },
}

BASE_PUSH_LOG_TEMPLATES = {
    'base': {
        'tskv_format': 'passport-push-log',
        'unixtime': TimeNow,
    },
}


@single_entrant_patch
class BaseLoggerFaker(object):
    DATETIME_FORMAT = '%d/%b/%Y:%H:%M:%S'
    logger_class_module = None

    base_entry = None
    entry_templates = None
    equality_checker = None

    templates = None

    def __init__(self):
        if not self.logger_class_module:
            raise NotImplementedError()  # pragma: no cover

        self.write_handler_mock = mock.Mock()

        self._patch = mock.patch(
            self.logger_class_module + '._write_to_log',
            self.write_handler_mock,
        )

        self.base_entry = {}
        self.entry_templates = deepcopy(self.templates or {})
        self.equality_checker = iterdiff(eq_)

    def bind_base(self, **kwargs):
        """
        Установка базового набора параметров, которые будут указываться
        для всех формируемых записей.
        """
        self.base_entry = merge_dicts(
            self.base_entry,
            kwargs,
        )

    def bind_entry(self, tag, _exclude=None, _inherit_from=None, **kwargs):
        """
        Создание шаблона параметров и сохранение его под
        указанным тэгом. Обратите внимание, что сохраняемый
        набор параметров формируется следующим образом:

        Шаблоны[тэг] = БазоваяЗапись + Шаблоны[тэг] + **kwargs.

        Если шаблон с таким тэгом еще не существует,
        то сохраняется переданный набор параметров без изменений.
        """
        if not _exclude:
            _exclude = []

        if not _inherit_from:
            pre_merged = self.entry_templates.get(tag, {})

        elif isinstance(_inherit_from, list):
            templates = [self.entry_templates.get(local_base, {}) for local_base in _inherit_from]
            pre_merged = merge_dicts(*templates)
        else:
            pre_merged = self.entry_templates.get(_inherit_from, {})

        prepared_entry = merge_dicts(
            self.base_entry,
            pre_merged,
            kwargs,
        )
        for field in _exclude:
            prepared_entry.pop(field, None)
        self.entry_templates[tag] = prepared_entry

    def entry(self, tag, _exclude=None, **kwargs):
        """
        Формирование записи в лог, происходит по следующей
        формуле:

        Результат = Шаблоны['base'] + Шаблоны[тэг] + **kwargs.
        """

        result = merge_dicts(
            self.entry_templates.get('base', {}),
            self.entry_templates.get(tag, {}),
            kwargs,
        )
        if _exclude:
            for field in _exclude:
                result.pop(field, None)

        for key, value in iteritems(result):
            if callable(value):
                result[key] = value()

        return result

    def start(self):
        self._patch.start()

    def stop(self):
        self._patch.stop()

    @staticmethod
    def get_unixtime_mock():
        return TimeNow()

    @classmethod
    def get_timestamp_mock(cls):
        return DatetimeNow(
            convert_to_datetime=True,
            format_=cls.DATETIME_FORMAT,
        )

    def _check_contents(self, expected_entries, offset=0, comparator=None):
        """
        Обобщенная функция сравнения содержимого перехваченных
        записей в лог.
        """
        assert_true(
            comparator,
            'No list comparator specified!',
        )

        # Для удобства проверки всего одного элемента преобразуем его в список
        if not isinstance(expected_entries, (list, tuple)):
            expected_entries = [expected_entries]

        # Смотрим только на последние N записей.
        if offset == -1:
            start = -len(expected_entries)
        else:
            start = offset

        comparator(self.get_entries(start), expected_entries)

    def get_entries(self, start=0):
        log_lines = self.write_handler_mock._mock_call_args_list[start:]
        return self._parse_lines(log_lines)

    def strict_ordering_checker(self, contents, expected_records):
        assert_true(
            len(contents) == len(expected_records),
            'Expected exactly %s statbox entries, found %s: %s' % (
                len(expected_records),
                len(contents),
                pseudo_diff(expected_records, contents),
            ),
        )
        for (actual, expected) in zip(contents, expected_records):
            self.equality_checker(actual, expected)

    def fuzzy_ordering_checker(self, contents, expected_records):
        assert_true(
            len(contents) >= len(expected_records),
            'Expected at least %s statbox entries, found %s.' % (
                len(expected_records),
                len(contents),
            ),
        )

        expected_stack = expected_records[:]
        expected_stack.reverse()
        expected = expected_stack[-1]
        for actual in contents:
            try:
                self.equality_checker(actual, expected)
                expected_stack.pop()
                if expected_stack:
                    expected = expected_stack[-1]
                else:
                    break
            except AssertionError:
                pass
        assert_true(
            not expected_stack,
            'Non-matched entries found: {}\n\nActual: {}'.format(
                ',\n'.join([pformat(e) for e in reversed(expected_stack)]),
                ',\n'.join([pformat(e) for e in contents]),
            ),
        )

    def assert_equals(self, expected_entries, offset=0):
        """
        Строгое сравнение содержимого Буфера перехваченных записей в лог
        с N ожидаемых записей.

        Параметр offset может использоваться для пропуска части записей,
        идущих в начале Буфера. Если же параметр offset равен -1,
        то сравнение будет происходить только с N _последних_ записей в Буфере.
        """
        self._check_contents(
            expected_entries,
            offset=offset,
            comparator=self.strict_ordering_checker,
        )

    def assert_contains(self, expected_entries, offset=0):
        """
        Нестрогое ("размытое") сравнение содержимого Буфера перехваченных
        записей в лог с N ожидаемых записей. Отличие от строго сравнения
        заключается в методе проверки: мы не считаем, что наличие какого-либо
        количества записей между двумя ожидаемыми является ошибкой. При этом
        порядок следования ожидаемых записей все равно важен.

        Пример:
        > Буфер:    [A, 1, 2, 3, B, 4 C]
        > Ожидаемые: [A, B, C]
        > Результат: ОК

        > Буфер:    [A, 1, 2, 3, B, 4 C]
        > Ожидаемые: [A, C, B]
        > Результат: Ошибка

        Параметр offset может использоваться для пропуска части записей,
        идущих в начале Буфера. Если же параметр offset равен -1,
        то сравнение будет происходить только с N _последних_ записей в Буфере.
        """
        self._check_contents(
            expected_entries,
            offset=offset,
            comparator=self.fuzzy_ordering_checker,
        )

    def assert_has_written(self, lines):
        """
        Утвержает, что питоновский Logger записал в журнал список lines.

        lines -- список словарей

        Например, чтобы проверить, что logger записал в журнал строки
        tskv foo=1 bar=2
        tskv foo=2 bar=1
        можно вызвать эту подпрограмму так,

        logger.assert_logger_wrote(
            [
                {
                    'foo': 1,
                    'bar': 2,
                },
                {
                    'foo': 2,
                    'bar': 1,
                }
            ],
        )

        """
        self.assert_equals(lines)

    def clear_entries(self):
        self.write_handler_mock.reset_mock()


class TskvLoggerFaker(BaseLoggerFaker):
    """Изолирует TskvLogger"""
    def _parse_lines(self, lines):
        parsed_entries = []
        for line in lines:
            line = line[0][0]
            if line.startswith('tskv\t'):
                line = line[5:]

            fields = [part.split('=', 1) for part in line.strip().split('\t')]
            parsed_entries.append(dict(fields))
        return parsed_entries


class JsonLoggerFaker(BaseLoggerFaker):
    def _parse_lines(self, lines):
        parsed_entries = []
        for line in lines:
            line = line[0][0]
            parsed_entries.append(json.loads(line))
        return parsed_entries


class GraphiteLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.graphite.GraphiteLogger'
    templates = BASE_GRAPHITE_TEMPLATES

    def __init__(self):
        super(GraphiteLoggerFaker, self).__init__()
        self.entry_templates['base']['timestamp'] = self.get_timestamp_mock


class StatboxLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.statbox.StatboxLogger'
    templates = BASE_STATBOX_TEMPLATES


class CryptastatLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.statbox.CryptastatLogger'
    templates = BASE_CRYPTASTAT_TEMPLATES


class PharmaLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.statbox.PharmaLogger'
    templates = {
        'base': {
            'tskv_format': 'pharma',
            'ts': TimeNow,
        },
    }


class AntifraudLoggerFaker(JsonLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.statbox.AntifraudLogger'
    templates = {
        'base': {
            'tskv_format': 'antifraud',
            't': lambda: TimeNow(as_milliseconds=True),
        },
    }


class YasmsPrivateLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.statbox.YasmsPrivateLogger'
    templates = {
        'base': {
            'unixtimef': lambda: TimeNow(),
            'unixtime': lambda: TimeNow(),
            'sms': '1',
        },
    }


class AccessLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.access.AccessLogger'
    templates = BASE_ACCESS_LOG_TEMPLATES


class AvatarsLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.avatars.AvatarsLogger'
    templates = BASE_AVATARS_LOG_TEMPLATES


class FamilyLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.family.FamilyLogger'
    templates = BASE_FAMILY_LOG_TEMPLATES


class PushLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.push.PushLogger'
    templates = BASE_PUSH_LOG_TEMPLATES


class PhoneLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.phone.PhoneLogger'

    @staticmethod
    def get_log_entry(uid, phone_str, yandexuid):
        return {
            'yandexuid': yandexuid,
            'phone': phone_str,
            'uid': str(uid),
            'unixtime': TimeNow(),
            'tskv_format': 'passport-phone-log',
        }


class DummyLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.dummy.DummyLogger'


class SocialBindingLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.social_binding.SocialBindingLogger'

    templates = {
        'base': {
            'timestamp': TskvLoggerFaker.get_timestamp_mock,
            'tskv_format': 'social-binding-log',
            'unixtime': TimeNow,
        },
        'bind_phonish_account_by_track': {
            'action': 'bind_phonish_account_by_track',
            'ip': '-',
            'track_id': '-',
            'uid': '-',
        },
    }


class CredentialsLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.statbox.CredentialsLogger'

    templates = {
        'base': {
            'tskv_format': 'passport-credentials',
            'unixtime': TimeNow,
        },
        'auth': {
            'credential_type': 'cookie',
        },
    }


class AccountModificationLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.statbox.AccountModificationLogger'
    templates = BASE_STATBOX_TEMPLATES


class AccountModificationInfosecLoggerFaker(TskvLoggerFaker):
    logger_class_module = 'passport.backend.core.logging_utils.loggers.statbox.AccountModificationInfosecLogger'
    templates = BASE_STATBOX_TEMPLATES
