# -*- coding: utf-8 -*-
import logging

from passport.backend.core.logging_utils.helpers import (
    encode_header,
    mask_sensitive_fields,
    trim_message,
)
from passport.backend.utils.string import smart_text
from six import iteritems
import ylog


class ExceptionFormatter(logging.Formatter):
    """
    Аналог ylog.ExceptionFormatter, скрывающий секретные данные.
    """

    def __init__(self, full=False):
        self.full = full
        super(ExceptionFormatter, self).__init__(
            fmt=ylog.handlers.FILE_FORMAT,
            datefmt=ylog.handlers.TIME_FORMAT,
        )

    def make_full_request_info(self, record):
        """
        Ожидаем, что в LogRecord могут быть дополнительные поля:
         - request_info - строка с описанием запроса (может отсутствовать)
         - request_id - id запроса (может быть пустым)
         - request_headers - словарь заголовков запроса (может отсутствовать)
         - request_values - словарь параметров запроса (может отсутствовать)
        """
        lines = []
        if hasattr(record, 'request_info'):
            lines.append('Request: %s' % record.request_info)
        if getattr(record, 'request_id', None):
            lines.append('Request id: %s' % record.request_id)
        if hasattr(record, 'request_headers'):
            lines.append(
                'Headers:\n  %s' % '\n  '.join(
                    encode_header(name, value)
                    for name, value in iteritems(mask_sensitive_fields(record.request_headers))
                ),
            )
        if hasattr(record, 'request_values'):
            lines.append('Request data: %s' % mask_sensitive_fields(record.request_values))
        if hasattr(record, 'error_details'):
            lines.append('Error details: %s' % trim_message(record.response_body))
        return lines

    def format(self, record):
        record.message = record.getMessage()
        if self._fmt.find('%(asctime)') >= 0:
            record.asctime = self.formatTime(record, self.datefmt)

        lines = [self._fmt % record.__dict__]

        if self.full:
            lines += self.make_full_request_info(record)
            if record.exc_info is not None:
                lines.append(self.formatException(record.exc_info))
            lines.append('')
        else:
            if record.exc_info is not None:
                exception, description, traceback = record.exc_info
                if traceback is not None:
                    inner = traceback
                    while inner.tb_next:
                        inner = inner.tb_next

                    line_number = inner.tb_lineno
                    f_globals = inner.tb_frame.f_globals
                    module_name = f_globals.get('__name__', '<unknown>')

                    record.message = '%-20s %s:%s %s' % (
                        exception.__name__,
                        module_name,
                        line_number,
                        description,
                    )
                    lines[0] = self._fmt % record.__dict__

        formatted_text = '\n'.join([smart_text(line, errors='replace') for line in lines])
        return formatted_text
