# -*- coding: utf-8 -*-

from copy import deepcopy
import json
import re

from passport.backend.utils import string
from six import string_types
from ticket_parser2 import (
    ServiceTicket,
    UserTicket,
)


MASK_PLACEHOLDER = '*****'

REPLACE_MAP = [
    ('\\', '\\\\'),
    ('\t', '\\t'),
    ('\n', '\\n'),
    ('\r', '\\r'),
    ('\0', '\\0'),
]


def escape_text_for_log(text, errors='strict'):
    text = string.smart_text(text, errors=errors)
    for char, replacement in REPLACE_MAP:
        text = text.replace(char, replacement)
    return text


def mask_dict_values(dict_, path_name):
    """
    Пример,

    d = {
        'foo': {
            'bar': 'spam',
        },
    }
    mask_dict_values(d, u'foo.bar')

    Ответ

    {
        'foo': {
            'bar': '*****',
        },
    }


    Пример,

    d = {
        'fruits': [
            {'name': 'apple', 'color': 'red'},
            {'name': 'grape', 'color': 'green'},
        ],
    }
    mask_dict_values(d, u'fruits[].color')

    Ответ

    d = {
        'fruits': [
            {'name': 'apple', 'color': '*****'},
            {'name': 'grape', 'color': '*****'},
        ],
    }
    """

    if not path_name:
        return

    path_list = path_name.split(u'.')
    _mask_dict_values(dict_, path_list, path_name)


def _mask_dict_values(dict_, path_list, full_path):
    if not isinstance(dict_, dict):
        return

    name = path_list[0]
    if len(path_list) == 1:
        for key in dict_:
            if key.lower() == name.lower():
                dict_[key] = universal_value_masker(full_path, dict_[key])
        return

    if _name_means_sequence(name):
        name = _strip_specifier(name)
        for key in dict_:
            if key.lower() == name.lower():
                for subdict in dict_[key]:
                    _mask_dict_values(subdict, path_list[1:], full_path),
    else:
        for key in dict_:
            if key.lower() == name.lower():
                _mask_dict_values(dict_[key], path_list[1:], full_path)


def _name_means_sequence(name):
    return name.endswith(u'[]')


def _strip_specifier(name):
    if _name_means_sequence(name):
        name = name[:-2]
    return name


def mask_signed_string(string_value):
    """
    Если передана подписанная строка - удаляет подпись, заменяя её на *****.
    Подпись - последняя часть строки после точки, к примеру как в куке
    https://wiki.yandex-team.ru/passport/sessionid/#formatkukisessionid
    """
    if not isinstance(string_value, string_types):
        return MASK_PLACEHOLDER

    fields = string_value.rsplit('.', 1)
    if len(fields) > 1:
        return fields[0] + '.' + MASK_PLACEHOLDER
    else:
        return MASK_PLACEHOLDER


def mask_sessionid(sessionid):
    return mask_signed_string(sessionid)


def mask_sessguard(sessguard):
    return mask_signed_string(sessguard)


def mask_cookies(cookies):
    if not isinstance(cookies, list):
        return MASK_PLACEHOLDER
    return [mask_cookies.expr.sub(mask_cookies.mask, c, 1) for c in cookies]


mask_cookies.expr = re.compile(r'=[^ ,;]*')
mask_cookies.mask = '=' + MASK_PLACEHOLDER


def universal_value_masker(path, value):
    """
    Основываясь на текущем пути и значении параметра,
    вычисляет новое значение параметра с замаскированными секретами для записи в лог
    """
    new_value = MASK_PLACEHOLDER
    if path in ('signed_value', 'container', 'service_guard_container'):
        new_value = mask_signed_string(value)
    elif path in ('sessionid', 'sslsessionid', 'Session_id'):
        new_value = mask_sessionid(value)
    elif path in 'sessguard':
        new_value = mask_sessguard(value)
    elif path in ('oauth_token', 'access_token', 'x_token', 'token'):
        mask_len = len(value) // 2
        new_value = string.mask_postfix(value, mask_len)
    elif path == 'X-Ya-Service-Ticket':
        new_value = ServiceTicket.remove_signature(value)
    elif path == 'X-Ya-User-Ticket':
        new_value = UserTicket.remove_signature(value)
    elif path == 'cookies':
        new_value = mask_cookies(value)
    return new_value


def mask_sensitive_fields(data, fields=None):
    if not data or not isinstance(data, dict):
        return data

    fields = fields or (
        # Параметры приходят в ручки регистраций
        'passwd',
        'passwd2',
        'password',
        'password_hash',
        # Параметр ручки test_pwd_hashes ЧЯ
        'hashes',
        # Параметр в ручках: регистрации с КВ-КО, валидации КО.
        'hint_answer',
        # Параметр в ручках: дорегистраций после авторизации, проверки капчи, проверки КО.
        'answer',
        'new_answer',  # смена КВКО
        # Параметры в анкете восстановления
        'json_data.passwords',
        'json_data.question_answer.answer',
        'json_data.phone_numbers',
        # Параметр в /1/bundle/account/secrets/
        'browser_key',
        # Зашифрованный контейнер с куками
        'container',
        # Контейнер с дополнительной sessguard-кукой для поддомена сервиса
        'service_guard_container',
        # Приходит при добровольной смене пароля
        'current_password',
        # Приходит в ручку /1/session/check/
        'session',
        'sslsession',
        # Уходит параметром в ЧЯ
        'sslsessionid',
        # Параметры отправляются в oauth
        'sessionid',  # так же уходит параметром в ЧЯ
        'sessguard',
        'client_secret',
        'x_token_client_secret',
        'oauth_token',
        'x_token',
        # В таком виде паспорт возвращает токен наружу
        'access_token',
        # OAuth-токен, приходящий в некоторые ручки
        'token',
        # Рефреш-токен
        'refresh_token',
        'refresh',
        # HTTP-заголовок с куками пользователя
        'Ya-Client-Cookie',
        'Cookie',
        # HTTP-заголовок с данными для авторизации
        'Ya-Consumer-Authorization',
        'Authorization',
        # Параметр в ответах ручек, выдающих куки пользователей
        'cookies',
        'csrf_token',
        # Одноразовый пароль на включении 2fa
        'otp',
        'totp',
        # Некий секрет, например, totp_secret на включении 2fa
        'secret',
        # URL, содержащий секрет для инициализации rfc 2fa
        'totp_url',
        # Ключ восстановления (ключ трека в БД)
        'secret_key',
        # Ссылка, позволяющая получить доступ к аккаунту
        'secret_link',
        # Код подтверждения email
        'key',
        # Параметры, передаваемые в API Биллинга
        'session_id',
        # Подписанная строка в ЧЯ
        'signed_value',
        # Пароль для архива с данными аккаунта
        'archive_password',
        'oauth_client_secret',
        'code',  # код для QR-кода
        'X-Ya-Service-Ticket',  # сервисный TVM-тикет
        'X-Ya-User-Ticket',  # пользовательский TVM-тикет
        'Ya-Proxy-TVM-Ticket',
        'authorization_code',
        'support_code',  # пинкодница
        'sign',  # подпись в запросе к TVM
        'text',
        # Телефон для флешкола, с которого звонят
        'caller',
        # В yasms словарь параметров для шаблона
        'text_template_params',
        # Код для диктовки в звонке
        'firstCode',
        'secondCode',
        # Код для челленджа на почту
        'email_check_ownership_code',
    )
    data = deepcopy(data)
    if 'json_data' in data:
        # Декодируем данные для API, принимающего json-документы
        try:
            data['json_data'] = json.loads(data['json_data'])
        except ValueError:
            pass

    for path in fields:
        mask_dict_values(data, path)

    return data


def trim_message(message):
    if len(message) < 1024:
        return message.replace('\n', r'\n')
    else:
        return '%s ... (%d total symbols)' % (message[:512].replace('\n', r'\n'), len(message))


def encode_header(name, value):
    try:
        return '%s: %s' % (string.smart_text(name), string.smart_text(value))
    except UnicodeDecodeError as ex:
        return 'Unable to encode header: %s' % ex
