# -*- coding: utf-8 -*-
from datetime import datetime
from time import time

from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.tskv import (
    JsonLogEntry,
    TskvLogEntry,
    TskvLogger,
)
from passport.backend.utils.time import get_unixtime


class StatboxLogEntry(TskvLogEntry):
    max_entry_length = 4000

    def __init__(self, **params):
        params.setdefault('tskv_format', 'passport-log')
        params.setdefault('py', True)  # это факт, что пришли из питонячьего паспорта
        params.setdefault('unixtime', get_unixtime())
        params.setdefault('timestamp', datetime.now().strftime(settings.STATBOX_DATETIME_FORMAT))
        params.setdefault('timezone', settings.TIME_ZONE_OFFSET)
        super(StatboxLogEntry, self).__init__(**params)


class StatboxLogger(TskvLogger):
    """
    Логгер для statbox. Имеет контекст (данные, которые записываются при каждой записи; не включают стандартные
    поля tskv_format, unixtime, py) и текущие данные (которые будут единоразово записаны при следующем вызове log).
    """

    default_logger_name = 'statbox'
    entry_class = StatboxLogEntry


class CryptastatLogEntry(TskvLogEntry):
    def __init__(self, **params):
        params.setdefault('tskv_format', 'passport-sensitive-log')
        params.setdefault('py', True)  # это факт, что пришли из питонячьего паспорта
        params.setdefault('unixtime', get_unixtime())
        super(CryptastatLogEntry, self).__init__(**params)


class CryptastatLogger(TskvLogger):
    default_logger_name = 'cryptastat'
    entry_class = CryptastatLogEntry


class PharmaLogEntry(TskvLogEntry):
    def __init__(self, **params):
        params.setdefault('tskv_format', 'pharma')
        params.setdefault('ts', get_unixtime())
        super(PharmaLogEntry, self).__init__(**params)


class PharmaLogger(TskvLogger):
    default_logger_name = 'pharma'
    entry_class = PharmaLogEntry


class AntifraudLogEntry(JsonLogEntry):
    def __init__(self, **params):
        params.setdefault('tskv_format', 'antifraud')
        params.setdefault('request', 'auth')
        params.setdefault('t', int(time() * 1000))
        super(AntifraudLogEntry, self).__init__(**params)


class AntifraudLogger(TskvLogger):
    default_logger_name = 'antifraud'
    entry_class = AntifraudLogEntry


class YasmsPrivateLogEntry(TskvLogEntry):
    def __init__(self, **params):
        params.setdefault('sms', '1')
        params.setdefault('sender', 'passport')
        if 'unixtime' not in params:
            params.setdefault('unixtime', get_unixtime())
            params.setdefault('unixtimef', float(get_unixtime()))
        super(YasmsPrivateLogEntry, self).__init__(**params)


class YasmsPrivateLogger(TskvLogger):
    default_logger_name = 'yasms_private'
    entry_class = YasmsPrivateLogEntry

    @property
    def has_data(self):
        if 'global_smsid' not in self.all_values:
            raise ValueError('Missing: global_smsid key')
        return super(YasmsPrivateLogger, self).has_data


class CredentialsLogEntry(TskvLogEntry):
    def __init__(self, **params):
        params.setdefault('tskv_format', 'passport-credentials')
        params.setdefault('unixtime', get_unixtime())
        super(CredentialsLogEntry, self).__init__(**params)


class CredentialsLogger(TskvLogger):
    default_logger_name = 'credentials'
    entry_class = CredentialsLogEntry


class AccountModificationLogger(TskvLogger):
    """
    Логгер для топика account modification с большим количеством партиций,
    события в который фильтруются на уровне бэкенда
    """

    default_logger_name = 'account_modification'
    entry_class = StatboxLogEntry


class AccountModificationInfosecLogger(TskvLogger):
    """
    Логгер для топика account modification с размаскированными данными
    """

    default_logger_name = 'account_modification_infosec'
    entry_class = StatboxLogEntry


def to_statbox(events):
    StatboxLogger().log(**events)
