# -*- coding: utf-8 -*-
from collections import namedtuple

from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)


class SubscriptionService(
    namedtuple(
        'SubscriptionService',
        'id, origin_prefixes, app_ids, slug, external_list_ids',
    )
):
    __slots__ = ()

    def __hash__(self):
        return self.id

    def __str__(self):
        return '{} id={} ext_ids={}'.format(self.slug, self.id, self.external_list_ids)


ORIGIN_PARTS_SEPARATOR = '_'


@lazy_loadable()
class MailSubscriptionServicesManager(object):
    def __init__(self):
        service_settings = settings.SENDER_MAIL_SUBSCRIPTION_SERVICES
        self._services = [SubscriptionService(**x) for x in service_settings]
        self._services_by_id = {s.id: s for s in self._services}
        self._services_by_app_id = self._make_mapping_by_list_attr('app_ids')
        self._services_by_origin_prefix = self._make_mapping_by_list_attr('origin_prefixes')

    def _make_mapping_by_list_attr(self, attr_name):
        mapping = {}
        for service in self._services:
            values = getattr(service, attr_name)
            for value in values:
                mapping.setdefault(value, set()).add(service)
        return mapping

    def get_all_services(self):
        return self._services

    def get_service_by_id(self, service_id):
        return self._services_by_id.get(service_id)

    def get_services_by_origin(self, origin):
        # Возвращаем сервис, origin_prefix которого наиболее точно подходит к переданному origin.
        # Для этого перебираем все возможные непустые префиксы переданного origin, начиная с самого длинного.
        prefix = origin
        while prefix:
            if prefix in self._services_by_origin_prefix:
                return self._services_by_origin_prefix[prefix]
            prefix, _, _ = prefix.rpartition(ORIGIN_PARTS_SEPARATOR)

    def get_services_by_app_id(self, app_id):
        return self._services_by_app_id.get(app_id, set())


def get_mail_subscription_services_manager():
    return LazyLoader.get_instance('MailSubscriptionServicesManager')
