# -*- coding: utf-8 -*-

from calendar import timegm
from email.utils import parsedate
import re

from nose.tools import (
    eq_,
    ok_,
)
from passport.backend.core.eav_type_mapping import EXTENDED_ATTRIBUTES_EMAIL_TYPE
from passport.backend.core.test.time_utils.time_utils import (
    DatetimeNow,
    TimeNow,
)
from passport.backend.core.xml.test_utils import assert_html_documents_equal
from passport.backend.utils.string import smart_text
from passport.backend.utils.time import datetime_to_integer_unixtime
from six import iterkeys


EMAIL_DATE_HEADER_RE = re.compile(r'^Date: ([^\n]+)$', re.M)

EMAIL_MULTIPART_BOUNDARY_RE = re.compile(
    r'^Content-Type: multipart/.*?boundary="(.*?)"$',
    re.M | re.DOTALL,
)


def mail_date_to_timestamp(date_string):
    message_time_tuple = parsedate(date_string)
    return timegm(message_time_tuple)


def get_multipart_boundaries(message):
    return dict([
        (
            'boundary_%d' % i,
            boundary,
        ) for i, boundary in enumerate(re.findall(EMAIL_MULTIPART_BOUNDARY_RE, message))
    ])


def assert_messages_equal(actual_message, expected_message, params=None):
    actual_ts, actual_message = _split_date_out(actual_message)
    _, expected_message = _split_date_out(expected_message)

    eq_(actual_ts, TimeNow())
    eq_(
        actual_message,
        expected_message % params if params else expected_message,
    )


def assert_email_message_sent(mailer_faker, email_address, subject, body):
    user_messages = []
    for message in mailer_faker.messages:
        if ('', email_address) in message.recipients:
            user_messages.append(message)

    ok_(user_messages, 'No messages sent to ' + email_address)

    subject_messages = [m for m in user_messages if m.subject == subject]

    user_subjects = ', '.join([m.subject for m in user_messages])
    ok_(subject_messages, 'No messages found with subject "%s": %s' % (subject, user_subjects))

    eq_(len(subject_messages), 1, 'Duplicate messages found: %r' % subject_messages)
    message = subject_messages[0]

    assert_html_documents_equal(message.body, body)


def create_native_email(login, domain, born_date='2000-01-01 00:00:01'):
    return {
        'default': True,
        'native': True,
        'address': '%s@%s' % (login, domain),
        'born-date': born_date,
    }


def create_validated_external_email(login, domain, rpop=False, silent=False, default=False, born_date='2000-01-01 00:00:01'):
    return {
        'rpop': rpop,
        'silent': silent,
        'default': default,
        'native': False,
        'validated': True,
        'address': '%s@%s' % (login, domain),
        'born-date': born_date,
    }


class EmailDatabaseMatcher(object):
    NAME_ON_MODEL_2_META = dict(
        address=('address', 'string'),
        created_at=('created', 'timestamp'),
        confirmed_at=('confirmed', 'timestamp'),
        bound_at=('bound', 'timestamp'),
        is_rpop=('is_rpop', 'boolean'),
        is_unsafe=('is_unsafe', 'boolean'),
        is_silent=('is_silent', 'boolean'),
    )

    def __init__(self, db_faker, uid, email):
        assert not email.is_native

        self._db_faker = db_faker
        self._uid = uid
        self._email = email

    def _check_attr(self, name, expected):
        return check_email_attr(
            db_faker=self._db_faker,
            uid=self._uid,
            email_id=self._email.id,
            field_name=name,
            value=expected,
        )

    def _check_attr_missing(self, name):
        return check_email_attr_missing(
            db_faker=self._db_faker,
            uid=self._uid,
            email_id=self._email.id,
            field_name=name,
        )

    def _check_boolean_attr(self, name, expected):
        if expected:
            self._check_attr(name, '1')
        else:
            self._check_attr_missing(name)

    def _check_timestamp_attr(self, name, expected):
        if expected:
            if isinstance(expected, DatetimeNow):
                expected = TimeNow()
            else:
                expected = str(datetime_to_integer_unixtime(expected))
            self._check_attr(name, expected)
        else:
            self._check_attr_missing(name)

    def _check_string_attr(self, name, expected):
        self._check_attr(name, expected)

    def match(self, name_on_model):
        expected = getattr(self._email, name_on_model)
        attr_name, attr_type = self.NAME_ON_MODEL_2_META[name_on_model]
        match_func = getattr(self, '_check_%s_attr' % attr_type.lower())
        match_func(attr_name, expected)

    def match_all(self):
        for name_on_model in iterkeys(self.NAME_ON_MODEL_2_META):
            self.match(name_on_model)


def check_email_attr(db_faker, uid, email_id, **kwargs):
    return db_faker.check_db_ext_attr(
        uid=uid,
        entity_type=EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        entity_id=email_id,
        **kwargs
    )


def check_email_attr_missing(db_faker, uid, email_id, **kwargs):
    return db_faker.check_db_ext_attr_missing(
        uid=uid,
        entity_type=EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        entity_id=email_id,
        **kwargs
    )


def _split_date_out(message):
    message = smart_text(message)
    match = EMAIL_DATE_HEADER_RE.search(message, re.M)
    return (
        mail_date_to_timestamp(match.group(1)),
        message.replace(match.group(1), ''),
    )
