# -*- coding: utf-8 -*-
from base64 import b64decode
from collections import namedtuple
from email import message_from_string
from email.header import decode_header
import email.utils
import functools

import mock
from passport.backend.core.test.test_utils import single_entrant_patch
import six


FakeMailMessage = namedtuple(u'FakeMailMessaged', u'body recipients subject sender')


@single_entrant_patch
class FakeMailer(object):
    def __init__(self):
        self._mock = mock.Mock()
        self._patch = mock.patch(
            'passport.backend.core.mailer.connection.Popen',
            mock.Mock(return_value=self._mock),
        )
        self.set_sendmail_return_code()

    def start(self):
        self._patch.start()

    def stop(self):
        self._patch.stop()

    def set_side_effect(self, side_effect):
        self._mock.stdin.write.side_effect = side_effect

    def set_sendmail_return_code(self, code=0):
        self._mock.returncode = code

    def get_message_content(self, message_index=0):
        return self._mock.stdin.write.call_args_list[message_index][0][0]

    @property
    def message_count(self):
        return self._mock.stdin.write.call_count

    @staticmethod
    def _decode_header(header_value):
        value, encoding = decode_header(header_value)[0]
        if isinstance(value, six.binary_type):
            value = value.decode(encoding or 'ascii')
        return value

    @property
    def messages(self):
        retval = []
        for call in self._mock.stdin.write.call_args_list:
            raw_message = call[0][0]
            message = message_from_string(raw_message.decode('utf-8'))
            encoding = message.get_content_charset() or 'utf-8'
            body = b64decode(message.get_payload()).decode(encoding)

            recipients = email.utils.getaddresses(message.get_all('To'))
            for i, (real_name_raw, email_address_raw) in enumerate(recipients):
                real_name = self._decode_header(real_name_raw)
                email_address = self._decode_header(email_address_raw)
                recipients[i] = (real_name, email_address)

            subject = self._decode_header(message['Subject'])

            try:
                # Случай, когда указано символьное имя отправителя
                sender_name_raw, sender_address = message['From'].strip().rsplit(' ', 1)
                sender_name = self._decode_header(sender_name_raw.strip())

                sender_info = (
                    sender_name,
                    sender_address.strip('<>'),
                )
            except ValueError:
                sender_info = (
                    '',
                    message['From'].strip('<>'),
                )

            retval.append(FakeMailMessage(body, recipients, subject, sender_info))

        return retval


@single_entrant_patch
class TemplateFakeMailer(object):
    def __init__(self):
        self._mock = mock.Mock(
            name='render_to_sendmail',
            side_effect=functools.partial(self._render_to_sendmail, self),
        )
        self._patch = mock.patch('passport.backend.core.mailer.utils.render_to_sendmail', self._mock)

        self.messages = list()

    def start(self):
        self._patch.start()
        return self

    def stop(self):
        self._patch.stop()

    @staticmethod
    def _render_to_sendmail(
        self,
        template_name,
        info,
        recipients,
        context,
        is_plain_text=False,
    ):
        self.messages.append(
            dict(
                context=context,
                from_=info.from_,
                recipients=recipients,
                subject=info.subject,
                template_name=template_name,
            ),
        )
        return 0
