# -*- coding: utf-8 -*-
"""
В данном модуле многие куски кода взяты из flask-mail, однако существенно переработаны.
"""
from email.encoders import encode_base64
from email.generator import Generator
from email.header import Header
from email.mime.multipart import MIMEMultipart
from email.mime.nonmultipart import MIMENonMultipart
from email.mime.text import MIMEText
from email.utils import (
    formataddr,
    formatdate,
    parseaddr,
)
from re import compile as re_compile
import time

from passport.backend.utils.string import smart_text
import six
from six import (
    string_types,
    StringIO,
)


_NEW_LINES_SERIES_RE = re_compile(r'[\n\r]+')


def _split_address(addr):
    if isinstance(addr, string_types):
        addr = parseaddr(addr)
    return addr


def _sanitize_address(addr):
    """
    Проверить и закодировать имя пользователя и адрес для отправки.

    @param addr: имя и адрес в виде кортежа ("username", "user@ya.ru"),
    либо спецификация адреса в соответствии с RFC2822
    @return строка, удовлетворяющая RFC 2822
    """
    name, addr = _split_address(addr)
    name = Header(name, 'utf-8').encode()
    try:
        addr.encode('ascii')
    except UnicodeEncodeError:  # IDN
        if '@' in addr:
            localpart, domain = addr.split('@', 1)
            # у Header какие-то сложности, encode возвращает строку на py3
            localpart = Header(localpart, 'utf-8').encode('utf-8').encode('utf-8')
            try:
                domain = domain.encode('idna')
                addr = b'@'.join([localpart, domain])
            except UnicodeError:
                addr = Header(addr, 'utf-8').encode()
        else:
            addr = Header(addr, 'utf-8').encode()
    return formataddr((smart_text(name), smart_text(addr)))


def _sanitize_addresses(addresses):
    return [_sanitize_address(address) for address in addresses]


def _get_mimetext(text, subtype='plain'):
    """
    Создать текстовое MIME-сообщение с заданным подтипом.

    @param text: текст сообщения
    @param subtype: подтип сообщения
    @return созданное MIME-сообщение
    """
    return MIMEText(text, _subtype=subtype, _charset='utf-8')


def _sanitize_subject(subject):
    subject = _NEW_LINES_SERIES_RE.sub(' ', subject)
    return subject.strip()


@six.python_2_unicode_compatible
class MessageBase(object):
    """
    Базовый класс для генерации сообщения с использованием
    стандартного пакета email.

    В классе-потомке требуется реализовать метод _populate_message, который
    должен сконструировать MIME-сообщение и положить его в переменную инстанса
    _mime_msg. Для типовых случаев определен класс HTMLMessage (HTML-сообщение без
    вложений), Mixin-классы MultipartMixin (существующее MIME-сообщение вкладывается
    в multipart-сообщение), AttachmentMixin (к существующему сообщению
    дописываются вложения). Эти классы могут быть использованы как конструктор для
    конкретных случаев, например:
        class HtmlWithRelatedAttachmentMessage(HTMLMessage, MultipartMixin('related'), AttachmentMixin):
            pass
    В описанных классах выполняется последовательность вызовов _populate_message (посредством super),
    поэтому порядок базовых классов важен.
    """
    def __init__(self, subject, recipients, from_, sender=None, body=None,
                 html=None, cc=None, bcc=None, reply_to=None, attachments=None,
                 extra_headers=None):
        """
        В конструкторе принимаем все допустимые параметры сообщения.
        MIME-сообщение генерируется сразу же после получения параметров.

        При указании адресов, допускаются две формы:
        1) кортеж вида ('Ivan Ivanov', 'ivan@yandex.ru')
        2) строка - спецификация адреса в соответствии с RFC2822 (п. 3.4), например, 'ivan@yandex.ru'
        @param subject: тема сообщения
        @param recipients: список адресов получаталей
        @param from_: адрес автора сообщения - заголовок From. См. RFC5322 п. 3.6.2 про отличия от Sender
        @param sender: адрес отправителя сообщения
        @param body: текст сообщения в формате plaintext
        @param html: текст сообщения в формате html
        @param cc: CC в виде списка адресов
        @param bcc: BCC в виде списка адресов
        @param reply_to: адрес, на который отправлять ответ на сообщение.
        По умолчанию ответ отправляется по адресу в заголовке From - см. RFC5322 п. 3.6.2
        @param attachments: список объектов Attachment
        @param extra_headers: словарь дополнительных заголовков для сообщения
        """

        self.subject = subject
        self.recipients = recipients
        self.from_ = from_
        self.sender = sender
        self.body = (body or '').encode('utf-8')
        self.html_body = html.encode('utf-8') if html else html
        self.cc = sorted(cc or [])
        self.bcc = sorted(bcc or [])
        self.reply_to = reply_to
        self.attachments = attachments or []
        self.extra_headers = extra_headers

        self._mime_msg = None
        self._populate_message()
        self._populate_root_headers()

    @property
    def from_address(self):
        return _sanitize_address((None, _split_address(self.from_)[1]))

    def as_string(self, unixfrom=False):
        """Return the entire formatted message as a string.
        Optional `unixfrom' when True, means include the Unix From_ envelope
        header.

        This is a convenience method and may not generate the message exactly
        as you intend because by default it mangles lines that begin with
        "From ".  For more flexibility, use the flatten() method of a
        Generator instance.
        """
        fp = StringIO()
        g = Generator(fp)
        g.flatten(self._mime_msg, unixfrom=unixfrom)
        return fp.getvalue()

    def _populate_message(self):
        """
        Создать MIME-сообщение для отправки.
        Метод должен задать значение переменной _mime_msg.
        """
        if not self._mime_msg:
            raise NotImplementedError(
                'MIME message not populated. Implement _populate_message in subclass!',
            )

    def _get_addresses_header(self, addresses):
        return ', '.join(sorted(set(_sanitize_addresses(addresses))))

    def _populate_root_headers(self):
        """
        Задать заголовки для корневого сообщения, включая предоставленные
        пользователем дополнительные заголовки.
        """
        msg = self._mime_msg
        msg['Subject'] = _sanitize_subject(self.subject)
        msg['From'] = _sanitize_address(self.from_)
        if self.sender:
            msg['Sender'] = _sanitize_address(self.sender)
        msg['To'] = self._get_addresses_header(self.recipients)

        msg['Date'] = formatdate(time.time())

        if self.cc:
            msg['Cc'] = self._get_addresses_header(self.cc)
        if self.bcc:
            msg['Bcc'] = self._get_addresses_header(self.bcc)
        if self.reply_to:
            msg['Reply-To'] = _sanitize_address(self.reply_to)
        if self.extra_headers:
            for k, v in self.extra_headers.items():
                msg[k] = v

    def __str__(self):
        return self.as_string()


class HTMLMessage(MessageBase):
    """
    Создание MIME-сообщения типа text/html
    """
    def _populate_message(self):
        self._mime_msg = _get_mimetext(self.html_body, subtype='html')
        super(HTMLMessage, self)._populate_message()


class PlainTextMessage(MessageBase):
    """
    Создание MIME-сообщения типа text/plain
    """
    def _populate_message(self):
        self._mime_msg = _get_mimetext(self.body, subtype='plain')
        super(PlainTextMessage, self)._populate_message()


def MultipartMixin(subtype):
    """
    Генерация вспомогательного класса, добавляющего корневое сообщение
    типа multipart/<subtype>. Существующее MIME-сообщение добавляется как вложение.

    @param subtype: подтип типа multipart
    """
    class MultipartMixinWrapped(MessageBase):
        def _populate_message(self):
            root_msg = MIMEMultipart(_subtype=subtype)
            root_msg.attach(self._mime_msg)
            self._mime_msg = root_msg
            super(MultipartMixinWrapped, self)._populate_message()
    return MultipartMixinWrapped


class AttachmentMixin(MessageBase):
    """
    Класс добавляет вложения в существующее корневое сообщение.
    MIME-тип берется из вложений.
    """
    def _populate_message(self):
        self._populate_attachments()
        super(AttachmentMixin, self)._populate_message()

    def _populate_attachments(self):
        """
        Добавить к MIME-сообщению все требуемые вложения.
        """
        attachments = self.attachments or []
        for attachment in attachments:
            type_, subtype = attachment.content_type.split('/')
            mime_attach = MIMENonMultipart(type_, subtype)
            mime_attach.set_payload(attachment.data)
            encode_base64(mime_attach)

            try:
                attachment.filename.encode('ascii')
                mime_attach.add_header(
                    'Content-Disposition',
                    '%s;filename="%s"' % (attachment.disposition, attachment.filename),
                )
            except UnicodeEncodeError:
                filename = attachment.filename
                mime_attach.add_header(
                    'Content-Disposition',
                    attachment.disposition,
                    filename=('UTF8', '', filename.encode('utf-8') if six.PY2 else filename),
                )
            for key, value in attachment.headers.items():
                mime_attach.add_header(key, value)

            self._mime_msg.attach(mime_attach)


class HtmlWithMixedAttachmentMessage(HTMLMessage, MultipartMixin('mixed'), AttachmentMixin):
    """
    Сообщение типа multipart/mixed, содежащее HTML-текст и вложения.
    """


class PlainTextWithMixedAttachmentMessage(PlainTextMessage, MultipartMixin('mixed'), AttachmentMixin):
    """
    Сообщение типа multipart/mixed, содежащее plain текст и вложения.
    """


class Attachment(object):
    """
    Класс для хранения информации о вложении.
    """
    def __init__(self, filename, content_type, data=None, disposition=None, headers=None):
        """
        @param filename: имя вложения (файла)
        @param content_type: MIME-тип данных
        @param data: сырые данные
        @param disposition: заголовок Content-Disposition
        @param headers: дополнительные заголовки для вложения
        """
        self.filename = filename
        self.content_type = content_type
        self.data = data
        self.disposition = disposition or 'attachment'
        self.headers = headers or {}


__all__ = (
    'Attachment',
    'MessageBase',
    'HTMLMessage',
    'MultipartMixin',
    'AttachmentMixin',
    'HtmlWithMixedAttachmentMessage',
    'PlainTextMessage',
    'PlainTextWithMixedAttachmentMessage',
)
