# -*- coding: utf-8 -*-

import logging

import jinja2
from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.core.mailer import (
    HTMLMessage,
    PlainTextMessage,
    send_message,
)
from passport.backend.core.types.login.login import masked_login
import six


log = logging.getLogger('passport.mailer.utils')


class MailInfo(object):
    def __init__(
        self,
        tld,
        from_,
        subject,
        reply_to=None,
        subject_format_type=None,
    ):
        self.tld = tld
        self.from_ = from_
        self.subject = subject
        self.reply_to = None
        self.subject_format_type = subject_format_type or 'old_python'


def _translate_from_address(info):
    return settings.MAIL_DEFAULT_FROM_ADDRESS % info.tld


def _default_from(info):
    if isinstance(info.from_, six.string_types):
        return info.from_, _translate_from_address(info)
    else:
        return info.from_


def _default_reply_to(info):
    if info.reply_to is None:
        return (
            settings.MAIL_DEFAULT_REPLY_TO[0],
            settings.MAIL_DEFAULT_REPLY_TO[1] % info.tld,
        )
    else:
        return info.reply_to


def render_to_sendmail(template_name, info, recipients, context, is_plain_text=False):
    """
    Принимаем информацию о письме, контекст для шаблона и отправляем письмо пользователю.

    @param template_name: имя шаблона для тела сообщения
    @param info: Информация о письме, имя отправителя, tld, тема
    @param recipients: список получателей
    @param context: контекст для подстановки в шаблон
    @param is_plain_text: признак того, что нужно отправить сообщение в формате text/plain
    @return: код завершения команды sendmail
    """
    subject = render_message_subject(info.subject, context, info.subject_format_type)
    info = MailInfo(
        subject=subject,
        from_=info.from_,
        tld=info.tld,
    )

    rendered_text = render_message_text(template_name, context)
    message = create_user_message(
        info,
        recipients,
        rendered_text,
        is_plain_text=is_plain_text,
    )
    return send_message(message)


def render_message_text(template_name, context):
    """
    Строим текст сообщения из jinja2-шаблона и контекста.

    @param template_name: имя шаблона для тела сообщения
    @param context: набор переменных для подстановки в шаблон
    @return: текст после рендеринга шаблона
    """
    template_loader = LazyLoader.get_instance('TemplateLoader')
    # current_app должно знать где искать шаблоны
    # TODO: выполнять загрузку всех шаблонов в execute_app
    template = template_loader.get_template(template_name)

    return template.render(
        MAIL_NOISE_IMAGE_URL=settings.MAIL_NOISE_IMAGE_URL,
        MAIL_SHADOW_IMAGE_URL=settings.MAIL_SHADOW_IMAGE_URL,
        **context
    )


def render_message_subject(subject, context, format_type):
    if format_type == 'T':
        template = jinja2.Template(
            subject,
            autoescape=False,
            variable_start_string='%',
            variable_end_string='%',
        )
        return template.render(**context)
    else:
        return subject % context


def create_user_message(info, recipients, text, is_plain_text=False):
    """
    Собираем email-сообщение для отправки пользователю.
    Стандартный адрес для ответа (noreply@...).

    @param info: Информация о письме, имя отправителя, tld, тема
    @param recipients: список адресатов
    @param text: текст сообщения
    @param is_plain_text: признак того, что нужно отправить сообщение в формате text/plain
    @return: объект сообщения
    """
    from_ = _default_from(info)
    reply_to = _default_reply_to(info)

    message_params = dict(
        subject=info.subject,
        recipients=recipients,
        from_=from_,
        reply_to=reply_to,
    )
    if is_plain_text:
        return PlainTextMessage(body=text, **message_params)
    else:
        return HTMLMessage(html=text, **message_params)


def make_email_context(language, account, context=None):
    template_context = {
        'language': language,
        'logo_url_key': 'logo_url',
        'signature_key': 'signature.secure',
        'feedback_key': 'feedback',
        'feedback_url_key': 'feedback_url',
    }
    if account is not None:
        first_name = (
            account.person.firstname or
            (account.person.display_name and account.person.display_name.name) or
            account.login
        )

        if (
            account.is_social or
            account.is_neophonish
        ):
            # У Социальщика есть только синтетический логин, а показывать этот
            # логин пользователю дурной тон, поэтому вместе логина будем
            # показывать first_name.
            login = first_name
        else:
            login = account.display_login or account.login

        template_context.update({
            'login': login,
            'MASKED_LOGIN': login,
            'FIRST_NAME': first_name,
        })
    template_context.update(context or {})
    return template_context


def login_shadower(context):
    return dict(
        context,
        login=masked_login(context['login']),
        MASKED_LOGIN=masked_login(context['MASKED_LOGIN']),
    )


def send_mail_for_account(template_name, mail_info, context, account,
                          context_shadower=None, send_to_external=True,
                          send_to_native=True, specific_email=None, is_plain_text=False):
    """
    Отправляет письма на выбранные почтовые ящики пользователя
    @param template_name: html-шаблон письма;
    @param mail_info: (tld, sender, subject);
    @param context: словарь значений для подстановки в шаблон;
    @param account: аккаунт, которому отправляем почту;
    @param context_shadower: функция-преобразователь контекста для писем, отправляемых не на дефолтный емейл
    @param send_to_external: отправлять ли письма на внешние подтверждённые ящики
    @param send_to_native: отправлять ли письма на нативные ящики
    @param specific_email: отправить письмо на данный ящик аккаунта
    @param is_plain_text: признак того, что нужно отправить сообщение в формате text/plain
    """
    native_default_address = ''

    emails = []

    if specific_email:
        emails.append(specific_email)

    if send_to_external:
        external_emails = [email.address for email in account.emails.confirmed_external if not email.is_rpop and not email.is_silent]
        max_count = settings.MAX_NOTIFICATION_EMAIL_COUNT
        if len(external_emails) > max_count:
            log.debug('Too many external emails for uid %d, using %d/%d', account.uid, max_count, len(emails))
            external_emails = external_emails[:max_count]
        emails += external_emails

    if send_to_native:
        if account.emails.default and account.emails.default.is_native:
            emails += [account.emails.default.address]
            native_default_address = account.emails.default.address

    if emails:
        log.debug('Trying to send mail for uid %d, addresses count: %d', account.uid, len(emails))
    else:
        log.warning('Trying to send mail for uid %d, but no addresses found', account.uid)

    emails.sort()  # значительно облегчает тестирование
    for email in emails:
        log.debug(
            'Sending mail for uid %d to %s. Default %s',
            account.uid,
            email,
            native_default_address,
        )

        shadow_context = email != native_default_address and context_shadower is not None
        render_to_sendmail(
            template_name,
            mail_info,
            [email],
            context_shadower(context) if shadow_context else context,
            is_plain_text=is_plain_text,
        )


def get_tld_by_country(country):
    """
    Определяет TLD для полей `From` и `Reply to` при отправке письма, на вход принимает страну пользователя.
    """
    if country is None:
        country = ''
    return settings.COUNTRY_CODE_TO_MAIL_ADDRESSES_TLD.get(country.lower(), settings.MAIL_ADDRESSES_DEFAULT_TLD)
