# -*- coding: utf-8 -*-

import logging

from passport.backend.core.conf import settings
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.language_detect import language_detect
from passport.backend.core.models.alias import (
    AltDomainAlias,
    BankPhoneNumberAlias,
    FederalAlias,
    KiddishAlias,
    KinopoiskAlias,
    KolonkishAlias,
    LiteAlias,
    MailAlias,
    MailishAlias,
    NeophonishAlias,
    PddAlias,
    PhonenumberAlias,
    PhonishAlias,
    PortalAlias,
    PublicIdAlias,
    ScholarAlias,
    SocialAlias,
    UberAlias,
    YambotAlias,
    YandexoidAlias,
)
from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    CollectionField,
    DateTimeField,
    Field,
    IntegerField,
    ModelField,
    TypedField,
    UnixtimeField,
)
from passport.backend.core.models.browser_key import BrowserKey
from passport.backend.core.models.domain import PartialPddDomain
from passport.backend.core.models.email import Emails
from passport.backend.core.models.family import AccountFamilyInfo
from passport.backend.core.models.hint import Hint
from passport.backend.core.models.karma import Karma
from passport.backend.core.models.passman_recovery_key import PassManRecoveryKey
from passport.backend.core.models.password import (
    Password,
    ScholarPassword,
)
from passport.backend.core.models.person import Person
from passport.backend.core.models.phones.phones import Phones
from passport.backend.core.models.plus import Plus
from passport.backend.core.models.rfc_totp_secret import RfcTotpSecret
from passport.backend.core.models.subscription import (
    build_sid_property,
    Subscription,
)
from passport.backend.core.models.takeout import Takeout
from passport.backend.core.models.totp_secret import TotpSecret
from passport.backend.core.models.webauthn import WebauthnCredentials
from passport.backend.core.services import Service
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_FEDERAL,
    ACCOUNT_TYPE_KIDDISH,
    ACCOUNT_TYPE_KINOPOISK,
    ACCOUNT_TYPE_KOLONKISH,
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_MAILISH,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_PHONISH,
    ACCOUNT_TYPE_SCHOLAR,
    ACCOUNT_TYPE_SOCIAL,
    ACCOUNT_TYPE_UBER,
    ACCOUNT_TYPE_YAMBOT,
)
from passport.backend.core.types.expirable_counter.expirable_counter import ExpirableCounter
from passport.backend.core.types.login.login import normalize_login
from passport.backend.core.types.mail_subscriptions import UnsubscriptionList
from passport.backend.core.undefined import Undefined
from passport.backend.utils.time import unixtime_to_datetime


log = logging.getLogger('passport.models.account')


ACCOUNT_ENABLED = 0
ACCOUNT_DISABLED = 1
# У аккаунта есть подписки на блокирующие сиды, поэтому его нельзя удалить
ACCOUNT_DISABLED_ON_DELETION = 2

MAIL_STATUS_NONE = None  # нет почтового ящика
MAIL_STATUS_ACTIVE = 1  # есть активный почтовый ящик
MAIL_STATUS_FROZEN = 2  # почтовый ящик заморожен

sid_property = build_sid_property(lambda self: self)


def get_preferred_language(account, selected_language=None):
    """
    Возвращает предпочитаемый пользователем язык, из тех языков,
    которые поддерживает интерфейс Паспорта.
    """
    if account and account.person and account.person.language:
        selected_language = selected_language or account.person.language

    return language_detect.get_preferred_language(selected_language)


def is_logouted_after(logout_datetime, check_datetime):
    # Утвержденная по ЧЯ логика проверки глобального и прочих логаутов
    return bool(logout_datetime) and logout_datetime > check_datetime


def parse_uid(data, account, *args):
    if 'uid' in data and data['uid'] is None and not account.uid:
        raise UnknownUid()

    if data.get('uid') is None:
        return False, None
    else:
        return True, data['uid']


def parse_failed_auth_challenge_checks_counter(data, account, *args):
    raw_value = data.get('failed_auth_challenge_checks_counter')
    if not raw_value:
        return True, account.failed_auth_challenge_checks_counter or ExpirableCounter(0, 0)

    return True, ExpirableCounter.parse(raw_value)


def parse_external_organization_ids(data, account, *args):
    raw_value = data.get('account.external_organization_ids')
    if not raw_value:
        return False, None

    return True, set(map(int, raw_value.split(',')))


class AccountDeletionOperation(Model):
    parent = None
    started_at = UnixtimeField('started_at')

    def parse(self, data):
        super(AccountDeletionOperation, self).parse(data)
        op_data = data.get('account_deletion_operation')
        if op_data is not None:
            self.started_at = unixtime_to_datetime(op_data['started_at'])
            del_op = self
        else:
            del_op = Undefined
        return del_op


class Account(Model):
    """
    Учетная запись пользователя
    https://doc.yandex-team.ru/Passport/mail-for-domain/api/accountdef.html
    """
    uid = Field(parse_uid)
    karma = ModelField(Karma)

    # Алиасы, определяющие тип аккаунта
    kiddish_alias = ModelField(KiddishAlias)
    kinopoisk_alias = ModelField(KinopoiskAlias)
    kolonkish_alias = ModelField(KolonkishAlias)
    lite_alias = ModelField(LiteAlias)
    mailish_alias = ModelField(MailishAlias)
    neophonish_alias = ModelField(NeophonishAlias)
    pdd_alias = ModelField(PddAlias)
    phonish_alias = ModelField(PhonishAlias)
    portal_alias = ModelField(PortalAlias)
    federal_alias = ModelField(FederalAlias)
    scholar_alias = ModelField(ScholarAlias)
    social_alias = ModelField(SocialAlias)
    uber_alias = ModelField(UberAlias)
    yambot_alias = ModelField(YambotAlias)

    # Дополнительные алиасы
    public_id_alias = ModelField(PublicIdAlias)
    phonenumber_alias = ModelField(PhonenumberAlias)
    altdomain_alias = ModelField(AltDomainAlias)
    yandexoid_alias = ModelField(YandexoidAlias)
    mail_alias = ModelField(MailAlias)
    bank_phonenumber_alias = ModelField(BankPhoneNumberAlias)

    # Логин в том виде, как его ввёл пользователь. Заполняется при регистрации, из ЧЯ не парсится.
    # Для портальных аккаунтов хранит весь логин, для ПДД - только логинную часть.
    user_defined_login = Field()

    # Список подписок пользователя
    subscriptions = CollectionField('subscriptions', Subscription)
    # Список всех email-адресов пользователя
    emails = ModelField(Emails)

    # Используется для Колонкишей. Хранится uid "нормального" пользователя, для которого создали Колонкиша.
    creator_uid = IntegerField('account.creator_uid')

    # Подтверждённый номер телефона пользователя
    phones = ModelField(Phones)

    # Webauthn-аутентификаторы, который пользователь привязал к аккаунту
    webauthn_credentials = ModelField(WebauthnCredentials)

    hint = ModelField(Hint)  # Хинт у аккаунта всегда один
    password = ModelField(Password)
    rfc_totp_secret = ModelField(RfcTotpSecret)
    totp_secret = ModelField(TotpSecret)
    totp_junk_secret = Field()  # отдельным полем, так как не должно удаляться при выключении 2фа
    scholar_password = ModelField(ScholarPassword)
    person = ModelField(Person)

    # Подписка на Плюс
    plus = ModelField(Plus)

    # Частичная информация о связанном с пользователем домене ПДД
    domain = ModelField(PartialPddDomain)

    # Информация о выгрузках пользовательских данных
    takeout = ModelField(Takeout)

    display_login = Field('display_login')

    # Статус заблокированности аккаунта
    disabled_status = Field('is_disabled')

    # Определяется по sid 668
    is_betatester = sid_property(668)

    # Определяется по sid 670
    is_corporate = sid_property(670)

    # Требуется сильный пароль.
    is_strong_password_required = sid_property(67)

    # Пользователь включил себе Пароли приложений, после этого пароль используется только в веб авторизациях.
    enable_app_password = BooleanField('enable_app_password')

    # юзер согласился с пользовательским ПДДшным соглашением
    # есть подписка на 102 сид
    is_pdd_agreement_accepted = sid_property(102)

    # юзер согласился с пользовательским соглашением Денег
    is_money_agreement_accepted = Field('account.is_money_agreement_accepted')

    # Требуется restricted session.
    # Какую бы пользователь сессию не запросил,
    # все равно выставляем ему restricted
    is_short_session_required = Field()

    # Требуется ли показать вопрос о длине сессии.
    is_long_session_question_required = Field()

    # Точное время регистрации.
    # TODO: придётся заполнять и тут и в person, до тех пор пока не
    # перенесём данное значение только сюда
    registration_datetime = DateTimeField('userinfo.reg_date.uid')

    # Время глобального разлогина (поле "Разлогиниться на всех компьютерах").
    # Если выставлено это поле, то все куки/ПП/сессии/токены, выставленные
    # ранее, считаются невалидными.
    global_logout_datetime = UnixtimeField('global_logout_datetime')

    # Время отзыва токенов пользователя
    tokens_revoked_at = UnixtimeField('revoker.tokens')

    # Время отзыва вебовских сессий
    web_sessions_revoked_at = UnixtimeField('revoker.web_sessions')

    # Время отзыва паролей приложений
    app_passwords_revoked_at = UnixtimeField('revoker.app_passwords')

    # Ключ браузера
    browser_key = ModelField(BrowserKey)

    # Запасной ключ для менеджера паролей
    passman_recovery_key = ModelField(PassManRecoveryKey)

    # Используется только в yandex-team.
    # Показывает, является ли аккаунт аккаунтом сотрудника.
    is_employee = Field('is_employee')

    # Используется только в yandex-team.
    # Показывает, является ли аккаунт аккаунтом почтовой рассылки.
    is_maillist = Field('is_maillist')

    # Является администратором ПДД-домена
    is_pdd_admin = sid_property(104)

    # Является ПДД-пользователем домена Директории
    is_pdd_workspace_user = BooleanField('have_organization_name')

    # Является администратором Коннекта
    is_connect_admin = BooleanField('account.is_connect_admin')

    # Показывает email-адрес по умолчанию.
    default_email = Field('default_email')

    # Используется ли аккаунт несколькими людьми
    is_shared = BooleanField('is_shared')

    # Часто ли аккаунт вламывают
    is_easily_hacked = BooleanField('account.is_easily_hacked')

    # Является ли пользователем сервиса Аудитория
    audience_on = BooleanField('audience_on')

    # Пользователю надо показывать промо 2ФА, тк его часто взламывают
    show_2fa_promo = BooleanField('show_2fa_promo')

    mail_db_id = Field('hosts.db_id.2')

    mail_status = IntegerField('subscription.mail.status')

    # Дата последней отправки письма о показе челленжа
    auth_email_datetime = UnixtimeField('auth_email_datetime')
    # Число неудачных ответов на челленж
    failed_auth_challenge_checks_counter = Field(parse_failed_auth_challenge_checks_counter)

    originated_from_profile_id = Field()  # заполняется на основе ответа social_api

    deletion_operation = ModelField(AccountDeletionOperation)

    # Какие данные дозапрашивали последний раз
    additional_data_asked = Field('account.additional_data_asked')

    # Когда спросить данные в следующий раз
    additional_data_ask_next_datetime = UnixtimeField('account.additional_data_ask_next_datetime')

    external_organization_ids = Field(parse_external_organization_ids)

    phonish_namespace = Field('account.phonish_namespace')

    # PASSP-25526 Вход по мессенджеру/волшебному письму разрешен/запрещен
    magic_link_login_forbidden = BooleanField('account.magic_link_login_forbidden')
    # Вход по Qr коду разрешен/запрещен
    qr_code_login_forbidden = BooleanField('account.qr_code_login_forbidden')
    # Вход по sms разрешен/запрещен
    sms_code_login_forbidden = BooleanField('account.sms_code_login_forbidden')

    # PASSP-25400 Билинговые фичи
    billing_features = Field('billing_features')

    # Информация о семье
    family_info = ModelField(AccountFamilyInfo)

    # Всегда требовать челлендж
    force_challenge = BooleanField('account.force_challenge')

    sms_2fa_on = BooleanField('account.sms_2fa_on')
    forbid_disabling_sms_2fa = BooleanField('account.forbid_disabling_sms_2fa')

    user_defined_public_id = Field('account.user_defined_public_id')
    public_id = Field('public_id')

    is_verified = BooleanField('account.is_verified')

    # не генерить нативные емейлы с яндексовым доменом
    hide_yandex_domains_emails = BooleanField('account.hide_yandex_domains_emails')

    is_child = BooleanField('account.is_child')

    content_rating_class = IntegerField('account.content_rating_class')
    music_content_rating_class = IntegerField('account.music_content_rating_class')
    video_content_rating_class = IntegerField('account.video_content_rating_class')

    # Отписки от почтовых рассылок
    unsubscribed_from_maillists = TypedField(
        'account.unsubscribed_from_maillists',
        UnsubscriptionList,
    )

    personal_data_public_access_allowed = BooleanField('account.personal_data_public_access_allowed')
    personal_data_third_party_processing_allowed = BooleanField('account.personal_data_third_party_processing_allowed')

    family_pay_enabled = Field('account.family_pay.enabled')

    is_documents_agreement_accepted = Field('account.is_documents_agreement_accepted')

    is_dzen_sso_prohibited = Field('account.dzen_sso_prohibited')

    last_child_family = Field('account.last_child_family')

    can_manage_children = BooleanField('account.can_manage_children')

    def is_subscribed(self, service):
        """
        Узнает, подписан ли аккаунт на данный сервис
        каким-либо образом (включая алиасы)
        :type service: Service
        :rtype: bool
        """
        if not isinstance(service, Service):
            raise ValueError('Invalid type of argument')

        # Для intranet такого алиаса домена нет
        galatasaray_domain_id = settings.ALT_DOMAINS.get('galatasaray.net', -1)

        if (
                (service.sid == 61 and self.altdomain_alias.domain_id == galatasaray_domain_id) or
                (service.sid == 669 and self.is_yandexoid)
        ):
            return True

        return bool(self.subscriptions.get(service.sid))

    def is_logouted_after(self, check_datetime):
        return is_logouted_after(self.web_sessions_logout_datetime, check_datetime)

    @property
    def web_sessions_logout_datetime(self):
        """Возвращает время отзыва веб-сессий (непосредственное или в результате глогаута)"""
        logout_datetimes = [
            dt
            for dt in (
                self.global_logout_datetime,
                self.web_sessions_revoked_at,
            )
            if dt
        ]
        return max(logout_datetimes) if logout_datetimes else None

    @property
    def type(self):
        if self.portal_alias:
            return ACCOUNT_TYPE_NORMAL
        elif self.federal_alias:  # должен стоять выше pdd
            return ACCOUNT_TYPE_FEDERAL
        elif self.pdd_alias:
            return ACCOUNT_TYPE_PDD
        elif self.lite_alias:
            return ACCOUNT_TYPE_LITE
        elif self.mailish_alias:
            return ACCOUNT_TYPE_MAILISH
        elif self.social_alias:
            return ACCOUNT_TYPE_SOCIAL
        elif self.phonish_alias:
            return ACCOUNT_TYPE_PHONISH
        elif self.kinopoisk_alias:
            return ACCOUNT_TYPE_KINOPOISK
        elif self.uber_alias:
            return ACCOUNT_TYPE_UBER
        elif self.yambot_alias:
            return ACCOUNT_TYPE_YAMBOT
        elif self.kolonkish_alias:
            return ACCOUNT_TYPE_KOLONKISH
        elif self.neophonish_alias:
            return ACCOUNT_TYPE_NEOPHONISH
        elif self.scholar_alias:
            return ACCOUNT_TYPE_SCHOLAR
        elif self.kiddish_alias:
            return ACCOUNT_TYPE_KIDDISH

        # ситуация недостижимая в реальности, поэтому не стесняемся кидать ValueError
        raise ValueError('Account without required aliases')

    @property
    def login(self):
        # Сохраняем старую багофичу: после регистрации портального аккаунта в поле login
        # раньше клался ненормализованный логин (user_defined_login). После вычитывания
        # аккаунта из ЧЯ login уже должен быть нормализованным.
        return (
            (self.portal_alias and self.user_defined_login) or
            (self.portal_alias and self.portal_alias.alias) or
            (self.pdd_alias and self.user_defined_login and self.domain.unicode_domain and self.user_defined_login + '@' + self.domain.unicode_domain) or
            (self.pdd_alias and self.pdd_alias.alias) or
            (self.lite_alias and self.lite_alias.alias) or
            (self.mailish_alias and self.mailish_alias.alias) or
            (self.social_alias and self.social_alias.alias) or
            (self.phonish_alias and self.phonish_alias.alias) or
            (self.uber_alias and self.uber_alias.alias) or
            (self.yambot_alias and self.yambot_alias.alias) or
            (self.kolonkish_alias and self.kolonkish_alias.alias) or
            (self.neophonish_alias and self.neophonish_alias.alias) or
            (self.scholar_alias and self.scholar_alias.alias) or
            (self.kiddish_alias and self.kiddish_alias.alias) or
            (self.federal_alias and self.federal_alias.alias) or
            Undefined
        )

    def set_portal_alias(self, login):
        """
        Выставляет аккаунту портальный алиас, обновляя user_defined_login.
        """
        self.portal_alias = PortalAlias(self, login=login)
        self.user_defined_login = login

    @property
    def is_user_enabled(self):
        """
        Включен ли аккаунт. Для ПДД учитывает включенность домена
        :return: bool
        """
        return self.is_enabled and (not self.is_pdd or self.domain.is_enabled)

    @property
    def is_lite(self):
        """
        Является ли аккаунт лайтовым.
        :return: bool
        """
        return self.type == ACCOUNT_TYPE_LITE

    @property
    def is_superlite(self):
        """
        Является ли аккаунт супер лайтовым (лайт и нет пароля)
        :return: bool
        """
        return self.is_lite and not self.have_password

    @property
    def is_normal(self):
        return self.type == ACCOUNT_TYPE_NORMAL

    @property
    def is_enabled(self):
        return self.disabled_status == ACCOUNT_ENABLED

    @is_enabled.setter
    def is_enabled(self, flag):
        self.disabled_status = ACCOUNT_ENABLED if flag else ACCOUNT_DISABLED

    @property
    def is_pdd(self):
        """
        Принадлежит ли аккаунт пользователю почты для доменов
        """
        return self.type == ACCOUNT_TYPE_PDD

    @property
    def is_social(self):
        return self.type == ACCOUNT_TYPE_SOCIAL

    @property
    def is_phonish(self):
        return self.type == ACCOUNT_TYPE_PHONISH

    @property
    def is_mailish(self):
        return self.type == ACCOUNT_TYPE_MAILISH

    @property
    def is_kiddish(self):
        return self.type == ACCOUNT_TYPE_KIDDISH

    @property
    def is_kinopoisk(self):
        return self.type == ACCOUNT_TYPE_KINOPOISK

    @property
    def is_scholar(self):
        return self.type == ACCOUNT_TYPE_SCHOLAR

    @property
    def is_uber(self):
        return self.type == ACCOUNT_TYPE_UBER

    @property
    def is_yambot(self):
        return self.type == ACCOUNT_TYPE_YAMBOT

    @property
    def is_kolonkish(self):
        return self.type == ACCOUNT_TYPE_KOLONKISH

    @property
    def is_yandexoid(self):
        return bool(self.yandexoid_alias)

    @property
    def is_neophonish(self):
        return self.type == ACCOUNT_TYPE_NEOPHONISH

    @property
    def is_federal(self):
        return self.type == ACCOUNT_TYPE_FEDERAL

    def has_sid(self, sid):
        """
        Отвечает на более узкий вопрос, есть ли сид в подписках аккаунта.
        Алиасы уже не считаются.
        """
        return bool(self.subscriptions.get(sid))

    @property
    def is_complete_pdd(self):
        r"""
        Дорегистрированный пользователь ПДД, который:
        * принял ПС ПДД
        И в случае, если это не пользователь Директории:
        * имеет средство восстановления (КВ\КО или защищенный телефон)
        * указаны фамилия и имя
        """
        if not self.is_pdd:
            raise TypeError('User is not a pdd user')

        if not self.is_pdd_agreement_accepted:
            return False

        if self.is_pdd_workspace_user:
            return True

        if not (self.person and self.person.firstname and self.person.lastname):
            return False

        if not (
            (self.hint and self.hint.normalized_answer and self.hint.question) or
            # FIXME: После окончания переливки телефонов в новую схему перейти
            # на исключительное использование self.phones
            self.phones.secure
        ):
            return False

        return True

    @property
    def is_complete_federal(self):
        """
        Дорегистрированный пользователь федерации: тот, который принял ПС
        """
        if not self.is_federal:
            raise TypeError('User is not a federal user')

        return self.is_pdd_agreement_accepted

    @property
    def is_incomplete_autoregistered(self):
        """
        Проверяет, что пользователь зарегистрирован автоматически и не сменил/создал пароль.
        """
        return not self.is_pdd and self.password.is_creating_required

    @property
    def normalized_login(self):
        login = normalize_login(self.login)
        if (self.is_pdd or self.is_federal) and '@' in login:
            login, _, _ = login.partition('@')
        return login

    @property
    def human_readable_login(self):
        if self.is_pdd or self.is_federal:
            return self.normalized_login + '@' + self.domain.unicode_domain
        else:
            return self.subscriptions[8].login

    @property
    def machine_readable_login(self):
        if self.is_pdd or self.is_federal:
            return self.normalized_login + '@' + self.domain.punycode_domain
        else:
            return self.normalized_login

    @property
    def have_password(self):
        return bool(self.password.is_set or self.totp_secret.is_set)

    @property
    def is_password_set_or_promised(self):
        # Пароль на аккаунте есть либо будет установлен при сериализации
        return bool(
            self.password is not None and self.password.is_set_or_promised or
            self.totp_secret is not None and self.totp_secret.is_set,
        )

    @property
    def has_family(self):
        return bool(self.family_info)
