# -*- coding: utf-8 -*-
from functools import partial

from passport.backend.core.conf import settings
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE as ANT
from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    Field,
)
from passport.backend.core.types.login.login import (
    normalize_login,
    raw_login_from_email,
)
from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.core.undefined import Undefined
from six import string_types


def _parse_additional_logins(alias_type, data, *args):
    logins = data.get('aliases', {})
    aliased_logins = set(logins.get(str(ANT[alias_type]), []))
    return bool(aliased_logins), aliased_logins


def _parse_additional_pdd_logins(data, *args):
    _, aliased_logins = _parse_additional_logins('pddalias', data, *args)
    aliased_logins = {
        raw_login_from_email(value)
        for value in aliased_logins
    }
    return bool(aliased_logins), aliased_logins


def _parse_alias(alias_type, data, *args):
    aliases = data.get('aliases', {})
    alias = aliases.get(str(ANT[alias_type]))
    return alias is not None, alias


def _parse_phone_number(data, *args):
    _, number = _parse_alias('phonenumber', data, *args)
    if number is None:
        return False, None

    if isinstance(number, PhoneNumber):
        return True, number
    else:
        number = '+%s' % number
        return True, PhoneNumber.parse(number, allow_impossible=True)


class AliasBase(Model):
    parent = None

    @property
    def alias(self):
        """Алиас в человекочитаемом виде"""
        raise NotImplementedError()  # pragma: no cover

    @property
    def serialized_alias(self):
        """Алиас для записи в БД и HistoryDB"""
        # Может перегружаться в потомках
        if self.alias is Undefined:
            return Undefined
        return self.alias.lower()

    def is_empty(self):
        return not self.alias

    def __ne__(self, other):
        return not self.__eq__(other)

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return self.alias == other.alias

    def __nonzero__(self):
        return not self.is_empty()

    def __bool__(self):
        return not self.is_empty()


class MutableAlias(AliasBase):
    pass


class ImmutableAlias(AliasBase):
    pass


class UndeletableAlias(ImmutableAlias):
    pass


class PhonenumberAlias(ImmutableAlias):
    number = Field(_parse_phone_number)
    enable_search = BooleanField('account.enable_search_by_phone_alias')

    @property
    def alias(self):
        if self.number is Undefined:
            return Undefined
        return self.number.digital

    def __eq__(self, other):
        if isinstance(other, PhoneNumber):
            return self.number == other
        elif isinstance(other, string_types):
            if other[0] != '+':
                other = '+' + other
            try:
                other_number = PhoneNumber.parse(other, allow_impossible=True)
                return self.number == other_number
            except InvalidPhoneNumber:
                return False
        elif isinstance(other, PhonenumberAlias):
            return self.number == other.number
        return False


class AltDomainAlias(ImmutableAlias):
    # Алиас из ЧЯ возвращается в формате login@altdomain
    login = Field(partial(_parse_alias, 'altdomain'))

    @property
    def domain_id(self):
        if self.login is Undefined:
            return Undefined
        _, _, domain = self.login.partition('@')
        return settings.ALT_DOMAINS.get(domain, Undefined)

    @property
    def login_part(self):
        if self.login is Undefined:
            return Undefined
        login = raw_login_from_email(self.login)
        return normalize_login(login)

    @property
    def alias(self):
        login = self.login_part
        domain_id = self.domain_id
        if login is Undefined or domain_id is Undefined:
            return Undefined
        return '%s/%s' % (domain_id, login)


class YandexoidAlias(ImmutableAlias):
    login = Field(partial(_parse_alias, 'yandexoid'))

    @property
    def alias(self):
        if self.login is Undefined:
            return Undefined
        return self.login.lower()


class PortalAlias(UndeletableAlias):
    login = Field(partial(_parse_alias, 'portal'))

    @property
    def alias(self):
        if self.login is Undefined:
            return Undefined
        return normalize_login(self.login)


class LiteAlias(UndeletableAlias):
    email = Field(partial(_parse_alias, 'lite'))

    @property
    def alias(self):
        if self.email is Undefined:
            return Undefined
        return self.email


class SocialAlias(UndeletableAlias):
    login = Field(partial(_parse_alias, 'social'))

    @property
    def alias(self):
        if self.login is Undefined:
            return Undefined
        return normalize_login(self.login)


class PhonishAlias(UndeletableAlias):
    login = Field(partial(_parse_alias, 'phonish'))

    @property
    def alias(self):
        if self.login is Undefined:
            return Undefined
        return normalize_login(self.login)


class MailishAlias(UndeletableAlias):
    mailish_id = Field(partial(_parse_alias, 'mailish'))

    @property
    def alias(self):
        if self.mailish_id is Undefined:
            return Undefined
        return self.mailish_id.lower()


class MailAlias(UndeletableAlias):
    email = Field(partial(_parse_alias, 'mail'))

    @property
    def alias(self):
        if self.email is Undefined:
            return Undefined
        return self.email


class PddAlias(UndeletableAlias):
    email = Field(partial(_parse_alias, 'pdd'))
    additional_logins = Field(_parse_additional_pdd_logins)

    def __init__(self, *args, **kwargs):
        self.additional_logins = set()
        super(PddAlias, self).__init__(*args, **kwargs)

    @property
    def alias(self):
        if self.email is Undefined:
            return Undefined
        return self.email

    @property
    def serialized_alias(self):
        if self.email is Undefined:
            return Undefined

        login, _, domain = self.email.partition('@')

        if not (self.parent.domain and self.parent.domain.id):
            raise ValueError('PDD user has no attached domain or no domain ID')
        domain_from_account = self.parent.domain.domain.lower()
        if domain.lower() != domain_from_account:
            raise ValueError(
                'PDD alias does not match domain: %s != %s' % (
                    domain.lower(),
                    domain_from_account,
                ),
            )
        return '%d/%s' % (self.parent.domain.id, login.lower())

    def add_login(self, login):
        self.additional_logins.add(login.lower())

    def has_login(self, login_to_check):
        login_to_check = login_to_check.lower()
        return login_to_check in self.additional_logins

    def remove_login(self, login):
        login = login.lower()
        if login not in self.additional_logins:
            raise ValueError('Attempt to remove non-existent alias "%s"' % login)
        self.additional_logins.remove(login)


class KinopoiskAlias(UndeletableAlias):
    alias = Field(partial(_parse_alias, 'kinopoisk'))


class UberAlias(UndeletableAlias):
    uber_id = Field(partial(_parse_alias, 'uber'))

    @property
    def alias(self):
        return self.uber_id

    @property
    def serialized_alias(self):
        return self.uber_id


class YambotAlias(UndeletableAlias):
    alias = Field(partial(_parse_alias, 'yambot'))


class KolonkishAlias(UndeletableAlias):
    alias = Field(partial(_parse_alias, 'kolonkish'))


class NeophonishAlias(UndeletableAlias):
    alias = Field(partial(_parse_alias, 'neophonish'))


class PublicIdAlias(MutableAlias):
    alias = Field(partial(_parse_alias, 'public_id'))
    old_public_ids = Field(partial(_parse_additional_logins, 'old_public_id'))

    def __init__(self, *args, **kwargs):
        self.old_public_ids = set()
        super(PublicIdAlias, self).__init__(*args, **kwargs)

    def add_old_public_id(self, login):
        self.old_public_ids.add(normalize_login(login))

    def has_old_public_id(self, login_to_check):
        return login_to_check in self.old_public_ids

    def remove_old_public_id(self, login):
        if login not in self.old_public_ids:
            raise ValueError('Attempt to remove non-existent old public id "%s"' % login)
        self.old_public_ids.remove(login)


class KiddishAlias(UndeletableAlias):
    alias = Field(partial(_parse_alias, 'kiddish'))


class ScholarAlias(UndeletableAlias):
    alias = Field(partial(_parse_alias, 'scholar'))


class BankPhoneNumberAlias(MutableAlias):
    alias = Field(partial(_parse_alias, 'bank_phonenumber'))


class FederalAlias(UndeletableAlias):
    # Алиас из ЧЯ возвращается в формате id_домена/id_пользователя_из_чужой_системы
    alias = Field(partial(_parse_alias, 'federal'))
