# -*- coding: utf-8 -*-

from copy import deepcopy

from passport.backend.core.models.base.fields import (
    DescriptorField,
    Field,
    FieldBase,
    ListField,
)
from passport.backend.core.undefined import Undefined
from six import (
    add_metaclass,
    iterkeys,
)


class Options(dict):
    __getattr__ = dict.__getitem__
    __setattr__ = dict.__setitem__

    def filter_fields(self, fields=None):
        if fields is not None:
            return filter(lambda x: x in self.fields, fields)
        else:
            return iterkeys(self.fields)


class ModelBase(type):
    def __new__(cls, name, bases, attrs):
        meta = Options(fields={})
        for parent in filter(lambda x: hasattr(x, '_meta'), bases):
            meta.fields.update(parent._meta.fields)
        for obj_name in attrs.keys():
            if isinstance(attrs[obj_name], FieldBase):
                meta.fields[obj_name] = attrs[obj_name]
                meta.fields[obj_name].attrname = obj_name
                if isinstance(attrs[obj_name], DescriptorField):
                    attrs[obj_name].default = Undefined
                else:
                    attrs[obj_name] = Undefined

        attrs['_meta'] = meta
        return type.__new__(cls, name, bases, attrs)


@add_metaclass(ModelBase)
class Model(object):

    def __new__(cls, *args, **kwargs):
        obj = super(Model, cls).__new__(cls)
        obj._id = id(obj)
        return obj

    def __init__(self, parent=None, **kwargs):
        if (parent is not None) and hasattr(self, 'parent'):
            self.parent = parent
        for key, value in kwargs.items():
            setattr(self, key, value)

    def parse(self, data, fields=None):
        return self._parse(data, fields)[1]

    def _parse(self, data, fields=None):
        fields = self._meta.filter_fields(fields)
        changed = False
        for key in fields:
            flag, value = self._meta.fields[key].parse(data, self, getattr(self, key, None))
            if flag:
                changed = True
                setattr(self, key, value)
        return changed, self

    def snapshot(self):
        return deepcopy(self)

    def __iter__(self):
        return iter((name, getattr(self, name)) for name in self._meta.fields)

    def __eq__(self, other):
        if other in (None, Undefined) or self.__class__ != other.__class__:
            return False
        return dict(self) == dict(other)

    def __ne__(self, other):
        return not self.__eq__(other)

    def __repr__(self):
        return "<%s: %r>" % (self.__class__.__name__, dict(filter(lambda x: x[1] is not Undefined, self)))

    def __setattr__(self, name, value):
        # Защищаемся от выставления незнакомых атрибутов на модель
        try:
            getattr(self, name)
        except AttributeError:
            if not name.startswith('_'):
                raise
        super(Model, self).__setattr__(name, value)

    def is_empty(self):
        return False


class ListModel(Model):
    parent = None

    # value нужно переопределить
    value = ListField(None)
    _to_append = Field()

    def append(self, value):
        self.value = [self._value_to_str(value)]
        self._to_append = True

    def set(self, value):
        # Вызов value может вести к аномалиям из-за гонок.
        if value is not None:
            self.value = [self._value_to_str(value)]
        else:
            self.value = None
        self._to_append = False

    def get(self):
        return map(self._str_to_value, self.value or [])

    def _value_to_str(self, value):
        return value

    def _str_to_value(self, _str):
        return _str


__all__ = (
    'Options',
    'ModelBase',
    'Model',
    'ListModel',
)
