# -*- coding: utf-8 -*-

from passport.backend.core.models.base.parse import (
    parse_boolean_field,
    parse_datetime_field,
    parse_integer_field,
    parse_list_field,
    parse_unixtime_field,
)
from passport.backend.core.undefined import Undefined
from six import iteritems


class FieldBase(object):
    def __init__(self, parser=None):
        if parser is not None:
            self.parser = self._create_parser(parser)
        else:
            self.parser = lambda x, *args: (False, None)

        self.attrname = None

    def parse(self, *args):
        return self.parser(*args)

    def _create_parser(self, parser):
        return parser


class Field(FieldBase):
    def _create_parser(self, key):
        def parse_key(data, *args):
            return key in data, data.get(key)

        return key if callable(key) else parse_key


class FlagsField(FieldBase):
    def __init__(self, key, bit_vector_class, **kwargs):
        self.data_key = key
        self.bit_vector_class = bit_vector_class
        super(FlagsField, self).__init__(key)

    def _create_parser(self, key):
        def parse_key(data, *args):
            if key in data:
                return True, self.bit_vector_class(data[key])
            else:
                return False, None
        return parse_key


class ModelField(FieldBase):
    def _create_parser(self, instance_class):
        def parse_model(data, parent, *args):
            obj = getattr(parent, self.attrname, None) or instance_class(parent)
            return True, obj.parse(data)

        return parse_model


class CollectionField(FieldBase):
    def __init__(self, key, instance_class, **kwargs):
        self.attrname = None
        self.data_key = key
        self.parser = self._create_parser(instance_class)

    def _create_parser(self, instance_class):
        def parse_collection(data, parent, value, *args):
            if self.data_key not in data:
                return False, None

            data = data[self.data_key]
            if isinstance(data, dict):
                collection = {} if value is Undefined else value
                for key, val in iteritems(data):
                    flag, obj = instance_class(parent)._parse(val)
                    if flag:
                        collection[key] = obj
            elif isinstance(data, list):
                collection = [] if value is Undefined else value
                for item in data:
                    flag, obj = instance_class(parent)._parse(item)
                    if flag:
                        collection.append(obj)
            else:
                return False, None

            return True, collection

        return parse_collection


class IntegerField(FieldBase):
    def __init__(self, key, default=None):
        super(IntegerField, self).__init__(parse_integer_field(key, default))


class BooleanField(FieldBase):
    def __init__(self, key):
        super(BooleanField, self).__init__(parse_boolean_field(key))


class UnixtimeField(FieldBase):
    def __init__(self, key):
        super(UnixtimeField, self).__init__(parse_unixtime_field(key))


class DateTimeField(FieldBase):
    def __init__(self, key):
        super(DateTimeField, self).__init__(parse_datetime_field(key))


class ListField(FieldBase):
    def __init__(self, key):
        super(ListField, self).__init__(parse_list_field(key))


class DescriptorField(FieldBase):
    def __init__(self, field, getter, setter):
        self._getter = getter
        self._setter = setter
        self._field = field

    def __get__(self, obj, type=None):
        try:
            return self._getter(obj)
        except AttributeError:
            return self.default

    def __set__(self, obj, value):
        self._setter(obj, value)

    def parse(self, *args):
        return self._field.parse(*args)


class TypedField(Field):
    def __init__(self, key, cls, allow_empty=True):
        super(TypedField, self).__init__(key)
        self.cls = cls
        self.allow_empty = allow_empty

    def parse(self, *args):
        flag, value = super(TypedField, self).parse(*args)
        if flag or self.allow_empty:
            return True, self.cls(value)

        return flag, value
