# -*- coding: utf-8 -*-
import copy
import json
import logging

from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    DateTimeField,
    Field,
    IntegerField,
    UnixtimeField,
)
from passport.backend.core.undefined import Undefined
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)
from six import iteritems


log = logging.getLogger('passport.models.domain')

# Основной домен
MASTER_DOMAIN = 'M'
# Подчиненный домен
ALIAS_DOMAIN = 'A'

CAN_USERS_CHANGE_PWD_OPTION = '1'
ORGANIZATION_NAME_OPTION = '2'
DISPLAY_MASTER_ID_OPTION = '3'
GLOGOUT_TIME_OPTION = '4'

OPTIONS_TO_DATA_MAPPING = {
    CAN_USERS_CHANGE_PWD_OPTION: 'can_users_change_password',
    'can_users_change_password': 'can_users_change_password',

    ORGANIZATION_NAME_OPTION: 'organization_name',
    'organization_name': 'organization_name',

    DISPLAY_MASTER_ID_OPTION: 'display_master_id',
    'display_master_id': 'display_master_id',

    GLOGOUT_TIME_OPTION: 'glogout_time',
    'glogout_time': 'glogout_time',
}

# чтоб в будущем переключить схемы было проще добавляем новые параметры сразу числами
OLD_SCHEME_OPTIONS = [
    'can_users_change_password',
    'organization_name',
    'display_master_id',
    GLOGOUT_TIME_OPTION,  # glogout_time
]

OLD_TO_NEW_SCHEME = {
    'can_users_change_password': CAN_USERS_CHANGE_PWD_OPTION,
    'organization_name': ORGANIZATION_NAME_OPTION,
    'display_master_id': DISPLAY_MASTER_ID_OPTION,
    'glogout_time': GLOGOUT_TIME_OPTION,
}

NEW_SCHEME_OPTIONS = [
    CAN_USERS_CHANGE_PWD_OPTION,
    ORGANIZATION_NAME_OPTION,
    DISPLAY_MASTER_ID_OPTION,
    GLOGOUT_TIME_OPTION,
]


def _parse_enabled(data, domain, *args):
    """
    Разбираем данные о флаге состояния домена, принимая
    на вход как ответ от userinfo, так и от hosted_domains.
    """

    if 'domain_ena' in data:
        return True, bool(int(data['domain_ena']))
    elif 'ena' in data:
        return True, bool(int(data['ena']))
    else:
        return False, None


def _parse_domain_type(data, domain, *args):
    """
    Разбираем данные о типе домена. Если у него указан
    "головной" домен, то считаем его алиасом.
    """
    if 'master_domain' in data:
        return True, ALIAS_DOMAIN if data['master_domain'] else MASTER_DOMAIN
    return False, False


def _extract_hosted_domain(data):
    """
    FIXME: Если нам передают ответ от hosted_domains, то мы
    можем из него обработать только первый доме, в то время как
    их может быть несколько. На момент парсинга мы можем ничего не
    знать о заполняемом домене (к примеру, на пустой объект Domain()
    вызывается parse с ответом от hosted_domains). Соответственно,
    никаких решений относительно того, какой элемент является для нас
    основным и что делать с остальными мы принять не можем.

    Также возможна ситуация, что нам передадут ответ hosted_domains
    по uid пользователя = много доменов, причем каждый из них может
    быть основным, а не алиасом.

    Именно поэтому мы в данный момент используем уже имеющуюся
    эвристику - рассматриваем только первый элемент ответа в надежде,
    что запрашивалась информация по конкретному домену.
    """
    try:
        return data['hosted_domains'][0]
    except (IndexError, KeyError):
        return data


def _parse_aliases(data, domain, *args):
    """
    Обрабатываем данные о списке подчиненных доменов (алиасов).
    """
    slaves = data.get('slaves')
    if slaves:
        try:
            raw_slave_list = data['slaves'].split(',')
            slave_list = []
            for slave_name in raw_slave_list:
                # Если к нам пришёл домен в punycode, то автоматически попытаемся декодировать его
                if 'xn--' in slave_name:
                    try:
                        slave_name = smart_bytes(slave_name).decode('idna')
                    except UnicodeError:
                        raise ValueError('Malformed IDNA slave: %s' % slave_name)
                slave_list.append(slave_name)

            return True, slave_list
        except AttributeError:
            log.warning(
                'Invalid slave domain list for domain "%s": %s',
                domain.domain,
                slaves,
            )
            return False, Undefined
    return True, []


class Domain(Model):
    """
    Модель ПДД-домена.

    Данная модель может десериализовываться как из данных ответа функции ЧЯ
    userinfo (поле uid массива users), так и из элемента полноценного ответа ЧЯ
    hosted_domains.

    Документация.
    http://doc.yandex-team.ru/blackbox/reference/method-userinfo-response-json.xml
    http://doc.yandex-team.ru/Passport/mail-for-domain/api/domain_def.xml
    """

    # UID администратора, который может управлять доменом.
    admin_uid = IntegerField('admin')

    # Используется ли маршрутизация почты через серверы Яндекса
    # (указывает ли на них MX-запись в DNS).
    is_yandex_mx = BooleanField('mx')

    # Обозначает UID ящика, в который отправляются все письма,
    # пришедшие на доменный адрес, которого не существует.
    default_uid = IntegerField('default_uid')

    # Строчка с именем головного домена. У алиасов это поле заполнено.
    master_domain = Field('master_domain')

    # Дата создания домена.
    registration_datetime = DateTimeField('born_date')

    # Тип домена (основной, алиас).
    type = Field(_parse_domain_type)

    # Список алиасов (подчиненных доменов) в виде строчек с доменными именами.
    aliases = Field(_parse_aliases)

    # Маппинг, использующийся при сериализации домена
    alias_to_id_mapping = None

    # Возможность смены паролей через интерфейс Яндекса:
    # False — запрещено менять пароли, True — разрешено менять пароли.
    # По умолчанию пользователь может менять пароль, для получения актуального значения
    # нужно дополнительно вызвать parse с результатом метода ЧЯ hosted_domains.
    can_users_change_password = Field('can_users_change_password')

    organization_name = Field('organization_name')

    # ID домена, который используется ЧЯ для отображения "красивого" e-mail'а
    display_master_id = Field('display_master_id')

    # время глогаута всех пользователей на домене
    glogout_time = UnixtimeField('glogout_time')

    # Когда-то здесь было поле 'change_password_url', которое было задействовано
    # в никогда не анонсировавшемся функционале и не содержало ничего интересного.
    # После обсуждения с Пашей Кириченко (@juanych) было решено выпилить его.
    # 12.02.2015

    parent = None

    # Идентификатор домена.
    id = IntegerField('domid')

    # Состояние домена.
    is_enabled = Field(_parse_enabled)

    # Доменное имя
    domain = Field('domain')

    def __init__(self, *args, **kwargs):
        super(Domain, self).__init__(*args, **kwargs)
        self._initial_options = {}
        self.alias_to_id_mapping = {}

    def parse(self, data, fields=None):
        """
        Эвристика для случая, когда мы обрабатываем ответ от метода ЧЯ
        hosted_domains.
        """
        domain_data = _extract_hosted_domain(data)
        self.can_users_change_password = '1'

        try:
            options_text = domain_data.get('options') or '{}'
            options = json.loads(options_text)
            self._initial_options = copy.copy(options)

            for key, field in iteritems(OPTIONS_TO_DATA_MAPPING):
                if key in options:
                    domain_data[field] = options[key]
        except ValueError:
            log.warning(
                'Invalid JSON in "options" field: %s',
                options_text,
            )

        instance = super(Domain, self).parse(
            domain_data,
            fields=fields,
        )
        instance.alias_to_id_mapping = {}

        # Если нам передали домен в punycode, то автоматически попытаемся декодировать его.
        if instance.domain and 'xn--' in instance.domain:
            try:
                instance.domain = smart_bytes(instance.domain).decode('idna')
            except UnicodeError:
                raise ValueError('Malformed IDNA domain: %s' % instance.domain)
        self.can_users_change_password = bool(int(self.can_users_change_password))
        return instance

    @property
    def punycode_domain(self):
        return smart_text(self.domain.encode('idna'))

    @property
    def unicode_domain(self):
        return self.domain

    @property
    def is_master(self):
        return self.type == MASTER_DOMAIN

    @property
    def is_alias(self):
        return self.type == ALIAS_DOMAIN

    def get_alias_id(self, alias_domain):
        # К сожалению, при чтении домена ИЗ ЧЯ мы не получаем id его алиасов автоматически.
        # Приходится заполнять alias_to_id_mapping вручную. Поэтому проверим, что не забыли его заполнить.
        if alias_domain not in self.alias_to_id_mapping:
            raise ValueError('Alias `%s` not found in self.alias_to_id_mapping' % alias_domain)
        return self.alias_to_id_mapping[alias_domain]


class PartialPddDomain(Domain):
    """
    Огрызок основной модели домена ПДД, который используется при разборе
    ответа от userinfo на стадии заполнения информации об аккаунте.

    Оставлен здесь для совместимости, так как полноценная модель домена
    на аккаунте не имеет смысла и вызывает проблемы при текущей реалиазации
    механизма сериализации.

    Избавиться от неё просто - убрав весь код, который использует эти данные
    на аккаунте и заменить их на явные вызовы hosted_domains с преобразованием
    в модель домена. На данный момент основной проблемой представляется то, что
    в нашей концепции сериализации нельзя получить достаточно данных для
    заполнения данных о домене только из ответа userinfo, а делать
    дополнительные вызовы сервисов в парсере не считается кошерным.

    Данная модель не сериализуется!
    """
