# -*- coding: utf-8 -*-

from itertools import chain
import logging

from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    DateTimeField,
    Field,
    IntegerField,
    UnixtimeField,
)
from passport.backend.core.undefined import Undefined
from passport.backend.utils.string import smart_text
import six
from six import itervalues


log = logging.getLogger('passport.models.email')


@six.python_2_unicode_compatible
class Email(Model):
    parent = None
    id = IntegerField('id')

    # Ненормализованный e-mail пользователя
    address = Field('address')

    # Признак почтового ящика, письма с которого забирает почтовый сборщик Яндекса.
    # Детали использования:
    # http://wiki.yandex-team.ru/passport/blackbox#07.12.2010rpopemailattribute
    is_rpop = BooleanField('rpop')

    # Признак внутреннего email.
    is_native = BooleanField('native')

    # Признак адреса, который был подтвержден небезопасно.
    is_unsafe = BooleanField('unsafe')

    # Признак того, что на данный email нельзя отправлять нотификации.
    is_silent = BooleanField('silent')

    # Признак того, что email является адресом по умолчанию.
    is_default = BooleanField('default')

    # Момент подтверждения данного e-mail'а через валидатор в питоновском
    # Паспорте
    confirmed_at = UnixtimeField('confirmed')

    # Момент создания записей о e-mail'е в новой паспортной базе
    created_at = UnixtimeField('created')

    # Момент "привязки" адреса к аккаунту
    bound_at = UnixtimeField('bound')

    # Признак подтвержденности адреса.
    _validated = BooleanField('validated')

    # Время проведения последней операции с этим email'ом
    # в старой БД валидатора.
    _last_changed_in_old_db = DateTimeField('born-date')

    def __init__(self, *args, **kwargs):
        super(Email, self).__init__(*args, **kwargs)
        self._username = None
        self._domain = None

    def _try_fill_empty_attributes(self):
        """
        Если email взят не из атрибутов (например, он нативный), то у него останутся незаполненными времена
        создания, подтверждения и привязки. Попробуем это исправить.
        """
        if not self._last_changed_in_old_db:
            return
        if self.created_at is Undefined:
            self.created_at = self._last_changed_in_old_db
        if self.bound_at is Undefined and self._validated:
            self.bound_at = self._last_changed_in_old_db
        if self.confirmed_at is Undefined and self._validated:
            self.confirmed_at = self._last_changed_in_old_db

    def _parse(self, data, fields=None):
        parse_response = super(Email, self)._parse(data, fields)
        self._try_fill_empty_attributes()
        return parse_response

    @property
    def is_confirmed(self):
        return bool(self.confirmed_at) or self._validated

    # Признак внешнего email.
    @property
    def is_external(self):
        return not self.is_native

    @property
    def is_suitable_for_restore(self):
        return bool(
            self.is_confirmed and
            self.is_external and
            not self.is_rpop and
            not self.is_unsafe and
            not self.is_silent,
        )

    @property
    def username(self):
        if not self._username and '@' in self.address:
            self._username, self._domain = self._split(self.address)
        return self._username

    @property
    def domain(self):
        if not self._domain and '@' in self.address:
            self._username, self._domain = self._split(self.address)
        return self._domain

    @staticmethod
    def _split(address):
        return address.rsplit('@', 1)

    @classmethod
    def split(cls, address):
        if '@' in address:
            return cls._split(address)
        else:
            return None, None

    @staticmethod
    def normalize_address(address, encode_domain=True):
        if not address or '@' not in address:
            raise ValueError('Can\'t normalize invalid email address: %r' % address)

        username, domain = address.rsplit('@', 1)
        username = username.strip()
        domain = domain.strip()

        if encode_domain:
            try:
                domain = smart_text(domain).encode('idna')
            except UnicodeError:
                raise ValueError('Domain can\'t be encoded in IDNA: %s' % domain)

        return '@'.join([username, smart_text(domain)]).lower()

    @property
    def normalized_address(self):
        return Email.normalize_address(self.address)

    def __str__(self):
        return smart_text(self.address)

    def __repr__(self):
        return '<Email: %s>' % str(self)

    def __lt__(self, other):
        if not isinstance(other, Email):
            raise NotImplementedError('Can compare Email to Email only, got {}'.format(type(other)))  # noqa
        return self.address < other.address


class Emails(Model):
    parent = None
    _emails = Field()

    def __init__(self, *args, **kwargs):
        super(Emails, self).__init__(*args, **kwargs)
        self._emails = {}
        self._default = None

    def _parse(self, data, fields=None):
        if not ('address-list' in data or 'emails' in data):
            return False, self

        # Сначала обработаем адреса из старой базы валидатора,
        # затем добавим адреса, полученные ЧЯ из extended_attributes
        # паспортной базы
        for entry in chain(
            data.get('address-list', []),
            data.get('emails', []),
        ):
            instance = Email().parse(entry)
            try:
                self[instance.address] = instance
            except ValueError as e:
                # Если какой-то из адресов некорректный, то пропускаем его и
                # пишем в лог, предварительно почистив от переносов строк.
                msg = 'Invalid email address "%s": %s' % (instance.address, e.args[0])
                msg = msg.strip().replace('\r', '').replace('\n', '')
                log.debug(msg)

        return True, self

    def __getitem__(self, address):
        normalized = Email.normalize_address(address)
        item = self._emails.get(normalized)

        if item is None:
            raise KeyError(address)

        return item

    def __setitem__(self, address, email):
        normalized = Email.normalize_address(address)
        self._emails[normalized] = email
        email.parent = email.parent or self

        if email.is_default:
            self._default = email

    def __delitem__(self, address):
        self.pop(address)

    def __contains__(self, address):
        normalized = Email.normalize_address(address)
        return normalized in self._emails

    def add(self, instance):
        self[instance.address] = instance

    def find(self, address):
        normalized = Email.normalize_address(address)
        return self._emails.get(normalized)

    def pop(self, address):
        normalized = Email.normalize_address(address)
        email = self._emails.pop(normalized)
        if email.is_default:
            self._default = None

    @property
    def default(self):
        return self._default

    @property
    def native(self):
        return [
            email
            for email in self.all
            if email.is_native
        ]

    @property
    def external(self):
        return [
            email
            for email in self.all
            if email.is_external
        ]

    @property
    def confirmed_external(self):
        return [
            email
            for email in self.all
            if email.is_confirmed and email.is_external
        ]

    @property
    def suitable_for_restore(self):
        return [
            email
            for email in self.all
            if email.is_suitable_for_restore
        ]

    @property
    def all(self):
        # тут всю жизнь жили без сортировки, но она нужна для тестов PY23
        # т.к. порядок сортировки itervalues разный между версиями питона
        return sorted(itervalues(self._emails), key=lambda e: e.address)

    def __len__(self):
        return len(self._emails)

    def __repr__(self):
        return '<Emails: [%s]>' % (
            ', '.join([repr(e) for e in self.all]),
        )

    def is_empty(self):
        return not self._emails

    def get_by_id(self, email_id):
        for email in self._emails.values():
            if email_id == email.id:
                return email
        return None


def build_emails(bb_emails):
    return Emails().parse({
        'address-list': bb_emails,
    })
