# -*- coding: utf-8 -*-

import uuid

from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    CollectionField,
    Field,
    FieldBase,
    IntegerField,
    ModelField,
)
from passport.backend.core.undefined import Undefined
from passport.backend.utils.string import smart_text
from six import (
    integer_types,
    iteritems,
    string_types,
)


class FamilyIdBase(Model):
    """
    Добавлялка магии family_id
    """
    _family_id = IntegerField('family_id')

    # Возвращаем family_id как строку с префиксом f
    @property
    def family_id(self):
        return ('f%s' % self._family_id) if self._family_id else self._family_id

    def parse(self, data, fields=None):
        super(FamilyIdBase, self).parse(data, fields)
        if 'family_id' in data:
            self.family_id = data['family_id']
        return self

    @family_id.setter
    def family_id(self, value):
        try:
            if isinstance(value, string_types):
                try:
                    if value[0] != 'f':
                        raise ValueError('Doesn\'t start with "f"')
                except IndexError:
                    raise ValueError('Empty value')
                self._family_id = int(value[1:])
            elif isinstance(value, integer_types):
                self._family_id = value
            else:
                raise ValueError('Not int or str')
        except ValueError as err:
            raise ValueError(
                'family_id must be int or string "fxxx" '
                'where xxx is a number (%s: %s)' % (repr(value), err),
            )


class FamilyMember(Model):
    """
    Запись о члене семьи
    """
    parent = None
    uid = IntegerField('uid')
    # Занимаемое место в семье: уникальное для family_id число
    place = IntegerField('place')

    def __repr__(self):
        return '<FamilyMember family_id=%s uid=%s place=%s>' % (
            self.parent.family_id if self.parent else self.parent,
            self.uid,
            self.place,
        )


class FamilyMembersMixin(object):
    def __contains__(self, item):
        if self.members is Undefined:
            return False
        if isinstance(item, FamilyMember):
            item = item.uid
        if isinstance(item, integer_types):
            return item in self.members
        elif isinstance(item, string_types):
            return int(item) in self.members
        else:
            raise ValueError('"Family contains" check accepts FamilyMember or str')

    def get_first_free_place(self, total_places):
        if len(self.members) >= total_places:
            return None
        occupied_places = {v.place for v in self.members.values()}
        for place in range(self.first_place_offset, self.first_place_offset + total_places):
            if place not in occupied_places:
                return place

    def add_member_uid(self, uid, place):
        if self.members is Undefined:
            self.members = {}
        if uid in self.members:
            raise ValueError('Member uid %s already exists' % uid)
        self.members[int(uid)] = FamilyMember(uid=uid, place=place)

    def remove_member_uid(self, uid):
        try:
            if self.members is Undefined:
                raise KeyError()
            del self.members[int(uid)]
        except KeyError:
            raise ValueError('Member uid %s does not exist' % uid)

    def get_member_uid_by_place(self, place):
        if self.members is Undefined:
            return None
        for member in self.members.values():
            if member.place == place:
                return member.uid


class AdultsField(FieldBase):
    def __init__(self, key):
        self._field = CollectionField(key, FamilyMember)
        self.attrname = None
        self.data_key = key

    def parse(self, data, *args):
        if self.data_key not in data:
            return False, None
        data = {i: u for i, u in iteritems(data[self.data_key]) if not u.get('is_kid')}
        data = {self.data_key: data}
        return self._field.parse(data, *args)


class KidsField(FieldBase):
    def __init__(self, key):
        self._field = CollectionField(key, FamilyMember)
        self.attrname = None
        self.data_key = key

    def parse(self, data, *args):
        if self.data_key not in data:
            return False, None
        data = {i: u for i, u in iteritems(data[self.data_key]) if u.get('is_kid')}
        data = {self.data_key: data}
        return self._field.parse(data, *args)


class KidsCollection(FamilyMembersMixin, Model):
    members = KidsField('users')
    first_place_offset = 100


class FamilyInfo(FamilyMembersMixin, FamilyIdBase):
    """
    Запись о семье
    """
    # family_id из FamilyIdBase
    admin_uid = IntegerField('admin_uid')
    meta = Field('meta')
    members = AdultsField('users')
    first_place_offset = 0
    kids = ModelField(KidsCollection)

    def is_uid_admin(self, uid):
        return self.admin_uid == int(uid)

    def __repr__(self):
        return '<FamilyInfo family_id=%s admin_uid=%s>' % (
            self.family_id,
            self.admin_uid,
        )


class AccountFamilyInfo(FamilyIdBase):
    """
    Запись о семье как дополнение к Account (get_family_info)
    """
    # family_id из FamilyIdBase
    admin_uid = IntegerField('admin_uid')

    def parse(self, data, fields=None):
        data = data.get('family_info', {})
        return super(AccountFamilyInfo, self).parse(data, fields)

    def __repr__(self):
        return '<UserFamilyInfo family_id=%s admin_uid=%s>' % (
            self.family_id,
            self.admin_uid,
        )

    def __nonzero__(self):
        return bool(self.family_id)

    def __bool__(self):
        return self.__nonzero__()


class FamilyInvite(FamilyIdBase):
    """
    Приглашение в семью
    """
    SEND_METHOD_EMAIL = 1
    SEND_METHOD_SMS = 2

    # family_id из FamilyIdBase
    invite_id = Field('invite_id')
    issuer_uid = IntegerField('issuer_uid')
    send_method = IntegerField('send_method')
    create_time = IntegerField('create_time')
    contact = Field('contact')

    @classmethod
    def generate(cls, family_id, send_method, contact, uid, create_time):
        invite = cls().parse({
            'family_id': family_id,
            'issuer_uid': uid,
            'invite_id': str(uuid.uuid4()),
            'create_time': create_time,
            'send_method': send_method,
            'contact': smart_text(contact),
        })
        return invite

    @classmethod
    def send_method_to_text(cls, send_method):
        if send_method is None or send_method is Undefined:
            return send_method
        elif send_method == 0:
            return 'none'
        elif send_method == cls.SEND_METHOD_EMAIL:
            return 'email'
        elif send_method == cls.SEND_METHOD_SMS:
            return 'sms'
        else:
            raise ValueError('Unknown send method %s' % send_method)
