# -*- coding: utf-8 -*-

import base64
from collections import namedtuple
from datetime import datetime
from hashlib import sha256
import logging

from passport.backend.core.builders.blackbox.blackbox import Blackbox
from passport.backend.core.builders.blackbox.constants import BLACKBOX_BRUTEFORCE_PASSWORD_EXPIRED_STATUS
from passport.backend.core.builders.blackbox.exceptions import BaseBlackboxError
from passport.backend.core.crypto.utils import hash_string
from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    DescriptorField,
    Field,
    UnixtimeField,
)
from passport.backend.core.models.subscription import build_sid_property
from passport.backend.core.password import policy
from passport.backend.core.types.login.login import raw_login_from_email
from passport.backend.core.undefined import Undefined
from passport.backend.utils.string import smart_bytes


log = logging.getLogger('passport.models.password')

sid_property = build_sid_property(lambda self: self.parent)


PASSWORD_ENCODING_VERSION_MD5_CRYPT = 1
PASSWORD_ENCODING_VERSION_BCRYPT = 2
PASSWORD_ENCODING_VERSION_MD5_HEX = 3
PASSWORD_ENCODING_VERSION_ESCRYPT = 4
PASSWORD_ENCODING_VERSION_MD5_CRYPT_UID = 5
PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON = 6
PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON = 7


# Возможные значения для поля forced_changing_reason
PASSWORD_CHANGING_REASON_HACKED = '1'
PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN = '2'
PASSWORD_CHANGING_REASON_PWNED = '3'

PASSWORD_CHANGING_REASON_BY_NAME = {
    'PASSWORD_CHANGING_REASON_HACKED': PASSWORD_CHANGING_REASON_HACKED,
    'PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN': PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN,
    'PASSWORD_CHANGING_REASON_PWNED': PASSWORD_CHANGING_REASON_PWNED,
}


def check_password_history(password, uid, depth, reason=None):
    if not uid or depth <= 0:
        return False

    data = Blackbox().pwdhistory(
        uid=uid,
        password=password,
        depth=depth,
        reason=reason,
    )
    return data['found_in_history']


def check_password_equals_hash(password, password_hash_with_version, uid):
    if not password_hash_with_version:
        return False
    encoded_hash = base64.b64encode(password_hash_with_version.encode()).decode()
    status = Blackbox().test_pwd_hashes(password=password, hashes=[encoded_hash], uid=uid)
    return status[encoded_hash]


def get_sha256_hash(password):
    return sha256(smart_bytes(password)).hexdigest()


def check_password_equals_sha256_hash(password, sha_256_password_hash):
    if not sha_256_password_hash:
        return False
    return get_sha256_hash(password) == sha_256_password_hash


def _parse_expired(data, *args):
    if 'bruteforce_policy' in data:
        bruteforce_status = data['bruteforce_policy'].get('value')
        is_expired = (bruteforce_status == BLACKBOX_BRUTEFORCE_PASSWORD_EXPIRED_STATUS)
        return True, is_expired
    return None, None


def _parse_serialized_password(value):
    try:
        version, password_hash = value.split(':', 1)
        if not version or not password_hash:
            raise ValueError('Password version or hash is empty')
        return int(version), password_hash
    except (ValueError, TypeError) as e:
        log.error('Invalid serialized password value: %s', e)


def _parse_password_hash(field_name):
    def _parse(data, *args):
        serialized_value = data.get(field_name)
        if serialized_value:
            values = _parse_serialized_password(serialized_value)
            if values:
                _, encrypted = values
                return True, encrypted
        return None, None

    return _parse


def _parse_password_encoding_version(field_name):
    def _parse(data, *args):
        serialized_value = data.get(field_name)
        if serialized_value:
            values = _parse_serialized_password(serialized_value)
            if values:
                version, _ = values
                return True, version
        return None, None

    return _parse


def build_password_subwords(account):
    """По указанному аккаунту собираем все слова, которые не должны присутствовать в пароле для этого аккаунта"""
    subwords = []
    if account.is_lite:
        # Лайт-аккаунты имеют login вида "username@domain.com"
        email = account.login
        login = raw_login_from_email(email)
        subwords.append(email)
    else:
        login = account.login
        domain = account.domain.domain if account.domain else None
        # Проверяем на соответствие с email только для ПДД-пользователей,
        # причем domain здесь закодирован в punicode, ибо ЧЯ отдает так
        if domain:
            email = '%s@%s' % (login, domain)
            subwords.append(email)

    subwords.append(login)
    return subwords


CreateHashResult = namedtuple('CreateHashResult', 'is_created version encrypted_password')


class PasswordHashProvider(object):
    def __init__(self, password, version, get_hash_from_blackbox=False, salt=None, intermediate_hash=None):
        self.password = password
        self.version = version
        self.get_hash_from_blackbox = get_hash_from_blackbox
        self.salt = salt
        self.intermediate_hash = intermediate_hash
        self.encrypted_password = None

    def _make_params_for_blackbox(self, uid=None):
        params = {
            'version': self.version,
            'uid': uid,
        }
        if self.password is not None:
            params['password'] = self.password
        elif self.intermediate_hash is not None:
            if self.version == PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON:
                params['md5crypt_hash'] = self.intermediate_hash
            elif self.version == PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON:
                params['rawmd5_hash'] = self.intermediate_hash
        return params

    def try_create_hash(self, uid=None):
        if self.get_hash_from_blackbox and not uid:
            return CreateHashResult(False, None, None)
        elif self.encrypted_password:
            return CreateHashResult(True, self.version, self.encrypted_password)
        elif self.get_hash_from_blackbox:
            serialized_value = Blackbox().create_pwd_hash(**self._make_params_for_blackbox(uid=uid))
            values = _parse_serialized_password(serialized_value)
            if not values:
                raise BaseBlackboxError()
            version, self.encrypted_password = values
            if version != self.version:
                log.error('Unexpected version from create_pwd_hash: %s', version)
                raise BaseBlackboxError()
            return CreateHashResult(True, self.version, self.encrypted_password)

        if self.version == PASSWORD_ENCODING_VERSION_MD5_CRYPT:
            self.encrypted_password = hash_string(self.password, self.salt).decode()
            return CreateHashResult(True, self.version, self.encrypted_password)
        else:
            raise ValueError('Don\'t know how to create hash version %s' % self.version)


def default_encrypted_password(password, uid, salt=None):
    provider = PasswordHashProvider(password, version=PASSWORD_ENCODING_VERSION_MD5_CRYPT, salt=salt)
    result = provider.try_create_hash(uid)
    assert result.is_created
    return '%s:%s' % (result.version, result.encrypted_password)


class BasePasswordMixin(object):
    @property
    def is_set(self):
        return bool(self.encrypted_password)

    @property
    def is_set_or_promised(self):
        # Пароль уже установлен, либо будет установлен при сериализации
        return self.is_set or bool(self.password_hash_provider)

    @property
    def serialized_encrypted_password(self):
        if not self.encrypted_password or not self.encoding_version:
            return Undefined
        return '%s:%s' % (self.encoding_version, self.encrypted_password)

    def set(
        self,
        password,
        quality,
        salt=None,
        version=PASSWORD_ENCODING_VERSION_MD5_CRYPT,
        get_hash_from_blackbox=False,
    ):
        """
        Сохраняет хеш пароля заданной версии с заданной или случайной солью.
        Возможно вычисления хеша с помощью ЧЯ, в этом случае вычисление может
        происходить во время сериализации, если аккаунт только создается, т.к.
        для вычисления может быть необходим UID.
        """
        self._try_create_hash(
            password=password,
            salt=salt,
            use_blackbox=get_hash_from_blackbox,
            version=version,
        )

        self.quality = quality
        self.quality_version = 3
        self.update_datetime = datetime.now()

        return self

    def set_hash(
        self,
        password_hash,
        version=PASSWORD_ENCODING_VERSION_MD5_CRYPT,
        try_create_hash=True,
    ):
        if version in (PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON, PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON) and try_create_hash:
            self._try_create_hash(
                intermediate_hash=password_hash,
                use_blackbox=True,
                version=version,
            )
        else:
            self.encrypted_password = password_hash
            self.encoding_version = version

        self.quality = Undefined
        self.quality_version = Undefined
        self.update_datetime = datetime.now()

        return self

    def _try_create_hash(
        self,
        version,
        use_blackbox=False,
        salt=None,
        password=None,
        intermediate_hash=None,
    ):
        self.password_hash_provider = PasswordHashProvider(
            get_hash_from_blackbox=use_blackbox,
            intermediate_hash=intermediate_hash,
            password=password,
            salt=salt,
            version=version,
        )
        uid = self.parent.uid if self.parent else None
        result = self.password_hash_provider.try_create_hash(uid=uid)
        if result.is_created:
            self.encoding_version, self.encrypted_password = result.version, result.encrypted_password


class Password(
    BasePasswordMixin,
    Model,
):
    """
    Модель пароля, тесно связанная с аккаунтом.
    """
    parent = None

    encrypted_password = Field(_parse_password_hash('password.encrypted'))
    # Версия кодирования пароля. Поле нужно для того, чтобы иметь возможность записывать в базу
    # хеши пароля в различных форматах.
    encoding_version = Field(_parse_password_encoding_version('password.encrypted'))

    password_hash_provider = None

    quality = Field('password_quality.quality.uid')
    quality_version = Field('password_quality.version.uid')
    length = Field()

    # истекший срок годности
    # Берется из ответа ЧЯ по методу login
    # Появляется когда у пользователя есть sid=67
    is_expired = Field(_parse_expired)

    # В старом ЧЯ: если установлен 3 бит на subscription.login_rule.8
    # В атрибуте зашита информация о причине смены пароля, см. возможные значения выше.
    forced_changing_reason = Field('password.forced_changing_reason')
    forced_changing_time = UnixtimeField('password.forced_changing_time')
    pwn_forced_changing_suspended_at = UnixtimeField('password.pwn_forced_changing_suspended_at')

    update_datetime = UnixtimeField('password.update_datetime')

    def setup_password_changing_requirement(self, is_required=True, changing_reason=PASSWORD_CHANGING_REASON_HACKED):
        """
        Установить или снять требование принудительной смены пароля
        @param is_required: признак установки требования (отрицательное значение - снятие требования)
        @param changing_reason: причина требования, используется только при положительном значении is_required
        """
        reason, when = None, None
        if is_required:
            reason = changing_reason
            when = datetime.now()
        else:
            # PASSP-34095: индульгенция на принуждённую смену пароля для PWNED
            if self.forced_changing_reason == PASSWORD_CHANGING_REASON_PWNED:
                self.pwn_forced_changing_suspended_at = datetime.now()

        self.forced_changing_reason = reason
        self.forced_changing_time = when

    @property
    def is_creating_required(self):
        """Принудительная смена пароля и контрольного вопроса на след. входе.

        login_rule=1 – смена пароля (редирект на mode=setpasswd)
        login_rule=0 – ничего не делать
        login_rule=2 – отменить смену пароля
        (для обычных юзеров после mode=setpasswd, для pdd-юзеров после mode=mdfillinfo – завершение регистрации);
        почему нельзя сбросить в 0 – никто уже не помнит

        """
        if 100 not in (self.parent.subscriptions or {}):
            return False
        else:
            return self.parent.subscriptions[100].login_rule == 1

    @is_creating_required.setter
    def is_creating_required(self, value):
        if 100 not in (self.parent.subscriptions or {}):
            if value:
                self.parent.parse({
                    'subscriptions': {
                        100: {
                            'sid': 100,
                            'login_rule': 1,
                        },
                    },
                })
        else:
            if value:
                self.parent.subscriptions[100].login_rule = 1
            else:
                del self.parent.subscriptions[100]

    @property
    def is_complication_required(self):
        """
        Требуется усложнение пароля
        """
        return self.parent.is_strong_password_required and self.is_weak

    @property
    def is_new_password_required(self):
        """
        Требуется новый пароль: нужно его создать или он требуется по иным причинам.
        Определяется по 4 флагам, в порядке сложности вычисления флага.
        """
        return self.is_creating_required or self.is_changing_required_by_any_reason

    @property
    def is_changing_required(self):
        return bool(self.forced_changing_reason)

    @property
    def is_changing_required_by_any_reason(self):
        """
        Требуется сменить пароль на новый по иным причинам:
        - выставлено требование принудительной смены пароля
        - действует политика защищённых аккаунтов со сложным паролем

        Определяется по 3 флагам, в порядке сложности вычисления флага.
        """
        return self.is_changing_required or self.is_expired or self.is_complication_required

    @property
    def is_password_flushed_by_admin(self):
        """
        Требование смены пароля админом ПДД. Используется для workspace-пользователей.
        """
        return self.forced_changing_reason == PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN

    @property
    def is_password_pwned(self):
        """
        Требование смены пароля, так как он слит.
        """
        return self.forced_changing_reason == PASSWORD_CHANGING_REASON_PWNED

    @property
    def is_weak(self):
        # проверяем только наличие качества - длину мы можем не знать
        if not self.is_set:
            raise ValueError('password not set')

        # Если нам ничего неизвестно о качестве пароля (возможно, это вручную
        # проставленный хэш), то на всякий случай считаем его слабым.
        if self.quality is Undefined or self.quality is None:
            return True

        password_policy = policy.strong() if self.parent.is_strong_password_required else policy.basic()

        middle_quality = password_policy.middle_quality
        min_length = password_policy.min_length
        return self.quality < middle_quality or (self.length and self.length < min_length)


class ScholarPassword(
    BasePasswordMixin,
    Model,
):
    encoding_version = Field(_parse_password_encoding_version('account.scholar_password'))
    encrypted_password = Field(_parse_password_hash('account.scholar_password'))
    parent = None
    password_hash_provider = None
    quality = None
    quality_version = None
    update_datetime = Field()

    def _get_serialized_encrypted_password(self):
        return super(ScholarPassword, self).serialized_encrypted_password

    def _set_serialized_encrypted_password(self, value):
        raise NotImplementedError()

    serialized_encrypted_password = DescriptorField(
        Field(),
        _get_serialized_encrypted_password,
        _set_serialized_encrypted_password,
    )

    def set(
        self,
        password,
        quality=None,
        salt=None,
        version=PASSWORD_ENCODING_VERSION_MD5_CRYPT,
        get_hash_from_blackbox=False,
    ):
        return super(ScholarPassword, self).set(
            get_hash_from_blackbox=get_hash_from_blackbox,
            password=password,
            quality=quality,
            salt=salt,
            version=version,
        )
