# -*- coding: utf-8 -*-
from datetime import (
    datetime,
    timedelta,
)
import logging
from os import urandom

from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import Field
from passport.backend.core.models.base.parse import parse_unixtime_field
from passport.backend.core.undefined import Undefined
from passport.backend.utils.common import bytes_to_hex


log = logging.getLogger('passport.models.persistent_track')

# Cсылка для автоматического восстановления по email
TRACK_TYPE_RESTORATION_AUTO_EMAIL_LINK = 1
# Ссылка, сгенерированная саппортами для восстановления
TRACK_TYPE_RESTORATION_SUPPORT_LINK = 2
# Одноразовая ссылка для входа в аккаунт (e.g. при регистрации без пароля)
TRACK_TYPE_AUTH_BY_KEY_LINK = 10
# Код подтверждения e-mail'а в валидаторе
TRACK_TYPE_EMAIL_CONFIRMATION_CODE = 42


PERSISTENT_TRACK_ID_BYTES_COUNT = 16


def generate_track_id():
    return bytes_to_hex(urandom(PERSISTENT_TRACK_ID_BYTES_COUNT))


def parse_track_content(data, *args):
    if 'content' in data:
        value = data['content']
        if value is None:
            # Если ЧЯ вернул значение None - не удалось распарсить JSON, запишем в лог
            log.error('Invalid JSON in persistent track')
        elif not isinstance(value, dict) or 'type' not in value:
            # Мы ожидаем увидеть объект с полем type
            log.error('Malformed persistent track content: %s', value)
        else:
            return True, value
    return False, None


class PersistentTrack(Model):
    """
    Трек, хранящийся в БД. Не путать с обычным треком в редисе!
    """
    parent = None

    track_id = Field('track_id')
    uid = Field('uid')
    created = Field(parse_unixtime_field('created'))
    expired = Field(parse_unixtime_field('expired'))

    content = Field(parse_track_content)

    @classmethod
    def create(cls, uid, track_type, expires_after=0, **options):
        """
        Создать новый трек для заданного UID с заданным типом
        @param uid: UID пользователя
        @param type_: тип трека, число
        @param expires_after: число секунд, через которое трек станет невалидным
        @options: дополнительные параметры, которые требуется сохранить в треке
        """
        track_id = generate_track_id()
        created = datetime.now()
        content = dict(options, type=track_type)
        return cls(
            track_id=track_id,
            uid=uid,
            created=created,
            expired=created + timedelta(seconds=expires_after),
            content=content,
        )

    @property
    def exists(self):
        return isinstance(self.content, dict)

    @property
    def track_key(self):
        """
        Ключ для идентификации трека (может использоваться в ссылках на восстановлении).
        """
        return '%s%x' % (self.track_id, int(self.uid))

    @property
    def is_expired(self):
        return datetime.now() >= self.expired

    @property
    def type(self):
        if not self.exists:
            return Undefined
        return self.content.get('type', Undefined)
