# -*- coding: utf-8 -*-

import logging

from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    Field,
)
from passport.backend.core.models.subscription import Subscription
from passport.backend.core.types.birthday import parse_birthday
from passport.backend.core.types.display_name import DisplayName
from passport.backend.core.types.gender import Gender
import pytz


MOSCOW_TIMEZONE = pytz.timezone('Europe/Moscow')

log = logging.getLogger('passport.models.person')


def _parse_timezone(data, *args):
    if 'person.timezone' not in data:
        return False, None

    timezone = data['person.timezone']

    if isinstance(timezone, pytz.tzinfo.BaseTzInfo):
        return True, timezone
    elif timezone:
        try:
            return True, pytz.timezone(timezone)
        except pytz.UnknownTimeZoneError:
            log.debug('User has bad timezone %s', timezone)
            return True, MOSCOW_TIMEZONE
    else:
        return True, MOSCOW_TIMEZONE


def _parse_display_name(data, *args):
    if 'display_name' not in data:
        return False, None

    display_data = data['display_name']
    name = display_data['name']
    public_name = display_data.get('public_name')
    provider = display_data.get('social', {}).get('provider')
    profile_id = display_data.get('social', {}).get('profile_id')

    display_name = DisplayName(name=name, public_name=public_name)
    if provider:
        display_name.set_social_provider(
            provider,
            int(profile_id) if profile_id else profile_id,
        )
    return True, display_name


def _parse_is_display_name_empty(data, *args):
    if 'display_name' not in data:
        return False, None

    display_name = data['display_name']
    is_empty = display_name.get('display_name_empty')
    if is_empty is None:
        return False, None
    return True, is_empty


def _parse_default_avatar(data, *args):
    if 'display_name' not in data:
        return False, None

    display_name = data['display_name']

    avatar_block = display_name.get('avatar')
    if not avatar_block:
        return False, None
    default_avatar_key = avatar_block.get('default')
    if not default_avatar_key:
        return False, None
    return True, default_avatar_key


def _parse_is_avatar_empty(data, *args):
    if 'display_name' not in data:
        return False, None

    display_name = data['display_name']
    is_empty = display_name.get('avatar', {}).get('empty')
    if is_empty is None:
        return False, None
    return True, is_empty


class Person(Model):
    """
    Персональные данные пользователя.
    Соответствие между Account и Person - 1:1
    """

    CONTACT_PHONE_NUMBER_SID = 89

    parent = None

    firstname = Field('person.firstname')
    lastname = Field('person.lastname')

    firstname_global = Field('person.firstname_global')
    lastname_global = Field('person.lastname_global')

    # TODO: Перенести в Account
    display_name = Field(_parse_display_name)
    is_display_name_empty = Field(_parse_is_display_name_empty)  # Является ли display_name вычисленным по другим полям
    default_avatar = Field(_parse_default_avatar)  # Ключ аватарки по умолчанию
    is_avatar_empty = Field(_parse_is_avatar_empty)  # Является ли дефолтная аватарка заглушкой

    gender = Field('userinfo.sex.uid')
    birthday = Field(parse_birthday('person.birthday'))  # Birthday class represents this fields

    # Страна. Двухбуквенный код, для страны, указанной пользователем.
    country = Field('person.country')

    city = Field('person.city')
    email = Field()  # просто строка, на самом деле бесполезная
    timezone = Field(_parse_timezone)
    language = Field('person.language')  # Cейчас двухбуквенный, но может измениться. String(exact_length=2)

    # PASSP-23241 атрибут, меняющий вычисление public_name на стороне ЧЯ
    dont_use_displayname_as_public_name = BooleanField('person.dont_use_displayname_as_public_name')
    # использовать полную фамилию для public name
    show_fio_in_public_name = BooleanField('person.show_fio_in_public_name')

    def is_male(self):
        return self.gender == Gender.Male

    def is_female(self):
        return self.gender == Gender.Female

    @property
    def contact_phone_number(self):
        sid = Person.CONTACT_PHONE_NUMBER_SID
        if sid not in self.parent.subscriptions:
            return None

        return self.parent.subscriptions[sid].login

    @contact_phone_number.setter
    def contact_phone_number(self, value):
        sid = Person.CONTACT_PHONE_NUMBER_SID
        if value and value.startswith('+'):
            value = value[1:]
        if sid not in self.parent.subscriptions:
            self.parent.subscriptions[sid] = Subscription(self.parent).parse({'sid': sid})

        self.parent.subscriptions[sid].login = value
