# -*- coding: utf-8 -*-
"""
Типичные состояния телефонов и телефонных операций.
"""

from datetime import (
    datetime,
    timedelta,
)
import json

import mock
from nose.tools import (
    eq_,
    ok_,
)
from passport.backend.core.builders.blackbox.faker.blackbox import (
    blackbox_sessionid_multi_response,
    blackbox_userinfo_response,
)
from passport.backend.core.builders.blackbox.parsers import (
    parse_blackbox_sessionid_response,
    parse_blackbox_userinfo_response,
    PHONE_OP_TYPE_NAME,
)
from passport.backend.core.conf import settings
from passport.backend.core.db.schemas import phone_operations_table
from passport.backend.core.dbmanager.manager import get_dbm
from passport.backend.core.dbmanager.sharder import get_db_name
from passport.backend.core.eav_type_mapping import (
    EXTENDED_ATTRIBUTES_PHONE_NAME_TO_TYPE_MAPPING,
    EXTENDED_ATTRIBUTES_PHONE_TYPE,
)
from passport.backend.core.models.account import Account
from passport.backend.core.models.phones.phones import SECURITY_IDENTITY
from passport.backend.core.test.test_utils import single_entrant_patch
from passport.backend.core.test.time_utils.time_utils import TimeNow
from passport.backend.core.types.bit_vector.bit_vector import (
    PhoneBindingsFlags,
    PhoneOperationFlags,
)
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.undefined import Undefined
from passport.backend.utils.time import zero_datetime
from six import iteritems


ALIAS_SID = 65
TEST_DATE = datetime(2014, 2, 26, 0, 0, 0)
CODE_VALUE = u'1234'


def build_account(db_faker=None, blackbox_faker=None, **kwargs):
    user_info = blackbox_userinfo_response(**kwargs)
    if blackbox_faker is not None:
        blackbox_faker.set_response_value(u'userinfo', user_info)
    user_info = parse_blackbox_userinfo_response(json.loads(user_info))
    account = Account().parse(user_info)

    if db_faker is not None:
        db_faker._serialize_to_eav(account)
    return account


def build_account_from_session(db_faker=None, blackbox_faker=None, **kwargs):
    session = blackbox_sessionid_multi_response(**kwargs)
    if blackbox_faker is not None:
        blackbox_faker.set_response_value(u'sessionid', session)
    session = parse_blackbox_sessionid_response(json.loads(session))
    account = Account().parse(session)

    if db_faker is not None:
        db_faker._serialize_to_eav(account)
    return account


def build_current_phone_binding(phone_id, phone_number, phone_bound, flags=None):
    ok_(phone_bound)
    if flags is None:
        flags = PhoneBindingsFlags()
    return {
        'type': 'current',
        'number': phone_number,
        'phone_id': phone_id,
        'bound': phone_bound,
        'flags': int(flags),
    }


def build_unbound_phone_binding(phone_id, phone_number):
    return {
        'type': 'unbound',
        'number': phone_number,
        'phone_id': phone_id,
        'bound': None,
        'flags': 0,
    }


def build_phone_being_bound(phone_id, phone_number, operation_id,
                            phone_created=TEST_DATE, operation_started=None,
                            operation_finished=None, code_value=CODE_VALUE,
                            code_confirmed=None, code_checks_count=None,
                            code_send_count=1, code_last_sent=TEST_DATE,
                            flags=None):
    """
    Параметры простого телефона, который привязывается.
    """
    ok_(phone_created is not None)
    if operation_started is None:
        operation_started = datetime.now()
    if operation_finished is None:
        operation_finished = evaluate_finished_time(operation_started)
    return dict(
        phones=[dict(
            id=phone_id,
            number=phone_number,
            created=phone_created,
            bound=None,
            confirmed=code_confirmed,
            secured=None,
        )],
        phone_bindings=[
            build_unbound_phone_binding(phone_id, phone_number),
        ],
        phone_operations=[dict(
            id=operation_id,
            phone_id=phone_id,
            phone_number=phone_number,
            type=u'bind',
            started=operation_started,
            finished=operation_finished,
            code_value=code_value,
            code_confirmed=code_confirmed,
            code_checks_count=code_checks_count,
            code_send_count=code_send_count,
            code_last_sent=code_last_sent,
            flags=flags,
        )],
    )


def build_secure_phone_being_bound(phone_id, phone_number, operation_id,
                                   phone_created=TEST_DATE,
                                   operation_started=None,
                                   operation_finished=None,
                                   password_verified=None,
                                   flags=None,
                                   code_value=CODE_VALUE,
                                   code_last_sent=TEST_DATE,
                                   code_confirmed=None,
                                   code_send_count=1,
                                   code_checks_count=0):
    """
    Параметры защищённого телефона, который привязывается.
    """
    ok_(phone_created is not None)
    if operation_started is None:
        operation_started = datetime.now()
    if operation_finished is None:
        operation_finished = evaluate_finished_time(operation_started)
    return dict(
        phones=[dict(
            id=phone_id,
            number=phone_number,
            created=phone_created,
            bound=None,
            confirmed=None,
            secured=None,
        )],
        phone_bindings=[
            build_unbound_phone_binding(phone_id, phone_number),
        ],
        phone_operations=[dict(
            id=operation_id,
            phone_id=phone_id,
            security_identity=SECURITY_IDENTITY,
            type=u'bind',
            started=operation_started,
            finished=operation_finished,
            password_verified=password_verified,
            flags=flags,
            code_value=code_value,
            code_last_sent=code_last_sent,
            code_confirmed=code_confirmed,
            code_send_count=code_send_count,
            code_checks_count=code_checks_count,
        )],
    )


def build_phone_unbound(phone_id, phone_number, phone_created=TEST_DATE,
                        phone_confirmed=TEST_DATE):
    """
    Параметры телефона, который отвязан.
    """
    ok_(phone_created is not None)
    ok_(phone_confirmed is not None)
    ok_(phone_confirmed >= phone_created)
    return dict(
        phones=[dict(
            id=phone_id,
            number=phone_number,
            created=phone_created,
            bound=None,
            confirmed=phone_confirmed,
            secured=None,
        )],
        phone_operations=[],
        phone_bindings=[
            build_unbound_phone_binding(phone_id, phone_number),
        ],
    )


def build_phone_bound(phone_id, phone_number, phone_created=Undefined,
                      phone_bound=Undefined, phone_confirmed=Undefined,
                      phone_admitted=None, is_default=False, is_bank=False, binding_flags=None):
    """
    Параметры простого телефона.
    """
    ok_(phone_created is not None)
    ok_(phone_bound is not None)
    ok_(phone_confirmed is not None)

    # Выставляем дефолты, так чтобы выполнялся инвариант
    # phone_confirmed >= phone_bound >= phone_created
    if all(d is Undefined for d in [phone_created, phone_bound, phone_confirmed]):
        phone_created = phone_bound = phone_confirmed = TEST_DATE
    earliest = min([d for d in [phone_created, phone_bound, phone_confirmed] if d is not Undefined])
    if phone_created is Undefined:
        phone_created = earliest
    if phone_bound is Undefined:
        phone_bound = phone_created
    if phone_confirmed is Undefined:
        phone_confirmed = phone_bound

    ok_(phone_confirmed >= phone_bound >= phone_created)

    if binding_flags is None:
        binding_flags = PhoneBindingsFlags()

    args = dict(
        phones=[dict(
            id=phone_id,
            number=phone_number,
            created=phone_created,
            bound=phone_bound,
            confirmed=phone_confirmed,
            admitted=phone_admitted,
            is_bank=is_bank,
            secured=None,
        )],
        phone_bindings=[
            build_current_phone_binding(phone_id, phone_number, phone_bound, binding_flags),
        ],
        phone_operations=[],
    )
    if is_default:
        args[u'attributes'] = {u'phones.default': phone_id}
    return args


def build_phone_secured(phone_id, phone_number, phone_created=TEST_DATE,
                        phone_bound=TEST_DATE, phone_confirmed=TEST_DATE,
                        phone_secured=TEST_DATE, phone_admitted=None,
                        is_default=False, is_bank=False, is_alias=False,
                        is_enabled_search_for_alias=True, binding_flags=None):
    """
    Параметры защищённого телефона.
    """
    ok_(phone_created is not None)
    ok_(phone_bound is not None)
    ok_(phone_secured is not None)
    ok_(phone_confirmed is not None)
    ok_(phone_confirmed >= phone_secured >= phone_bound >= phone_created)

    if binding_flags is None:
        binding_flags = PhoneBindingsFlags()

    args = dict(
        phones=[dict(
            id=phone_id,
            number=phone_number,
            created=phone_created,
            bound=phone_bound,
            confirmed=phone_confirmed,
            secured=phone_secured,
            admitted=phone_admitted,
            is_bank=is_bank,
        )],
        phone_operations=[],
        phone_bindings=[
            build_current_phone_binding(
                phone_id,
                phone_number,
                phone_bound,
                binding_flags,
            ),
        ],
        attributes={
            u'phones.secure': phone_id,
        },
    )
    if is_default:
        args[u'attributes'][u'phones.default'] = phone_id
    if is_alias:
        args[u'aliases'] = {
            u'phonenumber': PhoneNumber.parse(phone_number).digital,
        }
        if is_enabled_search_for_alias:
            args[u'attributes'][u'account.enable_search_by_phone_alias'] = u'1'
    return args


def build_remove_operation(operation_id, phone_id, started=None, finished=None,
                           password_verified=None, flags=None,
                           code_value=CODE_VALUE, code_last_sent=TEST_DATE,
                           code_checks_count=0, code_send_count=1,
                           code_confirmed=None):
    """
    Параметры операции удаления над защищённым телефоном.
    """
    if started is None:
        started = datetime.now()
    if finished is None:
        finished = evaluate_finished_time(started)
    return dict(
        phone_operations=[dict(
            id=operation_id,
            phone_id=phone_id,
            type=u'remove',
            started=started,
            finished=finished,
            security_identity=SECURITY_IDENTITY,
            password_verified=password_verified,
            flags=flags,
            code_value=code_value,
            code_confirmed=code_confirmed,
            code_checks_count=code_checks_count,
            code_send_count=code_send_count,
            code_last_sent=code_last_sent,
        )],
    )


def build_securify_operation(operation_id, phone_id, started=None,
                             finished=None, password_verified=None, flags=None,
                             code_value=CODE_VALUE, code_confirmed=None,
                             code_checks_count=0, code_send_count=1,
                             code_last_sent=TEST_DATE):
    """
    Параметры операции защиты над простым телефоном.
    """
    if started is None:
        started = datetime.now()
    if finished is None:
        finished = evaluate_finished_time(started)
    return dict(
        phone_operations=[dict(
            id=operation_id,
            phone_id=phone_id,
            security_identity=SECURITY_IDENTITY,
            type=u'securify',
            started=started,
            finished=finished,
            flags=flags,
            code_value=code_value,
            code_confirmed=code_confirmed,
            code_checks_count=code_checks_count,
            code_send_count=code_send_count,
            code_last_sent=code_last_sent,
            password_verified=password_verified,
        )],
    )


def build_simple_replaces_secure_operations(secure_operation_id,
                                            secure_phone_id,
                                            simple_operation_id,
                                            simple_phone_id,
                                            simple_phone_number,
                                            started=None,
                                            finished=None,
                                            password_verified=None,
                                            flags=None,
                                            secure_code_confirmed=None,
                                            secure_code_value=CODE_VALUE,
                                            secure_code_checks_count=0,
                                            secure_code_send_count=1,
                                            secure_code_last_sent=TEST_DATE,
                                            simple_code_confirmed=None,
                                            simple_code_value=CODE_VALUE,
                                            simple_code_checks_count=0,
                                            simple_code_send_count=1,
                                            simple_code_last_sent=TEST_DATE):
    """
    Параметры операции замены защищённого телефона на простой привязанный
    телефон.
    """
    if started is None:
        started = datetime.now()
    if finished is None:
        finished = evaluate_finished_time(started)
    return dict(
        phone_operations=[
            dict(
                type=u'replace',
                id=secure_operation_id,
                phone_id=secure_phone_id,
                started=started,
                finished=finished,
                security_identity=SECURITY_IDENTITY,
                password_verified=password_verified,
                code_value=secure_code_value,
                code_confirmed=secure_code_confirmed,
                code_checks_count=secure_code_checks_count,
                code_send_count=secure_code_send_count,
                code_last_sent=secure_code_last_sent,
                flags=flags,
                phone_id2=simple_phone_id,
            ),
            dict(
                type=u'mark',
                id=simple_operation_id,
                phone_id=simple_phone_id,
                started=started,
                finished=finished,
                password_verified=password_verified,
                phone_number=simple_phone_number,
                code_value=simple_code_value,
                code_confirmed=simple_code_confirmed,
                code_checks_count=simple_code_checks_count,
                code_send_count=simple_code_send_count,
                code_last_sent=simple_code_last_sent,
                flags=flags,
                phone_id2=secure_phone_id,
            ),
        ],
    )


def build_phone_being_bound_replaces_secure_operations(secure_operation_id,
                                                       secure_phone_id,
                                                       being_bound_operation_id,
                                                       being_bound_phone_id,
                                                       being_bound_phone_number,
                                                       started=None,
                                                       finished=None,
                                                       password_verified=None,
                                                       flags=None,
                                                       secure_code_value=CODE_VALUE,
                                                       secure_code_confirmed=None,
                                                       secure_code_checks_count=0,
                                                       secure_code_send_count=1,
                                                       secure_code_last_sent=TEST_DATE,
                                                       being_bound_code_value=CODE_VALUE,
                                                       being_bound_code_confirmed=None,
                                                       being_bound_code_checks_count=0,
                                                       being_bound_code_send_count=1,
                                                       being_bound_code_last_sent=TEST_DATE):
    """
    Параметры операции замены защищённого телефона на непривязанный телефон.
    """
    if started is None:
        started = datetime.now()
    if finished is None:
        finished = evaluate_finished_time(started)
    return dict(
        phone_operations=[
            dict(
                type=u'replace',
                id=secure_operation_id,
                phone_id=secure_phone_id,
                started=started,
                finished=finished,
                security_identity=SECURITY_IDENTITY,
                code_value=secure_code_value,
                code_confirmed=secure_code_confirmed,
                code_checks_count=secure_code_checks_count,
                code_send_count=secure_code_send_count,
                code_last_sent=secure_code_last_sent,
                password_verified=password_verified,
                flags=flags,
                phone_id2=being_bound_phone_id,
            ),
            dict(
                type=u'bind',
                id=being_bound_operation_id,
                phone_id=being_bound_phone_id,
                started=started,
                finished=finished,
                phone_number=being_bound_phone_number,
                code_value=being_bound_code_value,
                code_confirmed=being_bound_code_confirmed,
                code_checks_count=being_bound_code_checks_count,
                code_send_count=being_bound_code_send_count,
                code_last_sent=being_bound_code_last_sent,
                password_verified=password_verified,
                flags=flags,
                phone_id2=secure_phone_id,
            ),
        ],
    )


def build_mark_operation(operation_id, phone_number, phone_id, started=None,
                         finished=None, password_verified=None, flags=None,
                         code_value=CODE_VALUE, code_last_sent=TEST_DATE,
                         code_checks_count=0, code_send_count=1,
                         code_confirmed=None):
    """
    Параметры операции-флажка.
    """
    if started is None:
        started = datetime.now()
    if finished is None:
        finished = started + timedelta(seconds=settings.YASMS_MARK_OPERATION_TTL)
    return dict(
        phone_operations=[dict(
            id=operation_id,
            phone_id=phone_id,
            type=u'mark',
            started=started,
            finished=finished,
            security_identity=int(phone_number),
            password_verified=password_verified,
            flags=flags,
            code_value=code_value,
            code_confirmed=code_confirmed,
            code_checks_count=code_checks_count,
            code_send_count=code_send_count,
            code_last_sent=code_last_sent,
        )],
    )


def build_aliasify_secure_operation(operation_id, phone_id, started=None,
                                    finished=None, flags=None, code_value=CODE_VALUE,
                                    code_last_sent=TEST_DATE,
                                    code_checks_count=0, code_send_count=1,
                                    code_confirmed=None):
    if started is None:
        started = datetime.now()
    if finished is None:
        finished = evaluate_finished_time(started)

    if flags is None:
        flags = PhoneOperationFlags()
        flags.aliasify = True

    return dict(
        phone_operations=[dict(
            id=operation_id,
            phone_id=phone_id,
            type=u'aliasify',
            started=started,
            finished=finished,
            security_identity=SECURITY_IDENTITY,
            flags=flags,
            code_value=code_value,
            code_confirmed=code_confirmed,
            code_checks_count=code_checks_count,
            code_send_count=code_send_count,
            code_last_sent=code_last_sent,
        )],
    )


def build_dealiasify_secure_operation(operation_id, phone_id, started=None,
                                      finished=None, flags=None, code_value=CODE_VALUE,
                                      password_verified=None):
    if started is None:
        started = datetime.now()
    if finished is None:
        finished = evaluate_finished_time(started)
    return dict(
        phone_operations=[dict(
            id=operation_id,
            phone_id=phone_id,
            type=u'dealiasify',
            started=started,
            finished=finished,
            security_identity=SECURITY_IDENTITY,
            flags=flags,
            password_verified=password_verified,
            code_value=code_value,
            code_confirmed=None,
            code_checks_count=0,
            code_send_count=0,
            code_last_sent=None,
        )],
    )


def evaluate_finished_time(started):
    return started + timedelta(seconds=settings.PHONE_QUARANTINE_SECONDS)


def assert_no_phone_in_db(db_faker, uid, phone_id, phone_number, shard_db=u'passportdbshard1'):
    """
    Утверждает, что в БД нет сведений о номере у пользователя.

    Замечание: подпрограмма не увтерждает, что номер вообще был когда-либо в БД.
    """
    if isinstance(phone_number, PhoneNumber):
        phone_number = phone_number.e164

    for phone_attr in EXTENDED_ATTRIBUTES_PHONE_NAME_TO_TYPE_MAPPING:
        db_faker.check_db_ext_attr_missing(
            uid=uid,
            entity_type=EXTENDED_ATTRIBUTES_PHONE_TYPE,
            entity_id=phone_id,
            field_name=phone_attr,
        )
    db_faker.check_missing(
        u'phone_operations',
        db=shard_db,
        uid=uid,
        phone_id=phone_id,
    )
    db_faker.check_missing(
        'phone_bindings',
        db=shard_db,
        uid=uid,
        number=phone_number,
    )


def assert_no_default_phone_chosen(db_faker, uid, shard_db=u'passportdbshard1'):
    """
    У пользователя нет телефона, который бы он назначил дефолтным.
    """
    db_faker.check_missing(
        u'attributes',
        attr=u'phones.default',
        uid=uid,
        db=shard_db,
    )


def assert_no_secure_phone(db_faker, uid, shard_db=u'passportdbshard1'):
    """
    У пользователя нет защищённого телефона.
    """
    for attr in [u'phones.secure']:
        db_faker.check_missing(
            u'attributes',
            attr=attr,
            uid=uid,
            db=shard_db,
        )


def assert_phone_has_been_bound(db_faker, uid, phone_number, times=1, shard_db=u'passportdbshard1'):
    """
    Телефон привязывался times раз.
    """
    eq_(
        len(db_faker.select(
            u'phone_bindings_history',
            uid=uid,
            number=int(phone_number.lstrip('+')),
            db=shard_db,
        )),
        times,
    )


def assert_account_has_phonenumber_alias(db_faker, uid, alias, shard_db='passportdbshard1',
                                         enable_search=True):
    db_faker.check('aliases', 'phonenumber', alias, uid=uid, db='passportdbcentral')
    if enable_search:
        db_faker.check_db_attr(
            uid=uid,
            attr_name='account.enable_search_by_phone_alias',
            value='1',
            db=shard_db,
        )
    else:
        db_faker.check_missing(
            'attributes',
            attr='account.enable_search_by_phone_alias',
            uid=uid,
            db=shard_db,
        )


def assert_phonenumber_alias_missing(db_faker, uid, shard_db='passportdbshard1'):
    db_faker.check_missing('aliases', 'phonenumber', uid=uid, db='passportdbcentral')
    db_faker.check_missing(
        'attributes',
        attr='account.enable_search_by_phone_alias',
        uid=uid,
        db=shard_db,
    )


def assert_phonenumber_alias_removed(db_faker, uid, alias, shard_db='passportdbshard1'):
    """
    Утверждает, что телефонный алиас пользователя удалился.
    """
    db_faker.check_missing(
        u'aliases',
        u'phonenumber',
        uid=uid,
        value=alias,
        db='passportdbcentral',
    )
    db_faker.check(
        u'removed_aliases',
        u'phonenumber',
        alias,
        uid=uid,
        db='passportdbcentral',
    )


@single_entrant_patch
class PhoneIdGeneratorFaker(object):
    # TODO: заменить на общий db.faker.IdGeneratorFaker? Но тогда id станут сквозными
    #  (одна sequence на уиды, емейлы, телефоны и т.п.)
    def __init__(self):
        self._mock = mock.Mock(name=u'phone_id_generator')
        self._patch = mock.patch(
            u'passport.backend.core.serializers.run_incr_phone_id.run_incr_phone_id.run_incr_phone_id',
            self._mock,
        )

    def start(self):
        self._patch.start()

    def stop(self):
        self._patch.stop()

    def set_list(self, phone_ids):
        self._mock.side_effect = phone_ids

    @property
    def call_count(self):
        return self._mock.call_count


def predict_next_operation_id(uid):
    """
    Предсказывает идентификатор следующей созданной телефонной операции.
    """
    dbm = get_dbm(get_db_name(phone_operations_table.name, uid))
    result = dbm.get_engine().execute(phone_operations_table.insert())
    last_operation_id = result.inserted_primary_key[0]
    return last_operation_id + 1


class _PhoneStateChecker(object):
    """
    Фабрика строит Проверяющих телефон и физическую операцию на нём.
    """
    required_phone_attributes = set()
    optional_phone_attributes = set()
    unset_phone_attributes = set()

    is_phone_secure = False

    required_operation_attributes = set()
    optional_operation_attributes = set()
    unset_operation_attributes = set()

    operation_type = None
    is_operation_secure = False

    def __init__(self, **kwargs):
        """
        Входные параметры
            required_phone_attributes
                Атрибуты телефона, наличие которых проверяющий должен проверить.

            optional_phone_attributes
                Атрибуты телефона, наличие которых проверяющий может проверить,
                если его об этом попросить.

            unset_phone_attributes
                Атрибуты телефона, отсутствие которых проверяющий должен
                проверить.

            is_phone_secure
                Проверяющий должен проверить, что данный телефон защищён.

            required_operation_attributes
                Атрибуты операции на телефоне, наличие которых проверяющий
                должен проверить.

            optional_operation_attributes
                Атрибуты операции на телефоне, наличие которых проверяющий
                может проверить, если его об этом попросить.

            unset_operation_attributes
                Атрибуты операции на телефоне, отсутствие которых проверяющий
                должен проверить.

            operation_type
                Проверящий проверит, что тип операции на телефоне равен
                данному.
                Если тип операции не указан, то проверяющий проверит, что на
                телефоне нет операции.

            is_operation_secure
                Проверящий проверит, что операции на телефоне защищена.
        """
        for key in kwargs:
            setattr(self, key, kwargs[key])

    def __call__(self, account, phone_attributes={}, operation_attributes={}):
        """
        Проверяет атрибуты телефона и его операции.

        Проверяется наличие каждого обязательного атрибута, а если указано
        значение, то равенство атрибута указанному значению.

        Для необязательных атрибутов проверяющий проверит равенство указанному
        значению, а когда указано значение None, проверит отсутствие атрибута.
        """
        phone_id = phone_attributes[u'id']
        self._check(
            phone_attributes,
            operation_attributes,
            self._get_actual_phone_attrs_from_account(account, phone_id),
            self._get_actual_operation_attrs_from_account(account, phone_id),
            self._get_account_attrs_from_account(account),
        )

        phone = account.phones.by_id(phone_id)
        ok_(phone.binding)

        if self._phone_should_be_bound(phone_attributes):
            expected = phone_attributes.get(u'bound')
            if expected is not None:
                eq_(phone.binding.time, expected)
            else:
                ok_(phone.binding.time)

        if self._phone_should_be_unbound():
            ok_(not phone.binding.time)

    def check_db(self, db_faker, uid, phone_attributes={},
                 operation_attributes={}, binding_flags=None, shard_db=u'passportdbshard1'):
        phone_id = phone_attributes[u'id']
        actual_phone_attrs = self._get_actual_phone_attrs_from_db(
            db_faker,
            shard_db,
            uid,
            phone_id,
        )
        self._check(
            phone_attributes,
            operation_attributes,
            actual_phone_attrs,
            self._get_actual_operation_attrs_from_db(db_faker, shard_db, uid, phone_id),
            self._get_account_attrs_from_db(db_faker, shard_db, uid),
        )

        phone_number = actual_phone_attrs[u'number']
        phone_binding = db_faker.get(
            u'phone_bindings',
            uid=uid,
            number=int(phone_number),
            db=shard_db,
        )
        ok_(phone_binding)

        if self._phone_should_be_bound(phone_attributes):
            bind_time = actual_phone_attrs[u'bound']
            eq_(phone_binding.bound, bind_time)

            if binding_flags is not None:
                eq_(phone_binding.flags, int(binding_flags))

            phone_bindings_history = db_faker.select(
                u'phone_bindings_history',
                uid=uid,
                number=int(phone_number),
                db=shard_db,
            )
            ok_(bind_time in {b.bound for b in phone_bindings_history})

        if self._phone_should_be_unbound():
            eq_(phone_binding.bound, zero_datetime)

    def _phone_should_be_bound(self, expected_phone_attrs):
        bind_time = expected_phone_attrs.get('bound')
        return (
            u'bound' in self.required_phone_attributes or
            (u'bound' in self.optional_phone_attributes and bind_time is not None)
        )

    def _phone_should_be_unbound(self):
        return 'bound' in self.unset_phone_attributes

    def _check(self, expected_phone_attrs, expected_operation_attrs,
               actual_phone_attrs, actual_operation_attrs, account_attrs):
        expected_phone_attrs = self._reset_microseconds(
            expected_phone_attrs,
            {u'created', u'bound', u'confirmed', u'admitted', u'secured'},
        )
        self._check_attrs(
            actual_phone_attrs,
            expected_phone_attrs,
            self.required_phone_attributes,
            self.optional_phone_attributes,
            self.unset_phone_attributes,
        )

        if self.is_phone_secure:
            eq_(
                account_attrs[u'phones.secure'],
                actual_phone_attrs[u'id'],
                self._format_msg(
                    u'phones.secure',
                    actual_phone_attrs[u'id'],
                    account_attrs[u'phones.secure'],
                ),
            )

        if self.operation_type:
            eq_(actual_operation_attrs[u'type'], self.operation_type)
            if self.is_operation_secure:
                eq_(
                    actual_operation_attrs[u'security_identity'],
                    SECURITY_IDENTITY,
                )
            else:
                eq_(
                    actual_operation_attrs[u'security_identity'],
                    int(actual_phone_attrs[u'number']),
                )
            expected_operation_attrs = self._reset_microseconds(
                expected_operation_attrs,
                {u'started', u'finished', u'code_confirmed',
                 u'password_verified', u'code_last_sent'},
            )
            self._check_attrs(
                actual_operation_attrs,
                expected_operation_attrs,
                self.required_operation_attributes,
                self.optional_operation_attributes,
                self.unset_operation_attributes,
            )
        else:
            ok_(not actual_operation_attrs)

    @classmethod
    def _check_attrs(cls, actual, expected, required, optional, unset):
        for attr_name in required:
            expected_value = expected.get(attr_name)
            if expected_value is not None:
                actual_value = actual.get(attr_name)
                eq_(
                    actual_value,
                    expected_value,
                    cls._format_msg(attr_name, expected_value, actual_value),
                )
            else:
                ok_(actual.get(attr_name), (attr_name, actual.get(attr_name)))

        for attr_name in optional:
            expected_value = expected.get(attr_name, Undefined)
            if expected_value is None:
                ok_(not actual.get(attr_name), (attr_name, actual.get(attr_name)))
            elif expected_value is not Undefined:
                actual_value = actual.get(attr_name)
                eq_(
                    actual_value,
                    expected_value,
                    cls._format_msg(attr_name, expected_value, actual_value),
                )

        for attr_name in unset:
            ok_(not actual.get(attr_name), (attr_name, actual.get(attr_name)))

    @classmethod
    def _format_msg(cls, attr_name, expected_value, actual_value):
        return u'%s: got %r, expected %r' % (attr_name, actual_value, expected_value)

    def _get_actual_phone_attrs_from_account(cls, account, phone_id):
        ok_(account.phones.has_id(phone_id))
        phone = account.phones.by_id(phone_id)
        attrs = {}
        for name in {u'id', u'number', u'created', u'bound', u'confirmed',
                     u'secured', u'admitted'}:
            attrs[name] = getattr(phone, name)
        attrs[u'number'] = attrs[u'number'].e164
        return attrs

    def _get_actual_phone_attrs_from_db(cls, db_faker, shard_db, uid, phone_id):
        attrs = {u'id': phone_id}
        for name in {u'number', u'created', u'bound', u'confirmed', u'secured',
                     u'admitted'}:
            value = db_faker.get(
                u'extended_attributes',
                name,
                uid=uid,
                entity_type=EXTENDED_ATTRIBUTES_PHONE_TYPE,
                entity_id=phone_id,
                db=shard_db,
            )
            attrs[name] = value

        for name in {u'created', u'bound', u'confirmed', u'secured',
                     u'admitted'}:
            time = attrs[name]
            if time:
                attrs[name] = datetime.fromtimestamp(float(time))

        number = attrs[u'number']
        if number:
            attrs[u'number'] = u'+' + number
        return attrs

    def _get_actual_operation_attrs_from_account(cls, account, phone_id):
        phone = account.phones.by_id(phone_id)
        operation = phone.operation
        if not operation:
            return
        attrs = {}
        for name in {u'security_identity', u'type', u'started', u'code_value',
                     u'finished', u'flags', u'id', u'code_send_count',
                     u'code_last_sent', u'code_checks_count', u'code_confirmed',
                     u'password_verified', u'phone_id2'}:
            attrs[name] = getattr(operation, name)
        return attrs

    def _get_actual_operation_attrs_from_db(cls, db_faker, shard_db, uid, phone_id):
        relation = db_faker.get(
            u'phone_operations',
            phone_id=phone_id,
            uid=uid,
            db=shard_db,
        )
        if relation is None:
            return
        columns = [c.name for c in phone_operations_table.c]
        operation = dict(zip(columns, relation))

        operation[u'flags'] = PhoneOperationFlags(operation[u'flags'])
        operation[u'type'] = PHONE_OP_TYPE_NAME[operation[u'type']]
        for name in {u'started', u'finished', u'code_last_sent',
                     u'code_confirmed', u'password_verified'}:
            time = operation[name]
            if zero_datetime == time:
                operation[name] = None
        return operation

    def _get_account_attrs_from_account(cls, account):
        attrs = {}
        if account.phones:
            attrs[u'phones.secure'] = account.phones.secure_id
        return attrs

    def _get_account_attrs_from_db(cls, db_faker, shard_db, uid):
        attrs = {}
        for name in {u'phones.secure'}:
            value = db_faker.get(
                u'attributes',
                name,
                uid=uid,
                db=shard_db,
            )
            attrs[name] = value

        secure_id = attrs[u'phones.secure']
        if secure_id is not None:
            attrs[u'phones.secure'] = int(secure_id)

        return attrs

    def _reset_microseconds(cls, attrs, time_names):
        retval = {}
        for name, value in iteritems(attrs):
            if name in time_names:
                if value is not None:
                    value = value.replace(microsecond=0)
            retval[name] = value
        return retval


# Привязывается простой телефон
assert_simple_phone_being_bound = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number'},
    optional_phone_attributes={'confirmed', 'admitted'},
    unset_phone_attributes={'bound', 'secured'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'code_value',
        'finished',
        'flags',
    },
    optional_operation_attributes={
        'id',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
    },
    unset_operation_attributes={
        'password_verified',
        'phone_id2',
    },
    operation_type='bind',
    is_operation_secure=False,
)


# Привязывается защищённый телефон
assert_secure_phone_being_bound = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number'},
    optional_phone_attributes={'confirmed', 'admitted'},
    unset_phone_attributes={'bound', 'secured'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'code_value',
        'flags',
    },
    optional_operation_attributes={
        'id',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
        'password_verified',
    },
    unset_operation_attributes={
        'phone_id2',
    },
    operation_type='bind',
    is_operation_secure=True,
)


# Привязан простой телефон
assert_simple_phone_bound = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number', 'bound', 'confirmed'},
    optional_phone_attributes={'admitted'},
    unset_phone_attributes={'secured'},
)


# Привязан защищённый телефон
assert_secure_phone_bound = _PhoneStateChecker(
    required_phone_attributes={
        'id',
        'created',
        'number',
        'bound',
        'confirmed',
        'secured',
    },
    optional_phone_attributes={'admitted'},
    is_phone_secure=True,
)


# Простой телефон защищается
assert_simple_phone_being_securified = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number', 'bound', 'confirmed'},
    optional_phone_attributes={'admitted'},
    unset_phone_attributes={'secured'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'code_value',
        'flags',
    },
    optional_operation_attributes={
        'id',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
        'password_verified',
    },
    unset_operation_attributes={
        'phone_id2',
    },
    operation_type='securify',
    is_operation_secure=True,
)


# Защищённый телефон удаляется
assert_secure_phone_being_removed = _PhoneStateChecker(
    required_phone_attributes={
        'id',
        'created',
        'number',
        'bound',
        'confirmed',
        'secured',
    },
    optional_phone_attributes={'admitted'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'flags',
    },
    optional_operation_attributes={
        'id',
        'code_value',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
        'password_verified',
    },
    unset_operation_attributes={
        'phone_id2',
    },
    operation_type='remove',
    is_operation_secure=True,
    is_phone_secure=True,
)


# Простой телефон привязывается чтобы заменить защищённый
assert_simple_phone_being_bound_replace_secure = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number'},
    optional_phone_attributes={'confirmed', 'admitted'},
    unset_phone_attributes={'bound', 'secured'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'code_value',
        'flags',
        'phone_id2',
    },
    optional_operation_attributes={
        'id',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
        'password_verified',
    },
    operation_type='bind',
    is_operation_secure=False,
)


assert_simple_phone_replace_secure = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number', 'bound'},
    optional_phone_attributes={'confirmed', 'admitted'},
    unset_phone_attributes={'secured'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'code_value',
        'flags',
        'phone_id2',
    },
    optional_operation_attributes={
        'id',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
        'password_verified',
    },
    operation_type='mark',
    is_operation_secure=False,
)


assert_secure_phone_being_replaced = _PhoneStateChecker(
    required_phone_attributes={
        'id',
        'created',
        'number',
        'bound',
        'confirmed',
        'secured',
    },
    optional_phone_attributes={'admitted'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'flags',
        'phone_id2',
    },
    optional_operation_attributes={
        'id',
        'code_value',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
        'password_verified',
    },
    operation_type='replace',
    is_operation_secure=True,
    is_phone_secure=True,
)


# Телефон отвязан
assert_phone_unbound = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number', 'confirmed'},
    optional_phone_attributes={'admitted'},
    unset_phone_attributes={'secured', 'bound'},
)


# На телефоне находится операция-флажок
assert_phone_marked = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number'},
    optional_phone_attributes={'confirmed', 'admitted', u'bound', 'secured'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'flags',
    },
    optional_operation_attributes={
        'id',
        'code_value',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
        'password_verified',
    },
    unset_operation_attributes={
        'phone_id2',
    },
    operation_type='mark',
    is_operation_secure=False,
)


assert_secure_phone_being_aliasified = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number', 'confirmed', 'bound', 'secured'},
    optional_phone_attributes={'admitted'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'flags',
        'code_value',
    },
    optional_operation_attributes={
        'id',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
    },
    unset_operation_attributes={
        'phone_id2',
        'password_verified',
    },
    operation_type='aliasify',
    is_operation_secure=True,
)


assert_secure_phone_being_dealiasified = _PhoneStateChecker(
    required_phone_attributes={'id', 'created', 'number', 'confirmed', 'bound', 'secured'},
    optional_phone_attributes={'admitted'},
    required_operation_attributes={
        'security_identity',
        'type',
        'started',
        'finished',
        'flags',
        'code_value',
    },
    optional_operation_attributes={
        'id',
        'password_verified',
    },
    unset_operation_attributes={
        'phone_id2',
        'code_send_count',
        'code_last_sent',
        'code_checks_count',
        'code_confirmed',
    },
    operation_type='dealiasify',
    is_operation_secure=True,
)


def event_lines_phone_created(uid, phone_id, phone_number,
                              action='acquire_phone', created_unixtime=None,
                              user_agent='curl', consumer='dev'):
    if created_unixtime is None:
        created_unixtime = TimeNow()
    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id

    return (
        {'uid': str(uid), 'name': 'action', 'value': action},
        {'uid': str(uid), 'name': phone_fmt + 'action', 'value': 'created'},
        {'uid': str(uid), 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': str(uid), 'name': phone_fmt + 'created', 'value': created_unixtime},
        {'uid': str(uid), 'name': 'user_agent', 'value': user_agent},
        {'uid': str(uid), 'name': 'consumer', 'value': consumer},
    )


def event_lines_secure_bind_operation_created(uid, phone_id, phone_number,
                                              operation_id, operation_ttl=None,
                                              started_unixtime=None, is_new_phone=True):
    if is_new_phone:
        phone_action = 'created'
    else:
        phone_action = 'changed'

    if operation_ttl is None:
        operation_ttl = timedelta(seconds=settings.PHONE_QUARANTINE_SECONDS)

    if started_unixtime is None:
        started_unixtime = TimeNow()
        finished_unixtime = TimeNow(offset=operation_ttl.total_seconds())
    else:
        raise NotImplementedError()

    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    op_fmt = phone_fmt + 'operation.%d.' % operation_id

    retval = (
        {'uid': uid, 'name': phone_fmt + 'action', 'value': phone_action},
    )
    if is_new_phone:
        retval += (
            {'uid': uid, 'name': phone_fmt + 'created', 'value': TimeNow()},
        )
    retval += (
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': op_fmt + 'action', 'value': 'created'},
        {'uid': uid, 'name': op_fmt + 'finished', 'value': finished_unixtime},
        {'uid': uid, 'name': op_fmt + 'security_identity', 'value': str(SECURITY_IDENTITY)},
        {'uid': uid, 'name': op_fmt + 'started', 'value': started_unixtime},
        {'uid': uid, 'name': op_fmt + 'type', 'value': 'bind'},
    )
    return retval


def event_lines_secure_bind_operation_deleted(uid, phone_id, phone_number,
                                              operation_id, action='action',
                                              user_agent='curl', consumer='dev'):
    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    op_fmt = phone_fmt + 'operation.%d.' % operation_id
    return (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': 'changed'},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': op_fmt + 'action', 'value': 'deleted'},
        {'uid': uid, 'name': op_fmt + 'type', 'value': 'bind'},
        {'uid': uid, 'name': op_fmt + 'security_identity', 'value': str(SECURITY_IDENTITY)},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_secure_phone_bound(uid, phone_id, phone_number, operation_id=None,
                                   action='action', user_agent='curl', consumer='dev'):
    if operation_id is not None:
        phone_action = 'changed'
    else:
        phone_action = 'created'

    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id

    return (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': phone_action},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': phone_fmt + 'confirmed', 'value': TimeNow()},
        {'uid': uid, 'name': phone_fmt + 'bound', 'value': TimeNow()},
        {'uid': uid, 'name': phone_fmt + 'secured', 'value': TimeNow()},
        {'uid': uid, 'name': 'phones.secure', 'value': str(phone_id)},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_replace_secure_operation_created(uid, replacement_phone_id, replacement_phone_number,
                                                 replacement_operation_id, secure_phone_id,
                                                 secure_phone_number, secure_operation_id,
                                                 operation_ttl=None, is_replacement_phone_bound=False,
                                                 old_bind_operation_id=None,
                                                 action='phone_secure_replace_submit',
                                                 user_agent='curl', consumer='dev'):
    uid = str(uid)
    phone_fmt = 'phone.%d.' % replacement_phone_id
    op_fmt = phone_fmt + 'operation.%d.' % replacement_operation_id

    if operation_ttl is None:
        operation_ttl = timedelta(seconds=settings.PHONE_QUARANTINE_SECONDS)

    started_unixtime = TimeNow()
    finished_unixtime = TimeNow(offset=operation_ttl.total_seconds())

    replacement_phone_events = (
        {'uid': uid, 'name': phone_fmt + 'number', 'value': replacement_phone_number.e164},
        {'uid': uid, 'name': op_fmt + 'action', 'value': u'created'},
        {'uid': uid, 'name': op_fmt + 'security_identity', 'value': replacement_phone_number.digital},
        {'uid': uid, 'name': op_fmt + 'started', 'value': started_unixtime},
        {'uid': uid, 'name': op_fmt + 'finished', 'value': finished_unixtime},
        {'uid': uid, 'name': op_fmt + 'phone_id2', 'value': str(secure_phone_id)},
    )
    if is_replacement_phone_bound:
        replacement_phone_events += (
            {'uid': uid, 'name': op_fmt + 'type', 'value': u'mark'},
        )
    else:
        replacement_phone_events += (
            {'uid': uid, 'name': op_fmt + 'type', 'value': u'bind'},
            {'uid': uid, 'name': phone_fmt + 'created', 'value': TimeNow()},
        )
        if old_bind_operation_id:
            old_op_fmt = phone_fmt + 'operation.%d.' % old_bind_operation_id
            replacement_phone_events += (
                {'uid': uid, 'name': phone_fmt + 'action', 'value': u'changed'},
                {'uid': uid, 'name': old_op_fmt + 'action', 'value': u'deleted'},
                {'uid': uid, 'name': old_op_fmt + 'type', 'value': u'bind'},
                {'uid': uid, 'name': old_op_fmt + 'security_identity', 'value': replacement_phone_number.digital},
            )
        else:
            replacement_phone_events += (
                {'uid': uid, 'name': phone_fmt + 'action', 'value': u'created'},
            )

    phone_fmt = 'phone.%d.' % secure_phone_id
    op_fmt = phone_fmt + 'operation.%d.' % secure_operation_id

    secure_phone_events = (
        {'uid': uid, 'name': phone_fmt + 'number', 'value': secure_phone_number.e164},
        {'uid': uid, 'name': op_fmt + 'action', 'value': u'created'},
        {'uid': uid, 'name': op_fmt + 'type', 'value': u'replace'},
        {'uid': uid, 'name': op_fmt + 'security_identity', 'value': u'1'},
        {'uid': uid, 'name': op_fmt + 'started', 'value': started_unixtime},
        {'uid': uid, 'name': op_fmt + 'finished', 'value': finished_unixtime},
        {'uid': uid, 'name': op_fmt + 'phone_id2', 'value': str(replacement_phone_id)},
    )

    basic_events = (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
    )

    return basic_events + secure_phone_events + replacement_phone_events


def event_lines_simple_phone_bound(uid, phone_id, phone_number, operation_id=None,
                                   action='action', user_agent='curl', consumer='dev'):
    if operation_id is not None:
        phone_action = 'changed'
    else:
        phone_action = 'created'

    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id

    return (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': phone_action},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': phone_fmt + 'confirmed', 'value': TimeNow()},
        {'uid': uid, 'name': phone_fmt + 'bound', 'value': TimeNow()},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_simple_bind_operation_created(uid, phone_id, phone_number,
                                              operation_id, action='acquire_phone',
                                              operation_ttl=None, started_unixtime=None,
                                              user_agent='curl', consumer='dev',
                                              is_new_phone=True):
    if is_new_phone:
        phone_action = 'created'
    else:
        phone_action = 'changed'

    if operation_ttl is None:
        operation_ttl = timedelta(seconds=settings.PHONE_QUARANTINE_SECONDS)

    if started_unixtime is None:
        started_unixtime = TimeNow()
        finished_unixtime = TimeNow(offset=operation_ttl.total_seconds())
    else:
        raise NotImplementedError()

    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    op_fmt = phone_fmt + 'operation.%d.' % operation_id

    return (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': phone_action},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': op_fmt + 'action', 'value': 'created'},
        {'uid': uid, 'name': op_fmt + 'type', 'value': 'bind'},
        {'uid': uid, 'name': op_fmt + 'security_identity', 'value': phone_number.digital},
        {'uid': uid, 'name': op_fmt + 'started', 'value': started_unixtime},
        {'uid': uid, 'name': op_fmt + 'finished', 'value': finished_unixtime},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_simple_bind_operation_deleted(phone_number, action='action',
                                              **kwargs):
    return _event_lines_phone_operation_deleted(
        operation_name='bind',
        action=action,
        phone_number=phone_number,
        security_identity=phone_number.digital,
        **kwargs
    )


def event_lines_phone_secured(uid, phone_id, phone_number, action='action',
                              user_agent='curl', consumer='dev'):
    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    return (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': 'changed'},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': phone_fmt + 'confirmed', 'value': TimeNow()},
        {'uid': uid, 'name': phone_fmt + 'secured', 'value': TimeNow()},
        {'uid': uid, 'name': 'phones.secure', 'value': str(phone_id)},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_phone_confirmed(uid, phone_id, phone_number,
                                action='action', user_agent='curl', consumer='dev'):
    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    return (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': phone_fmt + 'confirmed', 'value': TimeNow()},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': 'changed'},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_phone_admitted(uid, phone_id, phone_number,
                               action='prolong_valid', user_agent='curl', consumer='dev'):
    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    return (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': phone_fmt + 'admitted', 'value': TimeNow()},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': 'changed'},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_securify_operation_created(action='acquire_phone', **kwargs):
    return _event_lines_phone_operation_created(
        operation_name='securify',
        action=action,
        security_identity=str(SECURITY_IDENTITY),
        **kwargs
    )


def event_lines_securify_operation_deleted(action='action', **kwargs):
    return _event_lines_phone_operation_deleted(
        operation_name='securify',
        action=action,
        security_identity=str(SECURITY_IDENTITY),
        **kwargs
    )


def event_lines_aliasify_operation_created(action='acquire_phone', **kwargs):
    return _event_lines_phone_operation_created(
        operation_name='aliasify',
        action=action,
        security_identity=str(SECURITY_IDENTITY),
        **kwargs
    )


def event_lines_aliasify_operation_deleted(action='action', **kwargs):
    return _event_lines_phone_operation_deleted(
        operation_name='aliasify',
        action=action,
        security_identity=str(SECURITY_IDENTITY),
        **kwargs
    )


def event_lines_phonenumber_alias_given_out(uid, phone_id, phone_number,
                                            action='action', user_agent='curl',
                                            consumer='dev'):
    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    return (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': 'alias.phonenumber.add', 'value': phone_number.international},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': 'changed'},
        {'uid': uid, 'name': phone_fmt + 'confirmed', 'value': TimeNow()},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_phonenumber_alias_removed(uid, phone_number, reason_uid=None,
                                          action='phone_alias_delete', user_agent='curl',
                                          consumer='dev'):
    uid = str(uid)
    lines = (
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': 'alias.phonenumber.rm', 'value': phone_number.international},
    )
    if reason_uid is not None:
        lines += ({'uid': uid, 'name': 'reason_uid', 'value': str(reason_uid)},)
    if user_agent is not None:
        lines += ({'uid': uid, 'name': 'user_agent', 'value': user_agent},)
    lines += ({'uid': uid, 'name': 'consumer', 'value': consumer},)
    return lines


def event_lines_phone_unbound(uid, phone_id, phone_number, reason_uid,
                              user_agent='curl', consumer='dev'):
    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    reason_uid = str(reason_uid)
    return (
        {'uid': uid, 'name': 'action', 'value': 'unbind_phone_from_account'},
        {'uid': uid, 'name': phone_fmt + 'action', 'value': 'changed'},
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': phone_fmt + 'bound', 'value': '0'},
        {'uid': uid, 'name': 'reason_uid', 'value': reason_uid},
        {'uid': uid, 'name': 'consumer', 'value': consumer},

        {
            'uid': reason_uid,
            'name': 'unbind_phone_from_account.%s' % uid,
            'value': phone_number.e164,
        },
        {'uid': reason_uid, 'name': 'user_agent', 'value': user_agent},
        {'uid': reason_uid, 'name': 'consumer', 'value': consumer},
    )


def event_lines_mark_operation_created(phone_number, action='acquire_phone',
                                       **kwargs):
    return _event_lines_phone_operation_created(
        phone_number=phone_number,
        operation_name='mark',
        action=action,
        security_identity=phone_number.digital,
        **kwargs
    )


def event_lines_mark_operation_deleted(phone_number, action='action',
                                       **kwargs):
    return _event_lines_phone_operation_deleted(
        phone_number=phone_number,
        operation_name='mark',
        action=action,
        security_identity=phone_number.digital,
        **kwargs
    )


def _event_lines_phone_operation_created(uid, phone_id, phone_number,
                                         operation_id, operation_name,
                                         security_identity, action,
                                         operation_ttl=None,
                                         started_unixtime=None,
                                         user_agent='curl', consumer='dev'):
    if operation_ttl is None:
        operation_ttl = timedelta(seconds=settings.PHONE_QUARANTINE_SECONDS)

    if started_unixtime is None:
        started_unixtime = TimeNow()
        finished_unixtime = TimeNow(offset=operation_ttl.total_seconds())
    else:
        raise NotImplementedError()

    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    op_fmt = phone_fmt + 'operation.%d.' % operation_id

    lines = (
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': op_fmt + 'action', 'value': 'created'},
        {'uid': uid, 'name': op_fmt + 'finished', 'value': finished_unixtime},
        {'uid': uid, 'name': op_fmt + 'security_identity', 'value': security_identity},
        {'uid': uid, 'name': op_fmt + 'started', 'value': started_unixtime},
        {'uid': uid, 'name': op_fmt + 'type', 'value': operation_name},
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
    )
    if user_agent is not None:
        lines += (
            {'uid': uid, 'name': 'user_agent', 'value': user_agent},
        )
    return lines


def _event_lines_phone_operation_deleted(uid, phone_id, phone_number,
                                         operation_id, operation_name,
                                         security_identity, action,
                                         user_agent='curl', consumer='dev'):
    uid = str(uid)
    phone_fmt = 'phone.%d.' % phone_id
    op_fmt = phone_fmt + 'operation.%d.' % operation_id
    return (
        {'uid': uid, 'name': phone_fmt + 'number', 'value': phone_number.e164},
        {'uid': uid, 'name': op_fmt + 'action', 'value': 'deleted'},
        {'uid': uid, 'name': op_fmt + 'security_identity', 'value': security_identity},
        {'uid': uid, 'name': op_fmt + 'type', 'value': operation_name},
        {'uid': uid, 'name': 'action', 'value': action},
        {'uid': uid, 'name': 'consumer', 'value': consumer},
        {'uid': uid, 'name': 'user_agent', 'value': user_agent},
    )
