# -*- coding: utf-8 -*-

from copy import (
    copy,
    deepcopy,
)
from datetime import (
    datetime,
    timedelta,
)
from functools import partial
import logging

from passport.backend.core.builders.blackbox.parsers import PHONE_OP_DEFAULT_VALUES
from passport.backend.core.conf import settings
from passport.backend.core.eav_type_mapping import (
    EXTENDED_ATTRIBUTES_PHONE_TYPE,
    EXTENDED_ATTRIBUTES_TYPE_TO_NAME_MAPPING,
)
from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    CollectionField,
    DescriptorField,
    Field,
    FlagsField,
    IntegerField,
    ModelField,
    UnixtimeField,
)
from passport.backend.core.models.phones.phone import parse_phone_number_field
import passport.backend.core.serializers.run_incr_phone_id.run_incr_phone_id
from passport.backend.core.types.bit_vector.bit_vector import PhoneOperationFlags
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.undefined import Undefined
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import (
    remove_none_values,
    unique_unhashable,
)
from passport.backend.utils.time import (
    datetime_to_integer_unixtime,
    unixtime_to_datetime,
)
from six import (
    iteritems,
    string_types,
)


log = logging.getLogger('passport.models.phones.phones')

EXTENDED_ATTRIBUTES_PHONE_TTN_MAPPING = EXTENDED_ATTRIBUTES_TYPE_TO_NAME_MAPPING[EXTENDED_ATTRIBUTES_PHONE_TYPE]

# Значение свойства security_identity операции, когда операции происходит над
# безопасным телефонным номером.
SECURITY_IDENTITY = 1


def _assert_phone_bound(phone):
    if not phone.bound:
        raise NumberNotBound()


def _assert_phone_secured(phone):
    if not phone.secured:
        raise NumberNotSecured()


def _build_logical_operation(operation, phone_manager):
    """
    Подпрограмма строит из физической операции логическую.

    Выходные параметры
        Логическая операция,
        Список идентификаторов физических операций из которых она состоит.
    """
    for cls in [
        ReplaceSecurePhoneWithNonboundPhoneOperation,
        ReplaceSecurePhoneWithBoundPhoneOperation,
        SecurifyOperation,
        RemoveSecureOperation,
        SimpleBindOperation,
        SecureBindOperation,
        AliasifySecureOperation,
        DealiasifySecureOperation,
        MarkOperation,
    ]:
        try:
            logical_op, ops = cls.build_by_data(operation, phone_manager)
            break
        except BuildOperationError:
            # Это ещё не конец, возможно, что из оставшихся данных получится
            # построить какую-нибудь другую операцию.
            continue
    else:
        raise NotImplementedError(
            u'Таким данным не соответствует ни одной операции %r, %r' %
            (operation, phone_manager.all()),
        )
    return (logical_op, ops)


def _quote(value):
    if value:
        value = '"%s"' % value
    return value


def _recover_from_errors(data):
    # Множество испорченных телефонов
    corrupted = set()
    # Множество телефонов, связанных операциями с побитыми телефонами
    touched = set()

    if 'phones' in data and data['phones']:
        phones = data['phones']
        for phone_id in phones:
            attrs = phones[phone_id].get('attributes', {})
            # Отсутствие номера считаем ошибкой
            if 'number' not in attrs:
                corrupted.add(phone_id)

        for phone_id in phones:
            if phone_id in corrupted:
                continue
            operation = phones[phone_id].get('operation')
            if operation:
                phone_id2 = operation.get('phone_id2')
                if phone_id2 in corrupted:
                    touched.add(phone_id)

        if corrupted:
            # Копируем данные, только если будем их менять.
            data = copy(data)
            phones = data['phones'] = deepcopy(data['phones'])

        for phone_id in list(phones.keys()):
            if phone_id in corrupted:
                del phones[phone_id]
            elif phone_id in touched:
                phones[phone_id]['operation'] = None

    for attr in {'phones.default', 'phones.secure'}:
        if attr in data:
            phone_id = int(data[attr])
            if phone_id in corrupted:
                log.warning('%s with phone_id %d is corrupted', attr, phone_id)
                del data[attr]

    if corrupted:
        log.warning('Phones %s are corrupted', ', '.join(map(str, corrupted)))
    if touched:
        log.warning(
            'Phones %s are valid, but connected with corrupted phones '
            'by operations',
            ', '.join(map(str, touched)),
        )
    return data


class _ConfirmationInfo(object):
    def __init__(self, code_value, code_last_sent, code_send_count,
                 code_checks_count, code_confirmed, yasms_used_gate_ids=None):
        self.code_value = code_value
        self.code_last_sent = code_last_sent
        self.code_send_count = code_send_count
        self.code_checks_count = code_checks_count
        self.code_confirmed = code_confirmed
        self.yasms_used_gate_ids = yasms_used_gate_ids

    def __eq__(self, other):
        if not isinstance(other, _ConfirmationInfo):
            raise NotImplementedError()  # pragma: no cover
        return (
            self.code_value == other.code_value and
            self.code_last_sent == other.code_last_sent and
            self.code_send_count == other.code_send_count and
            self.code_checks_count == other.code_checks_count and
            self.code_confirmed == other.code_confirmed
        )

    def __ne__(self, other):
        return not (self == other)  # pragma: no cover

    def reset_phone(self):
        """
        При смене привязываемого номера частично сбрасываем состояние
        """
        self.code_value = None
        self.code_confirmed = False
        self.code_checks_count = 0
        self.code_send_count = 0
        self.yasms_used_gate_ids = None


class TrackedConfirmationInfo(_ConfirmationInfo):
    def __init__(self, track):
        self.track = track
        code_last_sent = None
        yasms_used_gate_ids = None
        if track.phone_confirmation_last_send_at:
            code_last_sent = unixtime_to_datetime(float(self.track.phone_confirmation_last_send_at))
        if track.phone_confirmation_used_gate_ids:
            yasms_used_gate_ids = track.phone_confirmation_used_gate_ids
        super(TrackedConfirmationInfo, self).__init__(
            track.phone_confirmation_code,
            code_last_sent,
            track.phone_confirmation_sms_count.get(default=0),
            track.phone_confirmation_confirms_count.get(default=0),
            bool(track.phone_confirmation_is_confirmed),
            yasms_used_gate_ids=yasms_used_gate_ids,
        )

        code_first_sent = None
        if track.phone_confirmation_first_send_at:
            code_first_sent = unixtime_to_datetime(float(self.track.phone_confirmation_first_send_at))
        self.code_first_sent = code_first_sent

    def save(self):
        self.track.phone_confirmation_code = self.code_value
        code_first_sent_timestamp = None
        if self.code_first_sent:
            code_first_sent_timestamp = datetime_to_integer_unixtime(self.code_first_sent)
        self.track.phone_confirmation_first_send_at = code_first_sent_timestamp
        code_last_sent_timestamp = None
        if self.code_last_sent:
            code_last_sent_timestamp = datetime_to_integer_unixtime(self.code_last_sent)
        self.track.phone_confirmation_last_send_at = code_last_sent_timestamp
        for updated_value, counter in (
            (self.code_send_count, self.track.phone_confirmation_sms_count),
            (self.code_checks_count, self.track.phone_confirmation_confirms_count),
        ):
            diff = updated_value - counter.get(default=0)
            if diff < 0:
                counter.reset()
                diff = updated_value
            for _ in range(diff):
                counter.incr()

        self.track.phone_confirmation_is_confirmed = bool(self.code_confirmed)
        self.track.phone_confirmation_used_gate_ids = self.yasms_used_gate_ids

    def __eq__(self, other):
        return (
            super(TrackedConfirmationInfo, self).__eq__(other) and
            self.code_first_sent == other.code_first_sent
        )


class PhoneChangeSet(object):
    def __init__(self, unbound_numbers=None, bound_numbers=None, secured_number=None):
        # номера в форме e164
        self.unbound_numbers = unbound_numbers or set()
        self.bound_numbers = bound_numbers or set()
        self.secured_number = secured_number or None

    def __add__(self, other):
        if type(other) is not PhoneChangeSet:
            raise NotImplementedError(other)  # pragma: no cover
        return PhoneChangeSet(
            unbound_numbers=(self.unbound_numbers | other.unbound_numbers),
            bound_numbers=(self.bound_numbers | other.bound_numbers),
            secured_number=self.secured_number or other.secured_number,
        )

    def __eq__(self, other):
        if type(other) is not PhoneChangeSet:
            raise NotImplementedError(other)  # pragma: no cover
        return (self.unbound_numbers == other.unbound_numbers and
                self.bound_numbers == other.bound_numbers and
                self.secured_number == other.secured_number)

    def __ne__(self, other):
        return not (self == other)

    def __repr__(self):
        return '<%s bound_numbers=%r, unbound_numbers=%r, secured_number=%r>' % (
            self.__class__.__name__,
            sorted(self.bound_numbers),
            sorted(self.unbound_numbers),
            self.secured_number,
        )


class _BaseLogicalOperation(object):
    """
    Логическая операция суть композиция одной или нескольких физических
    операций.

    Методы логической операции поддерживают непротиворечивость её физических
    операций, применяют и отменяют физические операции, учитывая полное
    состояние логической операции.
    """

    # Название операции для журнала
    name = None
    _primary_op = None
    _secondary_op = None

    # Операция блокирует защищённый телефон
    _locks_secure_phone = False

    # Операция допускает помещение в карантин
    # Большинство операций никогда нельзя помещать в карантин
    _allows_qurantine = False

    @classmethod
    def build_by_data(cls, operation, phone_manager):
        """
        Построить логическую операцию из физической.

        Входные параметры
            operation
                Фзическая операция
            phone_manager
                Менеджер телефонов на аккаунте.

        Выходные параметры
            (
                Логическая операция,

                Множество из физических операций из которых построилась
                логическая операция,
            )

        Исключения
            BuildOperationError
                Из данной физической операции нельзя построить эту логическую
                операцию.
        """
        raise BuildOperationError()  # pragma: no cover

    @classmethod
    def create(cls, **kwargs):
        """
        Создать логическую операцию на аккаунте.
        """
        raise NotImplementedError(cls)  # pragma: no cover

    @classmethod
    def get_conflict_operations(cls, phone_manager, phone_number, statbox):
        """
        Возвращает список операций, которые конфликтуют с данной.
        """
        conflict_ops = []

        phone = phone_manager.by_number(phone_number)
        if phone:
            current_op = phone.get_logical_operation(statbox)
            if current_op:
                conflict_ops.append(current_op)

        if cls._locks_secure_phone:
            secure_op = phone_manager.get_secure_logical_operation(statbox)
            if secure_op:
                conflict_ops.append(secure_op)

        conflict_ops = unique_unhashable(conflict_ops)
        return conflict_ops

    def get_confirmation_info(self, phone_id):
        """
        Прочитать сведения о том, что заказчик операции подтвердил обладание
        телефонным номером.

        Входные параметры
            phone_id
                Идентификатор телефона

        Выходные параметры
            Объект ConfirmationInfo

        Исключения
            PhoneDoesNotTakePartInOperation
        """
        operation = self._get_operation_by_phone_id(phone_id)
        return _ConfirmationInfo(
            code_value=operation.code_value or None,
            code_last_sent=operation.code_last_sent or None,
            code_send_count=operation.code_send_count or 0,
            code_checks_count=operation.code_checks_count or 0,
            code_confirmed=operation.code_confirmed or None,
        )

    def set_confirmation_info(self, phone_id, confirmation_info):
        """
        Записать сведения о том, что пользователь подтвердил обладание
        телефонным номером.

        Входные параметры
            phone_id
                Идентификатор телефона
            confirmation_info
                Объект ConfirmationInfo

        Исключения
            PhoneDoesNotTakePartInOperation
        """
        self._set_confirmation_info(phone_id, confirmation_info)

    def _set_confirmation_info(self, phone_id, confirmation_info, assert_alive=True):
        if assert_alive:
            self._assert_alive()
        operation = self._get_operation_by_phone_id(phone_id)
        operation.code_value = confirmation_info.code_value
        operation.code_last_sent = confirmation_info.code_last_sent
        operation.code_send_count = confirmation_info.code_send_count
        operation.code_checks_count = confirmation_info.code_checks_count
        operation.code_confirmed = confirmation_info.code_confirmed

    @property
    def is_binding(self):
        """
        Операция является привязкой.
        """
        if self._primary_op.type == u'bind':
            return True
        elif (self._secondary_op is not None and
              self._secondary_op.type == u'bind'):
            return True
        else:
            return False

    @property
    def id(self):
        # Логическая операция состоит хотя бы из одной физической, таким образом
        # безопасно брать идентификатор операции таким образом.
        return self._primary_op.id or None

    @property
    def started(self):
        # Логическая операция состоит хотя бы из одной физической, таким образом
        # безопасно брать время начала жизни операции таким образом.
        #
        # Считается, что время начала жизни (время завершения жизни, флаги) всех
        # физических операций одной логической операции должно быть одинаковым.
        # Подклассы обязаны поддерживать это ограничение в конструкторе и
        # остальных методах.
        return self._primary_op.started or None

    @property
    def finished(self):
        # Логическая операция состоит хотя бы из одной физической, таким образом
        # безопасно брать время завершения жизни операции таким образом.
        #
        # Смотри комментарий к started.
        return self._primary_op.finished or None

    @property
    def flags(self):
        # Логическая операция состоит хотя бы из одной физической, таким образом
        # безопасно брать флаги операции таким образом.
        #
        # Смотри комментарий к started.
        return self._primary_op.flags

    @property
    def is_expired(self):
        return self._primary_op.is_expired

    @property
    def is_secure(self):
        return self._primary_op.is_secure

    def _get_password_verified(self):
        """
        Время, в которое заказчик операции подтвердил свою личность.

        Если заказчик ещё не подтерждал личность равняется None.
        """
        return self._primary_op.password_verified or None

    def _set_password_verified(self, time, assert_alive=True):
        if assert_alive:
            self._assert_alive()
        if self._get_password_verified():
            raise PasswordVerifiedAlready()
        self._primary_op.password_verified = time
        if self._secondary_op is not None:
            self._secondary_op.password_verified = time
        if self.is_ready_for_quarantine():
            self.start_quarantine()

    password_verified = property(_get_password_verified, _set_password_verified)

    def confirm_phone(self, phone_id, code, timestamp=None, should_check_code=True,
                      statbox_enabled=True):
        """
        Подтвердить обладание телефоном через код.

        Входные параметры
            phone_id
                Идентификатор телефона
            code
                Проверочный код
            timestamp
                Время, когда происходит подтверждение, если не задано
                берётся нынешний момент.
            should_check_code
                Требуется сверка кода

        Выходные параметры
            (
                код подтверждён (True, False),
                число оставшихся попыток,
            )

        Исключения
            PhoneConfirmedAlready
            PhoneDoesNotTakePartInOperation
            ConfirmationLimitExceeded
        """
        self._assert_alive()
        operation = self._get_operation_by_phone_id(phone_id)
        if operation.code_confirmed:
            raise PhoneConfirmedAlready()

        if should_check_code:
            is_code_valid, code_checks_left = self.check_code(phone_id, code)
            if not is_code_valid:
                return False, code_checks_left

        operation.code_confirmed = timestamp or datetime.now()
        phone = self._phone_manager.by_id(phone_id)
        self._update_confirmation_time(phone)

        if statbox_enabled:
            self.statbox.stash(
                action=self.statbox.Link(u'phone_confirmed'),
                number=phone.number.masked_format_for_statbox,
                phone_id=phone.id,
                confirmation_time=operation.code_confirmed,
                code_checks_count=operation.code_checks_count,
                operation_id=self.id,
                uid=self._account.uid,
            )

        if self.is_ready_for_quarantine():
            self.start_quarantine()

        code_checks_left = settings.SMS_VALIDATION_MAX_CHECKS_COUNT - operation.code_checks_count
        return True, code_checks_left

    def check_code(self, phone_id, code):
        self._assert_alive()

        operation = self._get_operation_by_phone_id(phone_id)
        if operation.code_confirmed:
            raise PhoneConfirmedAlready()

        if operation.code_checks_count >= settings.SMS_VALIDATION_MAX_CHECKS_COUNT:
            raise ConfirmationLimitExceeded()
        operation.code_checks_count += 1
        code_checks_left = settings.SMS_VALIDATION_MAX_CHECKS_COUNT - operation.code_checks_count

        if operation.code_value != code:
            phone = self._phone_manager.by_id(phone_id)
            self.statbox.stash(
                action=self.statbox.Link(u'confirm_phone'),
                error=u'code.invalid',
                number=phone.number.masked_format_for_statbox,
                phone_id=phone.id,
                code_checks_count=operation.code_checks_count,
                operation_id=self.id,
                uid=self._account.uid,
            )
            return False, code_checks_left

        return True, code_checks_left

    def apply(self, need_authenticated_user=True, timestamp=None,
              is_harvester=False, **kwargs):
        if not is_harvester:
            self._assert_alive()

        timestamp = timestamp or datetime.now()

        old_secure = self._phone_manager.secure
        old_phones = deepcopy(self._phone_manager.all())
        next_op = self._apply(need_authenticated_user, timestamp, is_harvester, **kwargs)
        new_phones = self._phone_manager.all()

        unbound_phones = set()
        for phone_id, old_phone in iteritems(old_phones):
            new_phone = new_phones.get(phone_id)
            if old_phone.bound and not (new_phone and new_phone.bound):
                unbound_phones.add(old_phone)

        bound_phones = set()
        for phone_id, new_phone in iteritems(new_phones):
            old_phone = old_phones.get(phone_id)
            if new_phone.bound and not (old_phone and old_phone.bound):
                bound_phones.add(new_phone)

        new_secure = self._phone_manager.secure
        if new_secure and new_secure != old_secure:
            secured_phone = new_secure
        else:
            secured_phone = None

        return (
            next_op,
            PhoneChangeSet(
                unbound_numbers={p.number.e164 for p in unbound_phones},
                bound_numbers={p.number.e164 for p in bound_phones},
                secured_number=secured_phone and secured_phone.number.e164,
            ),
        )

    def _apply(self, need_authenticated_user, timestamp, is_harvester):
        """
        Метод должен выполнять непосредственные действия по исполнению операции.

        Замечания
            Метод должен быть атомарным в смысле всё-или-ничего.
        """

    def cancel(self, is_harvester=False):
        if not is_harvester:
            self._assert_alive()
        self._cancel()
        self._post_cancel()

    def _cancel(self):
        """
        Метод должен выполнять непосредственные действия по отмене операции.
        """

    def _post_cancel(self):
        self.statbox.log(
            action=self.statbox.Link(u'phone_operation_cancelled'),
            operation_type=self.name,
            operation_id=self.id,
            uid=self._account.uid,
        )

    def to_simple_bind(self, phone, code):
        """
        Действия, которые нужно выполнить, если в момент исполнения этой
        операции, была вызвана ручка register(secure=False).

        Примечение
            Метод предназначен, только для ручки register. Запрещается
            использовать его в других ручках, т.к. неявное сведение операции
            одного типа к операции другого типа или же повторное начало
            операции для уже начатой -- это слишком сложно.
        """
        self._assert_alive()
        if phone.bound:
            raise NumberBoundAlready()
        operation = self._get_operation_by_phone_id(phone.id)
        if operation.is_secure:
            raise SecureBindToSimpleBindError()
        if not self.is_binding:
            raise OperationExists()
        return self

    def to_secure_bind(self, phone, code, should_ignore_binding_limit):
        """
        Действия, которые нужно выполнить, если в момент исполнения этой
        операции, была вызвана ручка register(secure=True).

        Смотри примечание к to_simple_bind.
        """
        self._assert_alive()
        if phone.bound:
            raise NumberBoundAlready()
        if self._phone_manager.secure:
            raise SecureNumberBoundAlready()

        secure_op = self._phone_manager.get_secure_logical_operation(self.statbox)
        if secure_op and secure_op != self:
            raise SingleSecureOperationError()

        if not self.is_binding:
            raise OperationExists()
        self._get_operation_by_phone_id(phone.id)
        return self

    def statbox_entries_for_create(self):
        phone = self._phone_manager.by_id(self._primary_op.phone_id)
        return [
            dict(
                action=u'phone_operation_created',
                operation_type=self.name,
                number=phone.number.masked_format_for_statbox,
                phone_id=phone.id,
                operation_id=self.id,
                uid=self._account.uid,
            ),
        ]

    def _assert_alive(self):
        if self.is_expired:
            raise OperationExpired()

    def start_quarantine(self):
        """
        Помещает операцию в карантин.

        Методу не важно, выполняются ли необходимые условия карантина.
        Проверить необходимые условия можно методом is_ready_for_quarantine.
        """
        if not self._allows_qurantine:
            # Предполагается, что программист не станет вызывать этот метод,
            # для операций, которым карантин запрещён в принципе (воспринимай
            # этот как подстеленную соломку).
            raise NotImplementedError()

        if self.in_quarantine:
            return

        # Выставляем новое время завершения жизни операции.
        finished = datetime.now() + timedelta(seconds=settings.PHONE_QUARANTINE_SECONDS)
        self._primary_op.finished = finished
        self._primary_op.flags.in_quarantine = True
        if self._secondary_op is not None:
            self._secondary_op.finished = finished
            self._secondary_op.flags.in_quarantine = True

    @property
    def in_quarantine(self):
        """
        Операция находится в карантине.
        """
        return bool(self.flags.in_quarantine)

    def is_ready_for_quarantine(self, need_authenticated_user=True):
        # Для большинства операций карантин невозможен, т.е. они никогда не
        # готовы к помещению в карантин.
        return False

    @classmethod
    def _get_ttl(cls):
        # Не сделал атрибутом класса, т.к. тогда в тестах мы не сможем менять
        # значение подменяя settings, поэтому сделал методом класса.
        return timedelta(seconds=settings.PHONE_QUARANTINE_SECONDS)

    def _get_operation_by_phone_id(self, phone_id):
        if self._primary_op.phone_id == phone_id:
            return self._primary_op
        elif (self._secondary_op is not None and
              self._secondary_op.phone_id == phone_id):
            return self._secondary_op
        else:
            raise PhoneDoesNotTakePartInOperation()

    def _get_unconfirmed_phones(self):
        unconfirmed_phones = set()
        if not self._primary_op.code_confirmed:
            unconfirmed_phones.add(
                self._phone_manager.by_id(self._primary_op.phone_id),
            )
        if (self._secondary_op is not None and
                not self._secondary_op.code_confirmed):
            unconfirmed_phones.add(
                self._phone_manager.by_id(self._secondary_op.phone_id),
            )
        return unconfirmed_phones

    def __eq__(self, other):
        if not isinstance(other, _BaseLogicalOperation):
            raise NotImplementedError(other)  # pragma: no cover
        return (self._primary_op, self._secondary_op) == (other._primary_op, other._secondary_op)

    def __ne__(self, other):
        return not (self == other)

    def __lt__(self, other):
        if not isinstance(other, _BaseLogicalOperation):
            raise NotImplementedError(other)  # pragma: no cover
        return self.id < other.id

    @property
    def _account(self):
        return self._phone_manager.parent

    def is_inapplicable(self, need_authenticated_user=True, is_harvester=False):
        if self.is_expired and not (is_harvester and self.in_quarantine):
            return OperationExpired()

    def __repr__(self):
        return ('<LogicalOperation: id=%s name=%s primary_operation=%r secondary_operation=%r>' % (
            self.id,
            self.name,
            self._primary_op,
            self._secondary_op,
        ))

    def _update_confirmation_time(self, phone):
        operation = phone.operation
        if phone.confirmed and operation and operation.code_confirmed:
            timestamp = max(phone.confirmed, operation.code_confirmed)
        else:
            timestamp = operation and operation.code_confirmed or phone.confirmed
        phone.confirmed = timestamp


class SimpleBindOperation(_BaseLogicalOperation):
    """
    Логическая операция привязки простого номера.
    """

    name = u'simple_bind'

    @classmethod
    def build_by_data(cls, operation, phone_manager):
        if not (operation.type == u'bind' and not operation.is_secure):
            raise BuildOperationError()
        return (cls(operation, phone_manager, statbox=None), {operation})

    @classmethod
    def create(cls, phone_manager, phone_id, code, should_ignore_binding_limit,
               statbox, ttl=None):
        phone = phone_manager.by_id(phone_id)
        if phone.bound:
            raise NumberBoundAlready()
        flags = PhoneOperationFlags()
        flags.should_ignore_binding_limit = should_ignore_binding_limit
        now = datetime.now()

        ttl = ttl or cls._get_ttl()
        phone.create_operation(
            type=u'bind',
            security_identity=int(phone.number),
            code_value=code,
            started=now,
            finished=now + ttl,
            code_checks_count=0,
            code_send_count=0,
            code_last_sent=None,
            code_confirmed=None,
            flags=flags,
        )
        logical_operation = cls(phone.operation, phone_manager, statbox)

        return logical_operation

    def __init__(self, operation, phone_manager, statbox):
        self._primary_op = self._simple_bind_operation = operation
        self._secondary_op = None
        self._phone_manager = phone_manager
        self.statbox = statbox

    def _apply(self, need_authenticated_user, timestamp, is_harvester):
        """
        Привязывает простой телефон.

        Замечания
            Не забудьте отвязать номер от других учётных записей.
        """
        super(SimpleBindOperation, self)._apply(
            need_authenticated_user=need_authenticated_user,
            timestamp=timestamp,
            is_harvester=is_harvester,
        )

        reason = self.is_inapplicable(need_authenticated_user)
        if reason is not None:
            raise reason

        phone = self._phone_manager.by_id(self._simple_bind_operation.phone_id)
        phone.bound = timestamp
        self._update_confirmation_time(phone)
        if phone.operation.flags is not None:
            phone.binding.should_ignore_binding_limit = phone.operation.flags.should_ignore_binding_limit
        phone.operation = None

        self.statbox.stash(
            action=self.statbox.Link(u'simple_phone_bound'),
            number=phone.number.masked_format_for_statbox,
            phone_id=phone.id,
            operation_id=self.id,
            uid=self._account.uid,
        )

    def _cancel(self):
        """
        Удаляет номер и операцию.
        """
        super(SimpleBindOperation, self)._cancel()
        self._phone_manager.remove(self._simple_bind_operation.phone_id)

    def to_secure_bind(self, phone, code, should_ignore_binding_limit):
        # Для привязки простого заменяем операцию не меняя идентификатор
        # телефона.
        super(SimpleBindOperation, self).to_secure_bind(
            phone,
            code,
            should_ignore_binding_limit,
        )
        old_operation_id = self.id
        phone.operation = None
        secure_bind_operation = SecureBindOperation.create(
            self._phone_manager,
            phone.id,
            code,
            should_ignore_binding_limit,
            statbox=self.statbox,
        )
        self.statbox.stash(
            action=self.statbox.Link(u'phone_operation_replaced'),
            phone_id=phone.id,
            number=phone.number.masked_format_for_statbox,
            old_operation_type=self.name,
            old_operation_id=old_operation_id,
            operation_type=secure_bind_operation.name,
            operation_id=self.id,
            uid=self._account.uid,
        )
        return secure_bind_operation

    def is_inapplicable(self, need_authenticated_user=True, is_harvester=False):
        reason = super(SimpleBindOperation, self).is_inapplicable(need_authenticated_user)
        if reason:
            return reason

        if not self._simple_bind_operation.code_confirmed:
            return OperationInapplicable(
                need_confirmed_phones=self._get_unconfirmed_phones(),
            )


class SecureBindOperation(_BaseLogicalOperation):
    """
    Логическая операция привязки защищённого номера.
    """

    name = u'secure_bind'
    _locks_secure_phone = True

    @classmethod
    def build_by_data(cls, operation, phone_manager):
        if not (operation.type == u'bind' and operation.is_secure):
            raise BuildOperationError()
        return (cls(operation, phone_manager, statbox=None), {operation})

    @classmethod
    def create(cls, phone_manager, phone_id, code, should_ignore_binding_limit,
               statbox, aliasify=False, ttl=None):
        phone = phone_manager.by_id(phone_id)
        if phone.bound:
            raise NumberBoundAlready()
        if phone_manager.secure:
            raise SecureNumberBoundAlready()
        if any(
            op.is_secure
            for op in phone_manager.get_logical_operations(statbox)
        ):
            raise SingleSecureOperationError()

        flags = PhoneOperationFlags()
        flags.should_ignore_binding_limit = should_ignore_binding_limit
        flags.aliasify = aliasify

        now = datetime.now()
        ttl = ttl or cls._get_ttl()
        phone.create_operation(
            type=u'bind',
            security_identity=SECURITY_IDENTITY,
            code_value=code,
            started=now,
            finished=now + ttl,
            code_checks_count=0,
            code_send_count=0,
            code_last_sent=None,
            code_confirmed=None,
            password_verified=None,
            flags=flags,
        )
        logical_operation = cls(phone.operation, phone_manager, statbox)

        return logical_operation

    def __init__(self, operation, phone_manager, statbox):
        self._primary_op = self._secure_bind_operation = operation
        self._secondary_op = None
        self._phone_manager = phone_manager
        self.statbox = statbox

    def _apply(self, need_authenticated_user, timestamp, is_harvester):
        """
        Привязывает защищённый телефон.

        Замечания
            Не забудьте отвязать номер от других учётных записей.
        """
        super(SecureBindOperation, self)._apply(
            need_authenticated_user=need_authenticated_user,
            timestamp=timestamp,
            is_harvester=is_harvester,
        )
        is_authenticated = self._secure_bind_operation.password_verified or not need_authenticated_user
        if not (self._secure_bind_operation.code_confirmed and is_authenticated):
            raise OperationInapplicable(
                need_password_verification=not is_authenticated,
                need_confirmed_phones=self._get_unconfirmed_phones(),
            )

        phone = self._phone_manager.by_id(self._secure_bind_operation.phone_id)
        phone.bound = phone.secured = timestamp
        self._update_confirmation_time(phone)
        if phone.operation.flags is not None:
            phone.binding.should_ignore_binding_limit = phone.operation.flags.should_ignore_binding_limit
        phone.operation = None
        self._phone_manager.secure = phone

        self.statbox.stash(
            action=self.statbox.Link(u'secure_phone_bound'),
            number=phone.number.masked_format_for_statbox,
            phone_id=phone.id,
            operation_id=self.id,
            uid=self._account.uid,
        )

    def _cancel(self):
        """
        Удаляет номер и операцию.
        """
        super(SecureBindOperation, self)._cancel()
        self._phone_manager.remove(self._secure_bind_operation.phone_id)


class ReplaceSecurePhoneWithNonboundPhoneOperation(_BaseLogicalOperation):
    """
    Логическая операция замены защищённого номера на непривязанный состоит из
    двух связанных физических операций: операции привязки простого номера и
    операции замены защищённого номера.
    """

    name = u'replace_secure_phone_with_nonbound_phone'
    _locks_secure_phone = True
    _allows_qurantine = True

    @classmethod
    def build_by_data(cls, operation, phone_manager):
        replace = operation
        if not replace.phone_id2:
            raise BuildOperationError()
        bind = phone_manager.by_id(replace.phone_id2).operation

        if replace.type != u'replace':
            replace, bind = bind, replace

        is_bind_ok = bind.type == u'bind' and not bind.is_secure
        is_replace_ok = replace.type == u'replace' and replace.is_secure
        are_phones_equal = (bind.phone_id == replace.phone_id2 and
                            replace.phone_id == bind.phone_id2)
        if not(is_bind_ok and is_replace_ok and are_phones_equal):
            raise BuildOperationError()

        return (cls(replace, bind, phone_manager, statbox=None), {bind, replace})

    @classmethod
    def create(cls, phone_manager, secure_phone_id, being_bound_phone_id,
               secure_code, being_bound_code, statbox, ttl=None):
        now = datetime.now()
        ttl = ttl or cls._get_ttl()
        kwargs = dict(
            started=now,
            finished=now + ttl,
            code_checks_count=0,
            code_send_count=0,
            code_last_sent=None,
            code_confirmed=None,
            password_verified=None,
        )

        secure = phone_manager.by_id(secure_phone_id)
        being_bound = phone_manager.by_id(being_bound_phone_id)

        if being_bound.bound:
            raise NumberBoundAlready()
        if not (phone_manager.secure and phone_manager.secure == secure):
            raise NumberNotSecured()

        secure.create_operation(
            type=u'replace',
            security_identity=SECURITY_IDENTITY,
            code_value=secure_code,
            phone_id2=being_bound_phone_id,
            **kwargs
        )

        being_bound.create_operation(
            type=u'bind',
            security_identity=int(being_bound.number),
            code_value=being_bound_code,
            phone_id2=secure_phone_id,
            **kwargs
        )

        logical_operation = cls(
            secure.operation,
            being_bound.operation,
            phone_manager,
            statbox,
        )

        return logical_operation

    def __init__(self, replace_operation, bind_operation, phone_manager,
                 statbox):
        self._primary_op = self._replace_secure_operation = replace_operation
        self._secondary_op = self._simple_bind_operation = bind_operation
        self._phone_manager = phone_manager
        self.statbox = statbox

    def _apply(self, need_authenticated_user, timestamp, is_harvester):
        """
        Привязывает простой телефон и, если возможно, делает его защищённым, а
        старый защищённый телефон удаляет.

        Замечания
            Не забудьте отвязать номер от других учётных записей.
        """
        # Нужно запомнить некоторые сведения, они пригодятся для пересоздания
        # операции.
        bind_phone = self._phone_manager.by_id(self._simple_bind_operation.phone_id)
        bind_confirmation_info = self.get_confirmation_info(bind_phone.id)
        old_started = self.started
        old_finished = self.finished
        old_operation_id = self.id
        old_password_verified = self.password_verified
        old_flags = self.flags

        super(ReplaceSecurePhoneWithNonboundPhoneOperation, self)._apply(
            need_authenticated_user=need_authenticated_user,
            timestamp=timestamp,
            is_harvester=is_harvester,
        )

        reason = self.is_inapplicable(
            need_authenticated_user=need_authenticated_user,
            is_harvester=is_harvester,
        )
        if reason is not None and not self._simple_bind_operation.code_confirmed:
            raise reason

        phone = self._phone_manager.by_id(self._simple_bind_operation.phone_id)
        phone.bound = timestamp
        self._update_confirmation_time(phone)
        if phone.operation.flags is not None:
            phone.binding.should_ignore_binding_limit = phone.operation.flags.should_ignore_binding_limit
        phone.operation = None

        self.statbox.stash(
            action=self.statbox.Link(u'simple_phone_bound'),
            number=phone.number.masked_format_for_statbox,
            phone_id=phone.id,
            operation_id=self.id,
            uid=self._account.uid,
        )

        replace_phone = self._phone_manager.by_id(self._replace_secure_operation.phone_id)
        replace_confirmation_info = self.get_confirmation_info(replace_phone.id)
        replace_phone.operation = None
        replace_operation = ReplaceSecurePhoneWithBoundPhoneOperation.create(
            phone_manager=self._phone_manager,
            secure_phone_id=replace_phone.id,
            simple_phone_id=bind_phone.id,
            secure_code=replace_confirmation_info.code_value,
            simple_code=bind_confirmation_info.code_value,
            operation_id=old_operation_id,
            statbox=self.statbox,
        )
        replace_operation._set_confirmation_info(bind_phone.id, bind_confirmation_info, assert_alive=False)
        replace_operation._set_confirmation_info(replace_phone.id, replace_confirmation_info, assert_alive=False)
        replace_operation._set_password_verified(old_password_verified, assert_alive=False)
        replace_operation._primary_op.started = old_started
        replace_operation._primary_op.finished = old_finished
        replace_operation._primary_op.flags = old_flags
        replace_operation._secondary_op.started = old_started
        replace_operation._secondary_op.finished = old_finished
        replace_operation._secondary_op.flags = old_flags

        try:
            replace_operation.apply(
                need_authenticated_user=need_authenticated_user,
                timestamp=timestamp,
                is_harvester=is_harvester,
            )
        except OperationInapplicable:
            # Состояние операции пока не позволяет провести замену защищённого
            # телефона полностью, однако телефон стал привязанным и на нём
            # начата операция замены защищённого на привязанный.
            return replace_operation

    def _cancel(self):
        """
        Удаляет привязываемый телефон и физическую операцию на нём с учётной
        записи.
        Удаляет физическую операцию на защищённом номере.
        """
        super(ReplaceSecurePhoneWithNonboundPhoneOperation, self)._cancel()
        self._phone_manager.remove(self._simple_bind_operation.phone_id)
        phone = self._phone_manager.by_id(self._replace_secure_operation.phone_id)
        phone.operation = None

    def is_ready_for_quarantine(self, need_authenticated_user=True):
        is_authenticated = self.password_verified or not need_authenticated_user
        return bool(
            self._simple_bind_operation.code_confirmed and
            is_authenticated and
            not self._replace_secure_operation.does_user_admit_phone
        )

    def is_inapplicable(self, need_authenticated_user=True, is_harvester=False):
        reason = super(ReplaceSecurePhoneWithNonboundPhoneOperation, self).is_inapplicable(
            need_authenticated_user=need_authenticated_user,
            is_harvester=is_harvester,
        )
        if reason is not None:
            return reason

        if self.in_quarantine and is_harvester:
            return

        is_authenticated = self.password_verified or not need_authenticated_user
        if not (self._simple_bind_operation.code_confirmed and
                self._replace_secure_operation.code_confirmed and
                is_authenticated):
            return OperationInapplicable(
                need_password_verification=not is_authenticated,
                need_confirmed_phones=self._get_unconfirmed_phones(),
            )

    @property
    def does_user_admit_phone(self):
        return self._replace_secure_operation.does_user_admit_phone

    def statbox_entries_for_create(self):
        secure = self._phone_manager.secure
        being_bound = self._phone_manager.by_id(self._simple_bind_operation.phone_id)
        return [
            dict(
                action=u'phone_operation_created',
                operation_type=self.name,
                secure_number=secure.number.masked_format_for_statbox,
                secure_phone_id=secure.id,
                being_bound_number=being_bound.number.masked_format_for_statbox,
                being_bound_phone_id=being_bound.id,
                operation_id=self.id,
                uid=self._account.uid,
            ),
        ]


class ReplaceSecurePhoneWithBoundPhoneOperation(_BaseLogicalOperation):
    """
    Логическая операция замены защищённого номера на простой состоит из
    двух связанных физических операций: операции заглушки над простым номером и
    операции замены защищённого номера.
    """

    name = u'replace_secure_phone_with_bound_phone'
    _locks_secure_phone = True
    _allows_qurantine = True

    @classmethod
    def build_by_data(cls, operation, phone_manager):
        secure = operation
        if not secure.phone_id2:
            raise BuildOperationError()
        simple = phone_manager.by_id(secure.phone_id2).operation

        if not secure.is_secure:
            secure, simple = simple, secure

        is_simple_ok = simple.type == u'mark' and not simple.is_secure
        is_secure_ok = secure.type == u'replace' and secure.is_secure
        are_phones_equal = (simple.phone_id == secure.phone_id2 and
                            simple.phone_id2 == secure.phone_id)
        if not(is_simple_ok and is_secure_ok and are_phones_equal):
            raise BuildOperationError()

        return (cls(secure, simple, phone_manager, statbox=None), {simple, secure})

    @classmethod
    def create(cls, phone_manager, secure_phone_id, simple_phone_id, secure_code,
               simple_code, statbox, operation_id=None, ttl=None):
        now = datetime.now()
        ttl = ttl or cls._get_ttl()
        kwargs = dict(
            started=now,
            finished=now + ttl,
            code_checks_count=0,
            code_send_count=0,
            code_last_sent=None,
            code_confirmed=None,
            password_verified=None,
        )

        secure = phone_manager.by_id(secure_phone_id)
        simple = phone_manager.by_id(simple_phone_id)

        if not simple.bound:
            raise NumberNotBound()
        if not (phone_manager.secure and phone_manager.secure == secure):
            raise NumberNotSecured()

        secure.create_operation(
            type=u'replace',
            security_identity=SECURITY_IDENTITY,
            code_value=secure_code,
            phone_id2=simple_phone_id,
            **kwargs
        )
        if operation_id is not None:
            secure.operation.id = operation_id

        simple.create_operation(
            type=u'mark',
            security_identity=int(simple.number),
            code_value=simple_code,
            phone_id2=secure_phone_id,
            **kwargs
        )

        logical_operation = cls(
            secure.operation,
            simple.operation,
            phone_manager,
            statbox,
        )

        return logical_operation

    def __init__(self, secure_operation, simple_operation, phone_manager,
                 statbox):
        self._phone_manager = phone_manager
        self._primary_op = self._secure_operation = secure_operation
        self._secondary_op = self._simple_operation = simple_operation
        self.statbox = statbox

    def _apply(self, need_authenticated_user, timestamp, is_harvester):
        """
        Удаляет защищённый номер и делает простой номер защищённым.
        """
        super(ReplaceSecurePhoneWithBoundPhoneOperation, self)._apply(
            need_authenticated_user=need_authenticated_user,
            timestamp=timestamp,
            is_harvester=is_harvester,
        )

        reason = self.is_inapplicable(
            need_authenticated_user=need_authenticated_user,
            is_harvester=is_harvester,
        )
        if reason is not None:
            raise reason

        old_secure_phone = self._phone_manager.by_id(self._secure_operation.phone_id)
        old_secure_phone_number = old_secure_phone.number
        old_secure_phone_id = old_secure_phone.id
        new_secure_phone = self._phone_manager.by_id(self._simple_operation.phone_id)

        self._update_confirmation_time(new_secure_phone)

        if old_secure_phone.is_bank:
            old_secure_phone.operation = None
            old_secure_phone.secured = None
        else:
            self._phone_manager.remove(old_secure_phone.id)

        new_secure_phone.operation = None
        new_secure_phone.secured = timestamp or datetime.now()
        self._phone_manager.secure = new_secure_phone

        self.statbox.stash(
            action=self.statbox.Link(u'secure_phone_replaced'),
            old_secure_number=old_secure_phone_number.masked_format_for_statbox,
            old_secure_phone_id=old_secure_phone_id,
            new_secure_number=new_secure_phone.number.masked_format_for_statbox,
            new_secure_phone_id=new_secure_phone.id,
            operation_id=self.id,
            uid=self._account.uid,
        )

    def _cancel(self):
        """
        Удаляет физические операции над защищённым и простым номерами.
        """
        super(ReplaceSecurePhoneWithBoundPhoneOperation, self)._cancel()
        simple = self._phone_manager.by_id(self._simple_operation.phone_id)
        simple.operation = None
        secure = self._phone_manager.by_id(self._secure_operation.phone_id)
        secure.operation = None

    def is_ready_for_quarantine(self, need_authenticated_user=True):
        is_authenticated = self.password_verified or not need_authenticated_user
        return bool(
            self._simple_operation.code_confirmed and
            is_authenticated and
            not self._secure_operation.does_user_admit_phone
        )

    def is_inapplicable(self, need_authenticated_user=True, is_harvester=False):
        reason = super(ReplaceSecurePhoneWithBoundPhoneOperation, self).is_inapplicable(
            need_authenticated_user=need_authenticated_user,
            is_harvester=is_harvester,
        )
        if reason is not None:
            return reason

        if self.in_quarantine and is_harvester:
            return

        is_authenticated = self.password_verified or not need_authenticated_user
        if not (self._simple_operation.code_confirmed and
                self._secure_operation.code_confirmed and
                is_authenticated):
            return OperationInapplicable(
                need_password_verification=not is_authenticated,
                need_confirmed_phones=self._get_unconfirmed_phones(),
            )

    @property
    def does_user_admit_phone(self):
        return self._secure_operation.does_user_admit_phone

    def statbox_entries_for_create(self):
        secure = self._phone_manager.secure
        simple = self._phone_manager.by_id(self._simple_operation.phone_id)
        return [
            dict(
                action=u'phone_operation_created',
                operation_type=self.name,
                secure_number=secure.number.masked_format_for_statbox,
                secure_phone_id=secure.id,
                simple_number=simple.number.masked_format_for_statbox,
                simple_phone_id=simple.id,
                operation_id=self.id,
                uid=self._account.uid,
            ),
        ]


class SecurifyOperation(_BaseLogicalOperation):
    """
    Логическая операция защиты простого номера.
    """

    name = u'securify'
    _locks_secure_phone = True

    @classmethod
    def build_by_data(cls, operation, phone_manager):
        if not (operation.type == u'securify' and operation.is_secure):
            raise BuildOperationError()
        return (cls(operation, phone_manager, statbox=None), {operation})

    @classmethod
    def create(cls, phone_manager, phone_id, code, statbox, ttl=None):
        """
        Создаём операцию защиты над телефоном с данным кодом подвтерждения.
        """
        phone = phone_manager.by_id(phone_id)
        if not phone.bound:
            raise NumberNotBound()
        if phone_manager.secure:
            raise SecureNumberBoundAlready()
        if any(
            op.is_secure
            for op in phone_manager.get_logical_operations(statbox)
        ):
            raise SingleSecureOperationError()
        now = datetime.now()
        ttl = ttl or cls._get_ttl()
        phone.create_operation(
            type=u'securify',
            security_identity=SECURITY_IDENTITY,
            code_value=code,
            started=now,
            finished=now + ttl,
        )
        logical_operation = cls(phone.operation, phone_manager, statbox)

        return logical_operation

    def __init__(self, operation, phone_manager, statbox):
        self._phone_manager = phone_manager
        self._primary_op = self._securify_operation = operation
        self._secondary_op = None
        self.statbox = statbox

    def _apply(self, need_authenticated_user, timestamp, is_harvester,
               ignore_phones_confirmation=False):
        """
        Делает простой номер защищённым.
        """
        super(SecurifyOperation, self)._apply(
            need_authenticated_user=need_authenticated_user,
            timestamp=timestamp,
            is_harvester=is_harvester,
        )

        is_authenticated = self.password_verified or not need_authenticated_user

        if not (
            (
                ignore_phones_confirmation or
                self._securify_operation.code_confirmed
            ) and
            is_authenticated
        ):
            raise OperationInapplicable(
                need_password_verification=not is_authenticated,
                need_confirmed_phones=self._get_unconfirmed_phones(),
            )

        phone = self._phone_manager.by_id(self._securify_operation.phone_id)
        self._update_confirmation_time(phone)
        phone.secured = timestamp
        phone.operation = None
        self._phone_manager.secure = phone

        self.statbox.stash(
            action=self.statbox.Link(u'phone_secured'),
            number=phone.number.masked_format_for_statbox,
            phone_id=phone.id,
            operation_id=self.id,
            uid=self._account.uid,
        )

    def _cancel(self):
        """
        Удаляет физическую операцию на простом номере.
        """
        super(SecurifyOperation, self)._cancel()
        phone = self._phone_manager.by_id(self._securify_operation.phone_id)
        phone.operation = None


class RemoveSecureOperation(_BaseLogicalOperation):
    """
    Логическая операция удаления защищённого номера.
    """

    name = u'remove_secure'
    _allows_qurantine = True

    @classmethod
    def build_by_data(cls, operation, phone_manager):
        if not (operation.type == u'remove' and operation.is_secure):
            raise BuildOperationError()
        return (cls(operation, phone_manager, statbox=None), {operation})

    @classmethod
    def create(cls, phone_manager, phone_id, code, statbox):
        """
        Создаём операцию удаления над телефоном с данным кодом подвтерждения.
        """
        phone = phone_manager.by_id(phone_id)
        if not phone.bound:
            raise NumberNotBound()
        if phone_manager.secure != phone:
            raise NumberNotSecured()
        if phone.is_bank:
            raise RemoveBankPhoneNumberError()
        now = datetime.now()
        phone.create_operation(
            type=u'remove',
            security_identity=SECURITY_IDENTITY,
            code_value=code,
            started=now,
            finished=now + cls._get_ttl(),
        )
        logical_operation = cls(phone.operation, phone_manager, statbox)

        return logical_operation

    def __init__(self, operation, phone_manager, statbox):
        self._primary_op = self._remove_secure_operation = operation
        self._secondary_op = None
        self._phone_manager = phone_manager
        self.statbox = statbox

    def _apply(self, need_authenticated_user, timestamp, is_harvester):
        """
        Удаляет защищённый номер.
        """
        super(RemoveSecureOperation, self)._apply(
            need_authenticated_user=need_authenticated_user,
            timestamp=timestamp,
            is_harvester=is_harvester,
        )

        is_authenticated = self.password_verified or not need_authenticated_user
        if not (
            self._remove_secure_operation.code_confirmed and is_authenticated or
            self.in_quarantine and is_harvester
        ):
            raise OperationInapplicable(
                need_password_verification=not is_authenticated,
                need_confirmed_phones=self._get_unconfirmed_phones(),
            )

        phone = self._phone_manager.by_id(self._remove_secure_operation.phone_id)
        phone_number = phone.number
        phone_id = phone.id
        self._phone_manager.remove(phone.id)

        self.statbox.stash(
            action=self.statbox.Link(u'secure_phone_removed'),
            number=phone_number.masked_format_for_statbox,
            phone_id=phone_id,
            operation_id=self.id,
            uid=self._account.uid,
        )

    def _cancel(self):
        """
        Удаляет физическую операцию на защищённом номере.
        """
        super(RemoveSecureOperation, self)._cancel()
        phone = self._phone_manager.by_id(self._remove_secure_operation.phone_id)
        phone.operation = None

    def is_ready_for_quarantine(self, need_authenticated_user=True):
        is_authenticated = self.password_verified or not need_authenticated_user
        return bool(is_authenticated and not self._remove_secure_operation.does_user_admit_phone)

    @property
    def does_user_admit_phone(self):
        return self._remove_secure_operation.does_user_admit_phone


class _SingleHeadOperation(_BaseLogicalOperation):
    name = None
    _type = None
    _is_unique = None
    _should_phone_be_bound = None
    _should_phone_be_secured = None
    _need_phone_confirmation = None
    _need_password_verification = None
    _flags = PhoneOperationFlags()

    @classmethod
    def build_by_data(cls, operation, phone_manager):
        if operation.type != cls._type:
            raise BuildOperationError()
        return (cls(operation, phone_manager, statbox=None), {operation})

    @classmethod
    def create(cls, phone_manager, phone_id, code, statbox, ttl=None):
        phone = phone_manager.by_id(phone_id)

        if cls._should_phone_be_bound:
            _assert_phone_bound(phone)
        if cls._should_phone_be_secured:
            _assert_phone_secured(phone)
        if cls._is_unique:
            security_identity = SECURITY_IDENTITY
        else:
            security_identity = int(phone.number.e164)

        started = datetime.now()

        ttl = ttl or cls._get_ttl()
        finished = started + ttl

        phone.create_operation(
            type=cls._type,
            security_identity=security_identity,
            code_value=code,
            started=started,
            finished=finished,
            flags=cls._flags,
        )
        logical_operation = cls(phone.operation, phone_manager, statbox)

        return logical_operation

    def __init__(self, operation, phone_manager, statbox):
        self._primary_op = operation
        self._secondary_op = None
        self._phone_manager = phone_manager
        self.statbox = statbox

    def _apply(self, need_authenticated_user, timestamp, is_harvester):
        super(_SingleHeadOperation, self)._apply(
            need_authenticated_user=need_authenticated_user,
            timestamp=timestamp,
            is_harvester=is_harvester,
        )
        reason = self.is_inapplicable(need_authenticated_user)
        if reason is not None:
            raise reason

        self._cancel()
        phone = self._phone_manager.by_id(self._primary_op.phone_id)

        self.statbox.stash(
            action=self.statbox.Link(u'phone_operation_applied'),
            number=phone.number.masked_format_for_statbox,
            phone_id=phone.id,
            operation_id=self.id,
            uid=self._account.uid,
        )

    def _cancel(self):
        super(_SingleHeadOperation, self)._cancel()
        phone = self._phone_manager.by_id(self._primary_op.phone_id)
        phone.operation = None

    def is_inapplicable(self, need_authenticated_user=True, is_harvester=False):
        reason = super(_SingleHeadOperation, self).is_inapplicable(
            need_authenticated_user,
            is_harvester,
        )
        if reason:
            return reason

        if self._need_phone_confirmation:
            unconfirmed_phones = self._get_unconfirmed_phones()
        else:
            unconfirmed_phones = []

        need_password_verification = (
            self._need_password_verification and
            need_authenticated_user and
            not self.password_verified
        )

        if unconfirmed_phones or need_password_verification:
            return OperationInapplicable(
                need_password_verification=need_password_verification,
                need_confirmed_phones=unconfirmed_phones,
            )


class MarkOperation(_SingleHeadOperation):
    name = u'mark'
    _type = u'mark'
    _is_unique = False
    _should_phone_be_bound = False
    _should_phone_be_secured = False
    _need_phone_confirmation = False
    _need_password_verification = False

    @classmethod
    def _get_ttl(cls):
        return timedelta(seconds=settings.YASMS_MARK_OPERATION_TTL)


class AliasifySecureOperation(_SingleHeadOperation):
    """
    Логическая операция создания телефонного алиаса.
    """

    name = u'aliasify_secure'
    _type = u'aliasify'
    _is_unique = True
    _should_phone_be_bound = True
    _should_phone_be_secured = True
    _need_phone_confirmation = True
    _need_password_verification = False

    _flags = PhoneOperationFlags()
    _flags.aliasify = True


class DealiasifySecureOperation(_SingleHeadOperation):
    """
    Логическая операция удаления телефонного алиаса.
    """

    name = u'dealiasify_secure'
    _type = u'dealiasify'
    _is_unique = True
    _should_phone_be_bound = True
    _should_phone_be_secured = True
    _need_phone_confirmation = False
    _need_password_verification = True


class Operation(Model):
    """
    Физическая операция суть элементарная операция хранимая в базе данных.

    Физическая операция защищена, если её security_identity ==
    SECURITY_IDENTITY.

    Над одним номером на аккаунте в каждый момент времени может идти только
    одна физическая операция.

    На аккаунте в каждый момент времени может идти только одна защищённая
    операция.
    """

    parent = None

    id = Field('id')
    security_identity = Field('security_identity')
    type = Field('type')
    started = UnixtimeField('started')
    finished = UnixtimeField('finished')
    code_value = Field('code_value')
    code_checks_count = Field('code_checks_count')
    code_send_count = Field('code_send_count')
    code_last_sent = UnixtimeField('code_last_sent')
    code_confirmed = UnixtimeField('code_confirmed')
    password_verified = UnixtimeField('password_verified')
    flags = FlagsField('flags', PhoneOperationFlags)
    phone_id2 = Field('phone_id2')

    def __init__(self, *args, **kwargs):
        super(Operation, self).__init__(*args, **kwargs)
        self._set_default_values()

    def _set_default_values(self):
        for key, value in iteritems(PHONE_OP_DEFAULT_VALUES):
            current_value = getattr(self, key, Undefined)
            if current_value is Undefined:
                if callable(value):
                    value = value()
                setattr(self, key, value)

    def _parse(self, data, fields=None):
        # Ключа operation может не быть (когда операцию не запрашивали), а
        # может быть со значением None (когда операцию запрашивали, но её нет).
        if 'operation' not in data:
            return False, Undefined

        op_data = data['operation']
        if op_data is None:
            return False, None

        # Перед изменением входных данных скопируем их
        op_data = op_data.copy()

        if op_data.get(u'code_value') is None:
            op_data.pop(u'code_value', None)
        if op_data.get(u'phone_id2') is None:
            op_data.pop(u'phone_id2', None)

        parse_response = super(Operation, self)._parse(op_data, fields=fields)
        self._set_default_values()
        return parse_response

    @property
    def phone_id(self):
        return self.parent.id

    @property
    def is_secure(self):
        return self.security_identity == SECURITY_IDENTITY

    @cached_property
    def is_expired(self):
        return self.finished <= datetime.now()

    @property
    def does_user_admit_phone(self):
        """
        Признак того, что номер доступен пользователю.
        """
        return bool(self.code_value)

    def __eq__(self, other):
        if not other:
            return False
        if not isinstance(other, Operation):
            raise NotImplementedError()  # pragma: no cover
        if self.id and other.id:
            return self.id == other.id
        return self is other

    def __ne__(self, other):
        return not (self == other)

    def __repr__(self):
        started = _quote(self.started)
        finished = _quote(self.finished)
        password_verified = _quote(self.password_verified)
        code_confirmed = _quote(self.code_confirmed)

        attrs = [
            ('type', self.type),
            ('security_identity', self.security_identity),
            ('started', started),
            ('finished', finished),
            ('password_verified', password_verified or None),
            ('code_confirmed', code_confirmed or None),
            ('does_user_admit_phone', self.does_user_admit_phone),
            ('flags', int(self.flags)),
            ('phone_id', self.phone_id),
            ('phone_id2', self.phone_id2 or None),
        ]
        attrs = ['%s=%s' % (k, v) for k, v in attrs]
        return '<Operation ' + ' '.join(attrs) + '>'

    def __hash__(self):
        return hash(self.__class__.__name__ + '_' + str(self.id))


class PhoneBinding(Model):
    time = UnixtimeField('binding_time')
    should_ignore_binding_limit = BooleanField('should_ignore_binding_limit')

    def __init__(self, *args, **kwargs):
        self.time = None
        self.should_ignore_binding_limit = False

        super(PhoneBinding, self).__init__(*args, **kwargs)

    def _parse(self, data, fields):
        if 'binding' not in data:
            return False, Undefined
        binding = data['binding']
        if binding is None:
            return True, None
        return super(PhoneBinding, self)._parse(binding, fields)

    @property
    def is_current(self):
        return self.time

    @property
    def is_unbound(self):
        return not self.time


class Phone(Model):
    parent = None

    id = Field('id')

    def _get_number(self):
        return self._number

    def _set_number(self, value):
        self._number = value
        if self.operation and self.operation.security_identity is not SECURITY_IDENTITY:
            self.operation.security_identity = int(value)

    number = DescriptorField(
        Field(partial(parse_phone_number_field, field_name='number')),
        _get_number,
        _set_number,
    )

    created = UnixtimeField('created')
    _bound = UnixtimeField('bound')
    binding = ModelField(PhoneBinding)
    confirmed = UnixtimeField('confirmed')
    admitted = UnixtimeField('admitted')
    secured = UnixtimeField('secured')
    is_bank = BooleanField('is_bank')
    operation = ModelField(Operation)
    should_ignore_binding_limit = Field()

    def _get_bound(self):
        return self._bound

    def _set_bound(self, _datetime):
        self._bound = _datetime

        if self._bound:
            if not self.binding:
                self.binding = PhoneBinding()
            self.binding.time = self._bound
        elif self.binding:
            self.binding = PhoneBinding()

    bound = property(_get_bound, _set_bound)

    @property
    def need_admission(self):
        """
        Признак того, что номер требует переподтверждения.
        Если пользователь указал, что номер ему недоступен,
        не требуем его переподтверждать.
        """
        if self.operation and not self.operation.does_user_admit_phone:
            return False
        if not any([self.confirmed, self.admitted]):
            return False
        elif self.confirmed and self.admitted:
            last_update = max(self.confirmed, self.admitted)
        else:
            last_update = self.confirmed or self.admitted
        return last_update + settings.YASMS_ADMIT_PHONE_VALID_PERIOD <= datetime.now()

    def _parse(self, data, fields=None):
        # Перед изменением входных данных скопируем их
        data = data.copy()

        attributes = data.get('attributes', {})
        for attr_name, value in iteritems(attributes):
            data[attr_name] = value

        return super(Phone, self)._parse(data, fields=fields)

    def create_operation(self, type, security_identity=None, started=None, finished=None,
                         code_value=None, code_checks_count=None, code_send_count=None,
                         code_last_sent=None, code_confirmed=None, password_verified=None,
                         flags=None, phone_id2=None):
        if self.operation:
            # Перед вызовом ручки стоит самостоятельно проверять, что операции еще нет.
            # Поэтому сюда мы попадать не должны вообще.
            raise AttributeError('Operation already exists')  # pragma: no cover

        operation_data = dict(
            type=type,
            security_identity=security_identity,
            started=started or datetime.now(),
            finished=finished,
            code_value=code_value,
            code_checks_count=code_checks_count,
            code_send_count=code_send_count,
            code_last_sent=code_last_sent,
            code_confirmed=code_confirmed,
            password_verified=password_verified,
            flags=flags,
            phone_id2=phone_id2,
        )
        operation_data = remove_none_values(operation_data)

        self.operation = Operation(**operation_data)
        self.operation.parent = self

    def confirm(self, timestamp=None):
        if timestamp and self.confirmed and timestamp <= self.confirmed:
            raise PhoneConfirmedAlready()

        timestamp = timestamp or datetime.now()
        self.confirmed = timestamp

    def set_as_secure(self):
        self.secured = datetime.now()
        self.parent.secure = self

    @property
    def uid(self):
        if not (self.parent and self.parent.parent):
            raise AttributeError('Detached object has no parent, thus does not know UID.')  # pragma: no cover
        return self.parent.parent.uid

    def get_logical_operation(self, statbox):
        if not self.operation:
            return
        phone_manager = self.parent
        logical_op, _ = _build_logical_operation(
            self.operation,
            phone_manager,
        )
        logical_op.statbox = statbox
        return logical_op

    def __eq__(self, other):
        return isinstance(other, Phone) and self.id == other.id

    def __ne__(self, other):
        return not (self == other)

    def __hash__(self):
        return hash(self.__class__.__name__ + '_' + str(self.id))


class Phones(Model):
    parent = None

    default_id = IntegerField('phones.default')
    secure_id = IntegerField('phones.secure')

    _phones = CollectionField('phones', Phone)

    def __init__(self, *args, **kwargs):
        super(Phones, self).__init__(*args, **kwargs)
        self._phones = {}

    def _parse(self, data, fields=None):
        data = _recover_from_errors(data)
        return super(Phones, self)._parse(data, fields)

    @property
    def secure(self):
        return self._phones.get(self.secure_id)

    @secure.setter
    def secure(self, phone):
        if phone is None:
            self.secure_id = None
            return
        if not phone.bound:
            raise ValueError(u"Not bound phone can't be secure")
        if not phone.secured:
            raise ValueError(u"Not secured phone can't be secure")
        self.secure_id = phone.id

    @property
    def default(self):
        """
        Если атрибут phones.default на аккаунте выставлен и телефон с этим
        phone_id у нас есть - возвращаем его.

        Иначе берём привязанный телефон с наибольшим id.
        """
        bound_phones = self.bound()

        if not bound_phones:
            return None

        if self.default_id:
            phone = bound_phones.get(self.default_id)
            if phone:
                return phone

        return bound_phones[max(bound_phones.keys())]

    @default.setter
    def default(self, phone):
        if phone is None:
            self.default_id = None
            return
        self.default_id = phone.id

    def by_id(self, phone_id, assert_exists=True):
        if assert_exists and phone_id not in self._phones:
            raise KeyError(phone_id)

        return self._phones.get(phone_id)

    def has_id(self, phone_id):
        return phone_id in self._phones

    def by_operation_id(self, operation_id):
        if not operation_id:
            raise ValueError(u'Invalid operation id: %s' % operation_id)
        for phone in self._phones.values():
            if phone.operation and phone.operation.id == operation_id:
                return phone

    def by_number(self, number):
        """
        :raise: InvalidPhoneNumber
        """
        if isinstance(number, string_types):
            number = PhoneNumber.parse(number)

        for phone in self._phones.values():
            if phone.number == number:
                return phone

    def has_number(self, number):
        return self.by_number(number) is not None

    def all(self):
        return dict(self._phones)

    def non_secure(self):
        return {k: v for k, v in self._phones.items() if k != self.secure_id}

    def bound(self):
        return {phone_id: phone for phone_id, phone in iteritems(self._phones) if phone.bound}

    def remove(self, key):
        """
        Удаляем телефон, не забывая про дефолтный и защищенный телефон.

        При удалении телефона, который является дефолтным, атрибут default_id
        должен удаляться (смотри default).
        """
        phone_id = key.id if isinstance(key, Phone) else key

        if phone_id not in self._phones:
            raise KeyError(phone_id)  # pragma: no cover

        phone = self._phones[phone_id]
        phone.operation = None

        # Удалим операции в которых телефон участвует как phone_id2.
        for other in self.all().values():
            if other.id == phone_id or not other.operation:
                continue
            if other.operation.phone_id2 == phone_id:
                other.operation = None

        self.unbound_phone(phone)

        # Непосредственное удаление телефона.
        del self._phones[phone_id]

    def unbound_phone(self, phone):
        """
        Отвязать телефон.

        Входные параметры
            phone
                Телефон без операций.
        """
        phone = self.by_id(phone.id)
        if phone.operation:
            raise ValueError(
                u'Невозможно отвязать номер, над которым идёт операция',
            )

        if phone.is_bank:
            raise RemoveBankPhoneNumberError()

        phone.bound = None
        phone.secured = None

        # Подчистим secure и default
        if self.secure_id == phone.id:
            self.secure = None

        if self.default_id == phone.id:
            self.default = None

    def create(self, number, bound=None, confirmed=None, admitted=None,
               secured=None, created=None, operation_data=None,
               existing_phone_id=None):
        """
        Создаем новый телефон. При этом идет запрос в базу и генерируется новый
        phone_id. Заполняем телефон переданными данными и, если передано
        operation_data, создаем операцию.

        Параметр existing_phone_id используется только для тестов.
        """
        if existing_phone_id:
            new_phone_id = existing_phone_id
        else:
            new_phone_id = passport.backend.core.serializers.run_incr_phone_id.run_incr_phone_id.run_incr_phone_id()

        if isinstance(number, string_types):
            number = PhoneNumber.parse(number)

        other_params = {
            'bound': bound,
            'confirmed': confirmed,
            'admitted': admitted,
            'secured': secured,
        }
        binding = PhoneBinding()
        if bound:
            binding.time = bound
        other_params['binding'] = binding
        other_params = remove_none_values(other_params)

        phone = Phone(
            id=new_phone_id,
            parent=self,
            created=created or datetime.now(),
            number=number,
            **other_params
        )
        self._phones[phone.id] = phone

        if operation_data:
            phone.create_operation(**operation_data)

        return phone

    def get_logical_operations(self, statbox):
        phones = self.all()
        operations = []
        while phones:
            _, phone = phones.popitem()
            if phone.operation:
                phones.pop(phone.operation.phone_id2, None)
                operations.append(phone.get_logical_operation(statbox))
        return operations

    def get_secure_logical_operation(self, statbox):
        for logical_op in self.get_logical_operations(statbox):
            if logical_op.is_secure:
                return logical_op


class BaseLogicalOperationError(Exception):
    pass


class OperationInapplicable(BaseLogicalOperationError):
    """
    Нельзя выполнить apply.
    """
    def __init__(self, need_password_verification=False, need_confirmed_phones=None):
        """
        Входные параметры
            need_password_verification
                Требуется ввод пароля.

            need_confirmed_phones
                Требуется подтвердить обладание списком телефонов.
        """
        self.need_password_verification = need_password_verification
        self.need_confirmed_phones = need_confirmed_phones or set()


class SingleSecureOperationError(BaseLogicalOperationError):
    """
    Нельзя выполнять больше одной операции с защищённым номером.
    """


class SecureBindToSimpleBindError(BaseLogicalOperationError):
    """
    Нельзя преобразовать операцию привязки защищённого номера
    (или совместимую) в операцию привязки простого номера.
    """


class SecureNumberBoundAlready(BaseLogicalOperationError):
    """
    Защищённый номер уже привязан.
    """


class NumberBoundAlready(BaseLogicalOperationError):
    """
    Простой номер уже привязан.
    """


class NumberNotBound(BaseLogicalOperationError):
    """
    Номер ещё не привязан.
    """


class NumberNotSecured(BaseLogicalOperationError):
    """
    Номер не защищён.
    """


class BuildOperationError(BaseLogicalOperationError):
    """
    Операцию нельзя построить из таких данных.
    """


class OperationExpired(BaseLogicalOperationError):
    """
    Операция протухла.
    """


class PasswordVerifiedAlready(BaseLogicalOperationError):
    """
    Пользователь уже аутентифицировался.
    """


class PhoneConfirmedAlready(BaseLogicalOperationError):
    """
    Обладание телефоном уже проверено.
    """


class PhoneDoesNotTakePartInOperation(BaseLogicalOperationError):
    """
    Телефон не участвует в данной операции.
    """


class ConfirmationLimitExceeded(BaseLogicalOperationError):
    """
    Превышен лимит на проверку кода для подтверждения телефона.
    """


class OperationExists(BaseLogicalOperationError):
    """
    Над телефоном идёт операция.
    """


class RemoveBankPhoneNumberError(BaseLogicalOperationError):
    """
    Нельзя удалять телефон, указанный как банковский телефон (алиас 25).
    """
