# -*- coding: utf-8 -*-

# Собирательная модель для всего, связанного с подпиской Плюс

import binascii
import json
import logging

from google.protobuf import json_format
from google.protobuf.message import DecodeError
from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    Field,
    IntegerField,
    UnixtimeField,
)
from passport.backend.utils.common import from_base64_standard
from passport.protobuf.plus_subscriber_state.plus_subscriber_state_pb2 import PlusSubscriberState


log = logging.getLogger('passport.models.plus')


def _parse_plus_subscriber_state(data, account, *args):
    """Парсит base64-строку с protobuf-ом из ответа ЧЯ в сериализованный в строку json.

    Именно в таком виде нам удобнее всего в коде работать с этим полем, без конвертации в какую-либо структуру.
    Если будет структура, где есть list из dict-ов, сломается differ. Учитывая это и то, что нам не нужно
    в целом работать с кишками этого поля (только перекладывать туда-сюда), то это самый простой и понятный вариант.
    Поле преобразуется в protobuf только при сериализации в базу. В остальных местах (historydb, statbox, etc)
    работаем просто как со строкой.
    """
    b64_value = data.get('account.plus.subscriber_state')

    if not b64_value:
        return False, None

    try:
        # Декодим со стандартным алфавитом (в ЧЯ соответствующий encode),
        # а также игнорируем отсутствующий padding (на всякий случай).
        proto_value = from_base64_standard(b64_value)
    except binascii.Error as e:
        log.error('Invalid serialized plus_subscriber_state base64 value: %s', e)
        return False, None

    try:
        pss = PlusSubscriberState()
        pss.ParseFromString(proto_value)
    except DecodeError as e:
        log.error('Invalid serialized plus_subscriber_state proto value: %s', e)
        return False, None

    try:
        value = convert_plus_subscriber_state_proto_to_json(pss)
    except ValueError as e:
        log.error('Failed while serializing plus_subscriber_state as json: %s', e)
        return False, None

    return True, value


def convert_plus_subscriber_state_proto_to_json(message):
    try:
        dict_value = json_format.MessageToDict(message, preserving_proto_field_name=True)
    except json_format.Error as e:
        raise ValueError(str(e))

    # Костыль! В официальной документации про конвертацию proto3 в json/dict
    # сказано, что 64-битные числа конвертируются не как числа, а как строки.
    # Мотивация такого решения крайне спорная, все жалобы на это отклоняются,
    # вот пример - https://github.com/protocolbuffers/protobuf/issues/2954
    # Поэтому просто конвертируем самостоятельно конкретные поля в полученном dict-е.
    # В библиотеке mlflow есть более общее решение как референс: https://github.com/mlflow/mlflow/pull/5010
    if dict_value.get('AvailableFeaturesEnd'):
        dict_value['AvailableFeaturesEnd'] = int(dict_value['AvailableFeaturesEnd'])
    if 'AvailableFeatures' in dict_value:
        for feature in dict_value['AvailableFeatures']:
            if feature.get('End'):
                feature['End'] = int(feature['End'])

    try:
        # Компактизируем json в одну строчку без лишних пробелов.
        result = json.dumps(dict_value, sort_keys=True, indent=None, separators=(',', ':'))
    except TypeError as e:
        raise ValueError(str(e))

    return result


class Plus(Model):
    parent = None

    # подписка активна / не активна
    enabled = BooleanField('account.plus.enabled')
    # read-only признак активности подписки, который можно отдавать наружу
    has_plus = BooleanField('account.have_plus')
    # время начала использования триала
    trial_used_ts = UnixtimeField('account.plus.trial_used_ts')
    # время, когда человек остановил подписку
    subscription_stopped_ts = UnixtimeField('account.plus.subscription_stopped_ts')
    # время, когда истечёт подписка
    subscription_expire_ts = UnixtimeField('account.plus.subscription_expire_ts')
    # время следующего списания денег
    next_charge_ts = UnixtimeField('account.plus.next_charge_ts')

    ott_subscription = Field('account.plus.ott_subscription')

    # роль в семье - платит за семью, член семьи, т.п.
    family_role = Field('account.plus.family_role')

    cashback_enabled = BooleanField('account.plus.cashback_enabled')

    # уровень подписки
    subscription_level = IntegerField('account.plus.subscription_level')
    # признак заморозки подписки
    is_frozen = BooleanField('account.plus.is_frozen')

    # текущее состояние плюсового подписчика
    subscriber_state = Field(_parse_plus_subscriber_state)
