# -*- coding: utf-8 -*-
from datetime import datetime

from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    Field,
    UnixtimeField,
)
from passport.backend.core.undefined import Undefined


def _parse_failed_pin_checks_count(data, totp, *args):
    return True, data.get('totp.failed_pin_checks_count', totp.failed_pin_checks_count) or 0


def parse_secret_ids(data, totp, *args):
    raw_secret_ids = data.get('totp.secret_ids')
    if not raw_secret_ids:
        return False, Undefined

    secret_ids = {}
    for item in raw_secret_ids.split(','):
        id_, timestamp = item.split(':', 1)
        secret_ids[int(id_)] = datetime.fromtimestamp(float(timestamp))

    return True, secret_ids


def parse_yakey_device_ids(data, totp, *args):
    raw_device_ids = data.get('account.totp.yakey_device_ids')
    if not raw_device_ids:
        return False, Undefined

    return True, raw_device_ids.split(',')


class TotpSecret(Model):
    """
    Модель секрета TOTP
    """
    parent = None

    totp_secret = Field()

    is_set = BooleanField('2fa_on')  # виртуальный атрибут

    failed_pin_checks_count = Field(_parse_failed_pin_checks_count)

    update_datetime = UnixtimeField('totp.update_datetime')

    # 2фа-секреты пользователя (словарь: id -> время добавления)
    secret_ids = Field(parse_secret_ids)

    # device_id, на которых настроен Ключ
    yakey_device_ids = Field(parse_yakey_device_ids)

    pin_length = Field('totp.pin_length')

    # Не читаем это поле из ЧЯ на модель,
    # т.к. потом именно значение из ЧЯ нужно сохранить в базу
    # т.е. читать будем руками и присваивать перед сохранением.
    check_time = Field()

    def set(self, secret):
        # secret - объект типа TotpSecretType
        self.totp_secret = secret
        self.is_set = True
        self.update_datetime = datetime.now()


class TotpSecretAutoId(Model):
    """
    Генератор последовательных идентификаторов.
    """

    id = Field()
