# -*- coding: utf-8 -*-
from passport.backend.core.models.base import Model
from passport.backend.core.models.base.fields import (
    BooleanField,
    Field,
    IntegerField,
    UnixtimeField,
)
from passport.backend.core.undefined import Undefined
import six


def _parse_external_id(data, instance, *args):
    raw_value = data.get('external_id')
    if not raw_value:
        return False, Undefined

    if six.PY3 and isinstance(raw_value, six.binary_type):
        raw_value = raw_value.decode()

    return True, raw_value


def _parse_public_key(data, instance, *args):
    raw_value = data.get('public_key')
    if not raw_value:
        return False, Undefined

    if six.PY3 and isinstance(raw_value, six.binary_type):
        raw_value = raw_value.decode()

    version, body = raw_value.split(':', 1)
    if version == '1':
        return True, body
    else:
        raise NotImplementedError('Unsupported version: `%s`' % version)


class WebauthnCredential(Model):
    parent = None
    id = IntegerField('id')

    # Глобально-уникальный идентификатор webauthn-секрета
    external_id = Field(_parse_external_id)

    # Публичная часть webauthn-секрета
    public_key = Field(_parse_public_key)

    # https://w3c.github.io/webauthn/#relying-party
    relying_party_id = Field('relying_party_id')

    # Счётчик использований данного секрета
    sign_count = IntegerField('sign_count', default=0)

    device_name = Field('device_name')
    os_family_id = IntegerField('os_family_id')
    browser_id = IntegerField('browser_id')
    is_device_mobile = BooleanField('is_device_mobile')
    is_device_tablet = BooleanField('is_device_tablet')

    created_at = UnixtimeField('created')

    def __repr__(self):
        return '<WebauthnCredential: id=%s, external_id=%s, rp_id=%s, device_name=%s>' % (
            self.id,
            self.external_id,
            self.relying_party_id,
            self.device_name,
        )

    def is_suitable_for_host(self, host):
        if not self.relying_party_id:
            return False
        return (
            host == self.relying_party_id or
            host.endswith('.%s' % self.relying_party_id)
        )


class WebauthnCredentials(Model):
    parent = None
    _creds_by_cred_id = Field()

    def __init__(self, *args, **kwargs):
        super(WebauthnCredentials, self).__init__(*args, **kwargs)
        self._creds_by_cred_id = {}

    def _parse(self, data, fields=None):
        if 'webauthn_credentials' not in data:
            return False, self

        for entry in data.get('webauthn_credentials', []):
            cred = WebauthnCredential().parse(entry)
            self.add(cred)

        return True, self

    def __contains__(self, external_id):
        return external_id in self._creds_by_cred_id

    def __repr__(self):
        return '<WebauthnCredentials: [%s]>' % (
            ', '.join([repr(e) for e in self._creds_by_cred_id.values()]),
        )

    def all(self):
        return sorted(self._creds_by_cred_id.values(), key=lambda item: item.id)

    def add(self, credential):
        credential.parent = credential.parent or self
        self._creds_by_cred_id[credential.external_id] = credential

    def remove(self, external_id):
        self._creds_by_cred_id.pop(external_id)

    def by_external_id(self, external_id):
        return self._creds_by_cred_id.get(external_id)
