# -*- coding: utf-8 -*-
"""
Тесты библиотек.
Используется поиск всех подпоследовательностей строки crtyfghvbnvasyaqwe, используя набор подпоследовательностей,
сгенерированных PasswordQualifier из набора стандартных последовательностей.
Количество попыток поиска - 10000

1) http://pypi.python.org/pypi/ahocorasick/0.9 - старая, неподдерживаемая библиотека, нет поддержки Python 3.
Постоянный Segmentation fault, не разу не удалось запустить поиск

2) http://0x80.pl/proj/pyahocorasick/
Для Python 2 - имплементация только на чистом питоне
Для Python 3 - C extension
Время работы: Python 2 - 0.385066986084
              Python 3 - 0.29024791717529297

3) http://pypi.python.org/pypi/acora/1.7
Очень долгое построение внутренних структур при добавлении большого количества последовательностей (больше тысячи).
В документации прямым текстом сказано:
Note that the current construction algorithm is not suitable for really large sets of keywords (i.e. more than a couple of thousand).
Время работы: 0.0521278381348

4) http://code.google.com/p/esmre/
Время работы: 0.457872867584

5) http://pypi.python.org/pypi/NoAho/0.9.02
Поддержка Python 3.
Для сборки надо использовать как можно более новый gcc. Не собрался под gcc 4.2.1
Время работы findall_long: Python 2 - 0.0457339286804
                           Python 3 - 0.06004905700683594

Так как общая задача при поиске последовательностей, входящих в пароль найти самые длинные,
то 5 вариант кажется наиболее оптимальным по производительности и поддержка 3 питона позволит в далеком будущем легко мигрировать.
"""

from collections import defaultdict
import string

from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.utils.aho_corasick import create_aho_corasick_tree
from six import iteritems


# клавиатурные последовательности
STANDARD_SEQUENCES = [
    {'chars': string.ascii_lowercase, 'min_length': 3, 'max_length': 10, 'description': 'алфавит'},
    {'chars': reversed(string.ascii_lowercase), 'min_length': 3, 'max_length': 10, 'description': 'алфавит'},
    {'chars': string.ascii_uppercase, 'min_length': 3, 'max_length': 10, 'description': 'алфавит'},
    {'chars': reversed(string.ascii_uppercase), 'min_length': 3, 'max_length': 10, 'description': 'алфавит'},
    {'chars': string.digits, 'min_length': 3, 'max_length': 10, 'description': 'цифры'},
    {'chars': reversed(string.digits), 'min_length': 3, 'max_length': 10, 'description': 'цифры'},
    {'chars': 'qwertyuiopasdfghjklzxcvbnm', 'min_length': 3, 'max_length': 10, 'description': 'слева направо, сверху вниз'},
    {'chars': 'mnbvcxzlkjhgfdsapoiuytrewq', 'min_length': 3, 'max_length': 10, 'description': 'справа налево, сверху вниз'},
    {'chars': 'qawsedrftgyhujikolp', 'min_length': 3, 'max_length': 10, 'description': 'верхние два ряда, слева направо, сверху вниз'},
    {'chars': 'azsxdcfvgbhnjmkl', 'min_length': 3, 'max_length': 10, 'description': 'нижние два ряда, слева направо, сверху вниз'},
    {'chars': 'qazwsxedcrfvtgbyhnujmikolp', 'min_length': 3, 'max_length': 10, 'description': 'сверху вниз, слева направо'},
    {'chars': 'qweasdzxcrtyfghvbnuiojkl', 'min_length': 3, 'max_length': 10, 'description': 'слева направо группами по три, сверху вниз'},
    {'chars': '1q2w3e4r5t6y7u8i9o0p', 'min_length': 3, 'max_length': 10, 'description': 'цифры и верхний ряд, слева направо, сверху вниз'},
    {'chars': 'q1w2e3r4t5y6u7i8o9p0', 'min_length': 3, 'max_length': 10, 'description': 'верхний ряд и цифры, слева направо, снизу вверх'},
    {'chars': 'QWERTYUIOPASDFGHJKLZXCVBNM', 'min_length': 3, 'max_length': 10, 'description': 'слева направо, сверху вниз'},
    {'chars': 'MNBVCXZLKJHGFDSAPOIUYTREWQ', 'min_length': 3, 'max_length': 10, 'description': 'справа налево, сверху вниз'},
    {'chars': 'QAWSEDRFTGYHUJIKOLP', 'min_length': 3, 'max_length': 10, 'description': 'верхние два ряда, слева направо, сверху вниз'},
    {'chars': 'AZSXDCFVGBHNJMKL', 'min_length': 3, 'max_length': 10, 'description': 'нижние два ряда, слева направо, сверху вниз'},
    {'chars': 'QAZWSXEDCRFVTGBYHNUJMIKOLP', 'min_length': 3, 'max_length': 10, 'description': 'сверху вниз, слева направо'},
    {'chars': 'QWEASDZXCRTYFGHVBNUIOJKL', 'min_length': 3, 'max_length': 10, 'description': 'слева направо группами по три, сверху вниз'},
    {'chars': '1Q2W3E4R5T6Y7U8I9O0P', 'min_length': 3, 'max_length': 10, 'description': 'цифры и верхний ряд, слева направо, сверху вниз'},
    {'chars': 'Q1W2E3R4T5Y6U7I8O9P0', 'min_length': 3, 'max_length': 10, 'description': 'верхний ряд и цифры, слева направо, снизу вверх'},
]

LOWER_CLASS = 0
UPPER_CLASS = 1
NUM_CLASS = 2
SPEC_CLASS = 3
CLASS_NAMES = ['lower', 'upper', 'numeric', 'special']

# каждый элемент списка означает какому классу символов принадлежит символ в ascii таблице
OCTET_CLASS = []

_class_symbols = [
    [SPEC_CLASS] * 48,    # первые 48 символов - управляющие и спец. символы
    [NUM_CLASS] * 10,    # 10 цифр
    [SPEC_CLASS] * 7,     # ещё 7 спец. символов
    [UPPER_CLASS] * 26,    # 26 букв в верхнем регистре
    [SPEC_CLASS] * 6,     # и ещё 6 спец. символов
    [LOWER_CLASS] * 26,    # 26 букв в нижнем регистре
    [SPEC_CLASS] * 5,     # последние 5 спец. символов
    [SPEC_CLASS] * 128    # и нижняя часть таблицы
]

for class_symbol in _class_symbols:
    OCTET_CLASS.extend(class_symbol)


@lazy_loadable(sequences=STANDARD_SEQUENCES)
class PasswordQualifier(object):
    def __init__(self, sequences, blacklist=None):
        if blacklist is None:
            blacklist = settings.PASSWORD_BLACKLIST
        self.sequences = self._prepare_sequences(sequences)  # последовательности
        self.blacklist = blacklist if isinstance(blacklist, set) else set(blacklist)  # черный список слов (самые популярные пароли)
        self.sequences_scanner = create_aho_corasick_tree(self.sequences)

    def _prepare_sequences(self, sequences):
        words = []
        for sequence in sequences:
            chars = list(sequence['chars'])
            sequence['sequences'] = []

            # Дублируем, чтобы закольцевать последовательность
            # Сделано, для удобства вырезания слова
            # Например, для последовательности qwerty будет chars=qwertyqwerty.
            # Если необходимо вырезать 10 символов начиная с 5, то получится yqwertyqwe:
            # Удобно! не нужно учитывать случаи, когда происходит выход за границы массива в случае отсутствия закольцованности
            # Следует помнить, что если max_length последовательности будет больше длины самой последовательности,
            # то прийдется добавлять еще списки в кольцо
            chars = chars + chars

            # int добавлен для поддержки python 3
            for begin in range(int(len(chars) / 2)):
                for length in range(sequence['min_length'], sequence['max_length'] + 1):
                    end = begin + length
                    word = chars[begin:end]
                    sequence['sequences'].append(word)
                    words.append(word)

        return words

    def get_quality(self, password, words, subwords):
        """
        :param words: @inkvi 04.04.2012 [login]
        :param subwords: @inkvi 04.04.2012 [login]
        """
        def stripped(words):
            for word in words:
                word = word.strip()
                if word:
                    yield word

        additional_bad_words = set(stripped(words))
        additional_bad_subwords = set(stripped(subwords))
        unique_chars = set(password)
        unique_chars_number = len(unique_chars)
        password_length = len(password)

        # {номер класса символов: количество символов класса, использованных в пароле}
        chars_number_by_class_number = defaultdict(int)
        class_switching_number = 0  # количество переключений классов символов
        last_char_class = None  # временная переменая

        for password_char in password:
            char_class = OCTET_CLASS[ord(password_char)]
            chars_number_by_class_number[char_class] += 1
            if last_char_class != char_class:
                class_switching_number += 1
            last_char_class = char_class

        classes_number = len(chars_number_by_class_number.keys())  # количество использованных классов символов

        # {имя класса символов: количество символов класса, использованных в пароле}
        chars_number_by_class_name = dict(
            ((CLASS_NAMES[class_number], number) for class_number, number in iteritems(chars_number_by_class_number))
        )

        sequences = self._find_sequences(password)
        # AhoIterator криво имплементирован: пройтись по контейнеру он может только один раз, поэтому приходится
        # дуплицировать итераторы либо преобразовывать в список
        # https://github.com/JDonner/NoAho/blob/master/noaho/noaho.pyx#L188
        sequences = list(sequences)
        sequences_number = len(sequences)

        # любая из найденных последовательностей полностью совпадает с паролем
        is_password_a_sequence = any(((end - begin) == len(password) for begin, end, seq in sequences))

        # пароль является словом из черного списка
        is_password_in_blacklist = password in self.blacklist

        # пароль является дополнительным словом
        is_password_an_additional_bad_word = password in additional_bad_words

        # количество вхождений дополнительных подслов в пароль
        additional_subwords_inclusions_number = sum(1 for subword in additional_bad_subwords if subword in password)

        # алгоритм подсчета бонусных и штрафных очков подробно описан на http://wiki.yandex-team.ru/passport/passwordstrength/new
        bonus = (
            (len(unique_chars) - 1) +  # вариативность символов
            (classes_number - 1) * 2 +  # количество использованных классов символов
            (password_length > 8) +
            (password_length > 12) +
            (class_switching_number > classes_number) +  # чередование
            1
        )

        penalty = (
            (password_length < 7) +
            (unique_chars_number < 4) +  # вариативность символов
            (sequences_number > 0) +  # вхождение клавиатурных последовательностей
            (100 if is_password_a_sequence else 0) +  # весь пароль – это последовательность
            (100 if is_password_in_blacklist else 0) +  # пароль находится в чёрном списке
            (100 if is_password_an_additional_bad_word else 0) +  # совпадение пароля с одним из дополнительных слов
            (additional_subwords_inclusions_number * 5) +  # вхождение дополнительных подслов в пароль
            1
        )

        real_quality = password_length * bonus // penalty
        quality = real_quality if real_quality < 100 else 100  # обрезаем оценку до 100 баллов
        grade = quality // 20 + 1  # градация делается по 20 пунктов

        return {
            'length': password_length,
            'unique_chars': unique_chars,
            'unique_chars_number': unique_chars_number,
            'chars_number_by_class': chars_number_by_class_name,
            'classes_number': classes_number,
            'class_switching_number': class_switching_number - 1,
            'sequences': sequences,
            'sequences_number': sequences_number,
            'is_sequence': is_password_a_sequence,
            'is_word': is_password_in_blacklist,
            'is_additional_word': is_password_an_additional_bad_word,
            'additional_subwords': additional_bad_subwords,
            'additional_subwords_number': additional_subwords_inclusions_number,
            'bonus': bonus,
            'penalty': penalty,
            'quality': quality,
            'real_quality': real_quality,
            'grade': grade
        }

    def _find_sequences(self, string):
        """
        Найти самые длинные последовательности, входящие в строку
        """
        return self.sequences_scanner.findall_long(string)


def get_password_qualifier():
    return LazyLoader.get_instance('PasswordQualifier')
