# -*- coding: utf-8 -*-

import re

from namedlist import namedlist
from passport.backend.core.builders.blackbox.constants import BLACKBOX_PWDHISTORY_REASON_COMPROMISED
from passport.backend.core.conf import settings
from passport.backend.utils.string import smart_text


PASSWORD_POLICY_NAMES = [
    'basic',
    'scholar',
    'strong',
]


MAX_PASSWORD_LENGTH = 255
# Разрешенные символы: -a-zA-Z0-9`!@#$%^&*()_=+[]{};:"\|,.<>/?
# Нет символов: '~ (PASSP-2170)
PASSWORD_PROHIBITED_SYMBOLS_RE = re.compile(r'[^\x21-\x26\x28-\x7D]')
SCHOLAR_PROHIBITED_SYMBOLS_RE = re.compile(smart_text(r'[^А-я0-9Ёё]'))

rules = [
    'max_length',
    'min_length',
    'min_quality',
    'middle_quality',  # если качество пароля ниже этого порога, выдаём предупреждение (а не ошибку) о его слабости
    'prohibited_symbols_re',
    'search_depth',  # глубина поиска в pwdhistory
    'reason',  # типы паролей для поиска в pwdhistory
]


Policy = namedlist('Policy', rules)


def basic():
    return Policy(
        max_length=MAX_PASSWORD_LENGTH,
        min_length=settings.BASIC_PASSWORD_POLICY_MIN_LENGTH,
        min_quality=settings.BASIC_PASSWORD_POLICY_MIN_QUALITY,
        middle_quality=settings.BASIC_PASSWORD_POLICY_MIDDLE_QUALITY,
        prohibited_symbols_re=PASSWORD_PROHIBITED_SYMBOLS_RE,
        search_depth=settings.BASIC_PASSWORD_POLICY_PWDHISTORY_SEARCH_DEPTH,
        reason=[BLACKBOX_PWDHISTORY_REASON_COMPROMISED],  # поиск по скомпрометированным паролям
    )


def strong():
    return Policy(
        max_length=MAX_PASSWORD_LENGTH,
        min_length=settings.STRONG_PASSWORD_POLICY_MIN_LENGTH,
        min_quality=settings.STRONG_PASSWORD_POLICY_MIN_QUALITY,
        middle_quality=settings.STRONG_PASSWORD_POLICY_MIDDLE_QUALITY,
        prohibited_symbols_re=PASSWORD_PROHIBITED_SYMBOLS_RE,
        search_depth=settings.STRONG_PASSWORD_POLICY_PWDHISTORY_SEARCH_DEPTH,
        reason=None,  # поиск по всем типам паролей в истории
    )


def scholar():
    return Policy(
        max_length=MAX_PASSWORD_LENGTH,
        middle_quality=None,
        min_length=settings.SCHOLAR_PASSWORD_POLICY_MIN_LENGTH,
        min_quality=None,
        prohibited_symbols_re=SCHOLAR_PROHIBITED_SYMBOLS_RE,
        reason=None,
        search_depth=None,
    )


def get_policy(policy_name):
    if policy_name == 'strong':
        return strong()
    elif policy_name == 'scholar':
        return scholar()
    else:
        return basic()
