# -*- coding: utf-8 -*-
import json
import logging

import netaddr
from netaddr import AddrFormatError
from netree import Lookup
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import InvalidIpHeaderError
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.utils.string import smart_str
import six
from six import iteritems
from uatraits import detector


CLIENT_PREFIX = 'Ya-Client-'

log = logging.getLogger('passport.portallib')

USER_FLAG = 0x1
TURBO_FLAG = 0x2


@lazy_loadable()
def IPReg(ipreg_lookup_file=None):
    lookup = Lookup()

    with open(ipreg_lookup_file or settings.IPREG_LOOKUP_FILE) as f:
        data = json.load(f)

    for line in data:
        low = netaddr.IPAddress(line['low'])
        high = netaddr.IPAddress(line['high'])

        is_user = USER_FLAG if line.get('flags', {}).get('user', False) else 0
        is_turbo = TURBO_FLAG if line.get('flags', {}).get('turbo', False) else 0

        mask = int(low) ^ int(high)
        length = mask.bit_length()

        lookup.add(low, length, is_user | is_turbo)
    return lookup


@lazy_loadable()
def uatraits():
    return detector(
        settings.UATRAITS_BROWSER_FILE,
        settings.UATRAITS_PROFILES_FILE,
        settings.UATRAITS_EXTRA_FILE,
    )


def clean_headers(headers):
    cleaned_headers = {}
    if not headers:
        return cleaned_headers
    for header_name, header_value in iteritems(headers):
        if header_name.startswith(CLIENT_PREFIX):
            header_name = header_name[len(CLIENT_PREFIX):]
        # преобразуем в str на всякий случай, т.к. uatraits падает при работе с unicode-строками и None
        cleaned_headers[header_name] = smart_str(header_value or '', errors='replace')
    return cleaned_headers


def get_ipreg():
    return LazyLoader.get_instance('IPReg')


def get_net(ip, headers=None, ipreg=None):
    try:
        ipreg = ipreg or get_ipreg()
        headers = clean_headers(headers)
        net = ipreg.get_net(ip)
        is_turbo = net['flags'] & TURBO_FLAG
        if is_turbo and 'X-Forwarded-For' in headers:
            real_ip = headers['X-Forwarded-For'].split(b',')[0]
            if six.PY3:
                real_ip = real_ip.decode('utf8')
            net = ipreg.get_net(real_ip)
        net['is_user'] = bool(net['flags'] & USER_FLAG)
        return net
    except AddrFormatError as e:
        log.error('Error calling ipreg: %s', e)
        raise InvalidIpHeaderError(str(e))


def get_uatraits():
    return LazyLoader.get_instance('uatraits')


def detect_user_agent(headers, uatraits=None):
    uatraits = uatraits or get_uatraits()
    return uatraits.detect_by_headers(clean_headers(headers))


def is_yandex_ip(ip, ipreg=None):
    """
    Функция определяет, является ли ip яндексовым и входит ли в пользовательские сети.
    Ожидает ip строкой.
    """
    ipreg = ipreg or get_ipreg()
    ip_obj = get_net(ip, ipreg=ipreg)
    return ip_obj['is_yandex'] and ip_obj['is_user']


def is_yandex_server_ip(ip, ipreg=None):
    """
    Функция определяет, является ли ip яндексовым и входит ли в серверные сети.
    Ожидает ip строкой.
    """
    ipreg = ipreg or get_ipreg()
    ip_obj = get_net(ip, ipreg=ipreg)
    return ip_obj['is_yandex'] and not ip_obj['is_user']
