# -*- coding: utf-8 -*-

from functools import partial
import logging

from passport.backend.core.host.host import get_current_host
from passport.backend.core.serializers.eav.serialize import serialize_snapshots
from passport.backend.core.serializers.logs import (
    CryptastatActionRunner,
    PharmaActionRunner,
    StatboxAccountRunner,
    StatboxDomainRunner,
)
from passport.backend.core.serializers.logs.historydb.runner import (
    HistorydbActionRunner,
    HistorydbFamilyActionRunner,
    HistorydbTskvActionRunner,
    HistorydbTskvFamilyActionRunner,
)
from passport.backend.core.serializers.logs.statbox.runner import (
    AccountModificationInfosecRunner,
    AccountModificationRunner,
    StatboxFamilyRunner,
)
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_FEDERAL,
    ACCOUNT_TYPE_KINOPOISK,
    ACCOUNT_TYPE_KOLONKISH,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_UBER,
    ACCOUNT_TYPE_YAMBOT,
)
from passport.backend.core.undefined import Undefined


log = logging.getLogger('passport.processor')


def build_passport_login_rule(is_changing_required):
    return is_changing_required << 2 | 1


def fix_passport_login_rule(old_snapshot, new_snapshot, diff):
    """
    Дополнительное процессирование диффа:
    1) собираем subscription.login_rule.8
       - password.is_changing_required
    """

    # В случае удаления аккаунта / удаления пароля с аккаунта (включения двухфакторной аутентификации),
    # фиксить ничего не нужно, т.к. пароля нет и нет его свойств.
    if new_snapshot is None or new_snapshot.password is None:
        return diff

    # Костыль для случая, когда у пользователя нет sid.8
    # и он не менялся нигде, тогда по идее он и не нужен
    if old_snapshot and all(v == Undefined for v in [old_snapshot.password.forced_changing_reason,
                                                     new_snapshot.password.forced_changing_reason]):
        return diff

    # Если мы создаем аккаунт ПДДшника или же неполноценный аккаунт, то не надо ему прописывать 8-й сид.
    if old_snapshot is None and new_snapshot.type in (
        ACCOUNT_TYPE_PDD,
        ACCOUNT_TYPE_KINOPOISK,
        ACCOUNT_TYPE_UBER,
        ACCOUNT_TYPE_YAMBOT,
        ACCOUNT_TYPE_KOLONKISH,
        ACCOUNT_TYPE_FEDERAL,
    ):
        return diff

    new_login_rule = build_passport_login_rule(new_snapshot.password.is_changing_required)
    diff_part = None
    if old_snapshot is None:
        diff_part = diff.added
    else:
        old_login_rule = build_passport_login_rule(old_snapshot.password.is_changing_required)
        if old_login_rule != new_login_rule:
            diff_part = diff.changed
    if diff_part is None:
        return diff

    subscriptions = diff_part.setdefault('subscriptions', {})
    sid_8 = subscriptions.setdefault(8, {})
    sid_8['login_rule'] = new_login_rule
    return diff


PREPROCESS_HACKS = {
    'Account': fix_passport_login_rule,
}


def run(old_snapshot, new_snapshot, diff, environment, events,
        disable_history_db=False, retries=None, with_low_timeout=False,
        force_history_db_external_events=False, initiator_uid=None, datetime_=None):
    """
    Входные параметры
        initiator_uid
            Идентификатор пользователя, действия этого пользователя привели к
            изменению данной модели.
    """

    model_name = (old_snapshot or new_snapshot).__class__.__name__
    diff_preprocessor = PREPROCESS_HACKS.get(model_name)
    if diff_preprocessor:
        diff = diff_preprocessor(old_snapshot, new_snapshot, diff)

    # Пишем изменения в базу
    run_eav(
        old_snapshot,
        new_snapshot,
        diff,
        retries=retries,
        with_low_timeout=with_low_timeout,
    )

    # Записываем проделанные изменения в логи, но только если это
    # связано с аккаунтом, доменом или семьёй.
    if model_name == 'Account':
        if not disable_history_db:
            run_historydb(
                old_snapshot,
                new_snapshot,
                diff,
                environment,
                events,
                get_current_host().get_id(),
                force_external_events=force_history_db_external_events,
                initiator_uid=initiator_uid,
                datetime_=datetime_,
            )
        run_statbox(old_snapshot, new_snapshot, diff, environment, events)
        run_cryptastat(old_snapshot, new_snapshot, diff, environment, events)
        run_pharma(old_snapshot, new_snapshot, diff, environment, events)
    elif model_name == 'Domain':
        run_statbox(old_snapshot, new_snapshot, diff, environment, events)
    elif model_name == 'FamilyInfo':
        if not disable_history_db:
            run_historydb(
                old_snapshot,
                new_snapshot,
                diff,
                environment,
                events,
                get_current_host().get_id(),
                force_external_events=force_history_db_external_events,
                initiator_uid=initiator_uid,
                datetime_=datetime_,
            )
        run_statbox(old_snapshot, new_snapshot, diff, environment, events)


def run_eav(old_snapshot, new_snapshot, diff, retries=None, with_low_timeout=False):
    serialize_snapshots(old_snapshot, new_snapshot, diff, retries=retries, with_low_timeout=with_low_timeout)


def run_historydb(old_snapshot, new_snapshot, diff, environment, events,
                  host_id, force_external_events=False, initiator_uid=None,
                  datetime_=None):
    snapshot = new_snapshot or old_snapshot
    model_name = snapshot.__class__.__name__
    # events - dict может мутировть внутри первого раннера
    tskv_events = events.copy()
    if model_name == 'FamilyInfo':
        runner = HistorydbFamilyActionRunner(
            events=events,
            user_ip=environment.user_ip,
            user_agent=environment.user_agent,
            yandexuid=environment.cookies.get('yandexuid'),
            uid=None,
            host_id=host_id,
            force_external_events=force_external_events,
            datetime_=datetime_,
            initiator_uid=initiator_uid,
        )
        tskv_runner = HistorydbTskvFamilyActionRunner(
            events=tskv_events,
            user_ip=environment.user_ip,
            user_agent=environment.user_agent,
            yandexuid=environment.cookies.get('yandexuid'),
            uid=None,
            host_id=host_id,
            force_external_events=force_external_events,
            datetime_=datetime_,
            initiator_uid=initiator_uid,
        )
    else:
        runner = HistorydbActionRunner(
            events=events,
            uid=snapshot.uid,
            user_ip=environment.user_ip,
            user_agent=environment.user_agent,
            yandexuid=environment.cookies.get('yandexuid'),
            host_id=host_id,
            force_external_events=force_external_events,
            datetime_=datetime_,
            initiator_uid=initiator_uid,
        )
        tskv_runner = HistorydbTskvActionRunner(
            events=tskv_events,
            user_ip=environment.user_ip,
            user_agent=environment.user_agent,
            yandexuid=environment.cookies.get('yandexuid'),
            uid=snapshot.uid,
            host_id=host_id,
            force_external_events=force_external_events,
            datetime_=datetime_,
            initiator_uid=initiator_uid,
        )
    # TSV формат один эвент это одна запись части одной атомарной операции
    # Пример: операция delete_foo
    # 1 2022-06-27T11:10:4 bar 4096637325 unbind_child child_id metadata - - -
    # 1 2022-06-27T11:10:4 bar 4096637325 delete_child child_id metadata - - -
    # 1 2022-06-27T11:10:4 bar 4096637325 delete_foo foo_id metadata - - -
    for event in runner.serialize(old_snapshot, new_snapshot, diff):
        runner.execute(event)
    # TSKV формат один эвент это полная запись об одной атомарной операции Пример: операция delete_foo version=1
    # time=2022-06-27T11:10:4 service=bar unixtime=4096637325 metadata unbind_child=child_id  delete_child=child_id delete_foo=foo_id
    for event in tskv_runner.serialize(old_snapshot, new_snapshot, diff):
        tskv_runner.execute(event)


def run_statbox(old_snapshot, new_snapshot, diff, environment, events,
                runner_cls=None):
    snapshot = old_snapshot or new_snapshot
    extra_params = {}
    model_name = snapshot.__class__.__name__

    log_account_modification = False
    if model_name == 'Account':
        runner_cls = runner_cls or StatboxAccountRunner
        log_account_modification = True
        extra_params['uid'] = snapshot.uid
    elif model_name == 'Domain':
        runner_cls = runner_cls or StatboxDomainRunner
        extra_params['domain_id'] = snapshot.id
    elif model_name == 'FamilyInfo':
        runner_cls = runner_cls or StatboxFamilyRunner
        extra_params['family_id'] = snapshot.family_id

    if not runner_cls:
        raise ValueError('run_statbox: runner_cls is not specified for model %r' % (model_name,))

    runners = [
        runner_cls(
            events=events,
            user_ip=environment.user_ip,
            user_agent=environment.user_agent,
            **extra_params
        )
    ]

    if log_account_modification:
        runners.append(
            AccountModificationRunner(
                events=events,
                user_ip=environment.user_ip,
                user_agent=environment.user_agent,
                **extra_params
            ),
        )
        runners.append(
            AccountModificationInfosecRunner(
                events=events,
                user_ip=environment.user_ip,
                user_agent=environment.user_agent,
                **extra_params
            ),
        )

    for runner in runners:
        for event in runner.serialize(old_snapshot, new_snapshot, diff, events['action']):
            runner.execute(event)


run_cryptastat = partial(run_statbox, runner_cls=CryptastatActionRunner)
run_pharma = partial(run_statbox, runner_cls=PharmaActionRunner)
