# -*- coding: utf-8 -*-
import copy
import logging

from google.protobuf.message import (
    DecodeError,
    EncodeError,
)
from google.protobuf.pyext._message import RepeatedCompositeContainer
from passport.backend.core.protobuf.exceptions import (
    CorruptDumpError,
    SerializationError,
)
from passport.backend.core.protobuf.fields import ProtobufList
from passport.backend.utils.string import smart_bytes
from six import iteritems
from six.moves import zip_longest


try:
    import collections.abc as collections_abc
except ImportError:
    import collections as collections_abc


log = logging.getLogger('passport.protobuf')


class ProtobufSerializableEntry(object):
    """
    Метакласс добавляет/изменяет методы:
     * load  classmethod для десерилизации
     * dump  для сериализации
     * __eq__  при сравнении проверяются все protobuf-поля
     * __init__  конструктор принимает все protobuf-поля как kwargs

    Целевой python-класс может содержать какую-либо логику и может быть
    сериализован/десериализован через класс-контейнер из поля __protobuf__
    """

    def __init__(self, *args, **kwargs):
        attributes = {el: getattr(self, el) for el in dir(self)}

        for field_name, pb2_field in iteritems(self.get_protobuf_fields()):
            if field_name in kwargs:
                setattr(self, field_name, kwargs[field_name])
            elif field_name in attributes:
                setattr(self, field_name, copy.deepcopy(attributes[field_name]))

    def dump(self):
        pb2 = self.serialize_entry()
        try:
            return pb2.SerializeToString()
        except EncodeError as e:
            raise SerializationError(e)

    @classmethod
    def load(cls, data):
        pb2 = cls.__protobuf__()
        try:
            pb2.ParseFromString(smart_bytes(data))
        except DecodeError as e:
            raise CorruptDumpError(e)

        if hasattr(pb2, 'version') and hasattr(cls, 'version'):
            if pb2.version != cls.version:
                raise ValueError(
                    'Class %s expects entry version %d, but found %d' %
                    (cls, cls.version, pb2.version)
                )

        # Инициализация целевого объекта
        kwargs = cls.deserialize_entry(pb2)
        return cls(**kwargs)

    def serialize_entry(self):
        """Сохраняем атрибуты объекта в protobuf-контейнер для дальнейшей сериализации"""
        pb2 = self.__protobuf__()
        for field_name in self.get_protobuf_fields():
            field = getattr(pb2, field_name)
            value = getattr(self, field_name)

            if type(value) in (list, set, tuple):
                self.serialize_scalar_list(field, value, field_name)

            elif isinstance(value, ProtobufList):
                value.serialize_composite_list(field)

            else:
                try:
                    if value is not None:
                        setattr(pb2, field_name, value)
                except TypeError:
                    raise SerializationError('Field %s has wrong type value %r' % (field_name, value))

        return pb2

    @classmethod
    def deserialize_entry(cls, pb2):
        """Вынимаем данные из protobuf-объекта в набор полей для передачи конструктору объекта"""
        kwargs = {}
        for field_name in cls.get_protobuf_fields():
            value = getattr(pb2, field_name)
            field = getattr(cls, field_name, None)

            if isinstance(value, RepeatedCompositeContainer):
                if field.get_entry_class() in (int, bool, float, str):
                    value = list(value)
                else:
                    collection = copy.deepcopy(field)
                    collection_class = collection.__class__
                    value = collection_class.deserialize_composite_list(value, field.entry_class)

            kwargs[field_name] = value

        return kwargs

    @classmethod
    def get_protobuf_fields(cls):
        return cls.__protobuf__.DESCRIPTOR.fields_by_name

    def __eq__(self, other):
        for field_name in self.get_protobuf_fields():
            our = getattr(self, field_name)
            their = getattr(other, field_name)
            if isinstance(our, collections_abc.Iterable) and isinstance(their, collections_abc.Iterable):
                for our_el, their_el in zip_longest(our, their):
                    if not our_el.__eq__(their_el):
                        return False
            elif their != our:
                return False
        return True

    def serialize_scalar_list(self, pb2_list_field, list_value, field_name):
        """Записываем все элементы списка в поле protobuf-контейнер"""
        for item in list_value:
            try:
                pb2_list_field.append(item)
            except TypeError:
                raise SerializationError('List field %s has wrong type value %r' % (field_name, list_value))
