# -*- coding: utf-8 -*-
try:
    import collections.abc as collections_abc
except ImportError:
    import collections as collections_abc


class ProtobufList(collections_abc.Iterable):

    """Все generic-методы для коллекции"""

    def __getitem__(self, item):
        return self._collection[item]

    def __iter__(self):
        return iter(self._collection)

    def __len__(self):
        return len(self._collection)

    def __nonzero__(self):
        return bool(self._collection)

    def __eq__(self, other):
        return (
            type(self) == type(other) and
            self._collection == other._collection
        )

    def __ne__(self, other):
        return not self.__eq__(other)

    def from_iterable(self, iterable):
        self._collection = list(iterable)

    def append(self, value):
        self._collection.append(value)

    def serialize_composite_list(self, pb2_field):
        for entry in self._collection:
            pb2_entry = entry.serialize_entry()
            new_entry = pb2_field.add()
            new_entry.CopyFrom(pb2_entry)

    @classmethod
    def deserialize_composite_list(cls, list_field, entry_class):
        result = list()

        for pb2_entry in list_field:
            kwargs = entry_class.deserialize_entry(pb2_entry)
            entry = entry_class(**kwargs)
            result.append(entry)

        r = cls(entry_class)
        r.from_iterable(result)
        return r

    def __init__(self, entry_class):
        self._collection = list()
        self.entry_class = entry_class

    def get_entry_class(self):
        return self.entry_class

    def clear(self):
        self._collection = list()

    def remove(self, value):
        return self._collection.remove(value)


__all__ = (
    'ProtobufList',
)
