# -*- coding: utf-8 -*-
from collections import (
    defaultdict,
    namedtuple,
)
from copy import copy
import logging
from time import (
    sleep,
    time,
)

from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.host.host import is_host_current
from passport.backend.core.logging_utils.loggers import (
    FAILED_RESPONSE_CODE,
    GraphiteLogger,
    SUCCESS_RESPONSE_CODE,
)
import redis
import six


_redis = namedtuple('redis_holder', 'redis retry_timeout retries')

log = logging.getLogger('passport.tracks.RedisManager')


class BaseRedisError(BaseCoreError):
    """Базовая обертка для ошибок redis"""


class RedisError(BaseRedisError):
    """Самая общая ошибка Redis"""


class RedisWatchError(RedisError):
    """Наблюдаемый ключ изменился"""


class RedisAuthenticationError(RedisError):
    """Все имеющиеся пароли не работают"""


class RedisRecoverableAuthenticationError(RedisError):
    """Пароль неправильный, но есть еще пароли, и наверное стоит попробовать еще раз"""


class RedisWithSeveralPasswords:
    def __init__(self, passwords, redis_constructor, **kwargs):
        self._kwargs = kwargs
        self._backup_passwords = passwords
        self._redis_constructor = redis_constructor
        self.next_password()

    def __getattr__(self, name):
        return getattr(self._redis, name)

    def next_password(self):
        try:
            self._redis = self._redis_constructor(
                password=self._backup_passwords.pop(0),
                **self._kwargs
            )
        except IndexError:
            raise RedisAuthenticationError()


class RedisManager(object):

    def __init__(self, redis_constructor=None, graphite_logger=None):
        self.configured = False
        self.redis_constructor = redis_constructor or redis.Redis

        self.master = None
        self.slaves = []

        self.graphite_logger = graphite_logger or GraphiteLogger(service='redis')

    def configure(self, config):
        if not config:
            raise ValueError('Servers configuration can\'t be empty')

        for key, config in config.items():
            if config.get('type', 'master') == 'master':
                self.master = self._redis_from_config(config)
            else:
                self.slaves.append(self._redis_from_config(config))

        if not self.master:
            log.info('no master config')
        if not self.slaves:
            log.info('no slave configs')

        self.configured = True

    def _redis_from_config(self, cfg):
        host = cfg.get('host')
        port = cfg.get('port')
        passwords = cfg.get('passwords') or [None]
        socket_timeout = cfg.get('socket_timeout')
        retry_timeout = cfg.get('retry_timeout', settings.REDIS_RETRY_TIMEOUT)
        retries = cfg.get('retries', settings.REDIS_RETRY_MAX_COUNT)
        use_tls = cfg.get('use_tls', False)
        return _redis(
            RedisWithSeveralPasswords(
                host=host,
                port=port,
                socket_timeout=socket_timeout,
                passwords=passwords,
                ssl=use_tls,
                redis_constructor=self.redis_constructor,
            ),
            retry_timeout,
            retries,
        )

    def get_rw_connection(self):
        return self.master

    def get_ro_connection(self):
        if len(self.slaves) > 0:
            # TODO: round-robin or random
            return self.slaves[0]
        return self.get_rw_connection()

    def _execute(self, redis_node, operation, connection=None):
        retries = redis_node.retries
        retry_timeout = redis_node.retry_timeout
        network_error = False
        response_code = SUCCESS_RESPONSE_CODE
        connection_info = redis_node.redis.connection_pool.connection_kwargs
        host_name = connection_info['host']
        caught_exception = None
        for i in six.moves.range(retries):
            # Костыль: используем copy, чтобы иметь возможность вызывать один и тот же pipe несколько раз подряд
            conn = copy(connection) if connection is not None else redis_node.redis
            start_time = time()
            try:
                return operation(conn)
            except redis.WatchError as e:
                caught_exception = e
                response_code = FAILED_RESPONSE_CODE
                log.warning('Redis watch error: host %s:%s "%s"', host_name, connection_info['port'], caught_exception)
                raise RedisWatchError(caught_exception)  # ретраиться бессмысленно
            except redis.RedisError as e:
                caught_exception = e
                response_code = FAILED_RESPONSE_CODE
                log.warning('Redis error: host %s:%s "%s"', host_name, connection_info['port'], caught_exception)
                log.debug('Retries remaining: %d', retries - i - 1)
                if isinstance(e, redis.ResponseError) and 'WRONGPASS' in str(e):
                    redis_node.redis.next_password()
                    if connection is not None:
                        # если мы сейчас внутри пайплайна, то восстановиться мы не можем
                        raise RedisRecoverableAuthenticationError()
                else:
                    network_error = True
                    if retries - i - 1 > 0:
                        sleep(retry_timeout)
            finally:
                if not is_host_current(host_name):
                    # Т.к. локальные походы в Redis засоряют журнал, будем
                    # писать в него только походы в неместные службы Redis'а.
                    self.graphite_logger.log(
                        duration=time() - start_time,
                        response=response_code,
                        network_error=network_error,
                        srv_hostname=host_name,
                        srv_ipaddress=host_name,
                        retries_left=retries - i - 1,
                    )
        log.error('Redis call failed (host %s:%s): "%s"', host_name, connection_info['port'], caught_exception)
        raise RedisError(caught_exception)

    def hmset(self, key, data):
        log.debug("hmset called for key: '%s'", key)
        return self._execute(self.get_rw_connection(), lambda conn: conn.hmset(key, data))

    def hgetall(self, key):
        log.debug("hgetall called for key: '%s'", key)
        return self._execute(self.get_ro_connection(), lambda conn: conn.hgetall(key))

    def hset(self, key, field, value):
        log.debug("hset called for key '%s.%s'", key, field)
        return self._execute(self.get_rw_connection(), lambda conn: conn.hset(key, field, value))

    def hget(self, key, field):
        log.debug("hget called for key: '%s.%s'", key, field)
        return self._execute(self.get_ro_connection(), lambda conn: conn.hget(key, field))

    def hdel(self, key, *fields):
        log.debug("hdel called for key '%s and fields %s'", key, fields)
        return self._execute(self.get_rw_connection(), lambda conn: conn.hdel(key, *fields))

    def get(self, key):
        log.debug("get called for key '%s'", key)
        return self._execute(self.get_ro_connection(), lambda conn: conn.get(key))

    def mget(self, keys):
        log.debug("mget called for keys: '%s'", keys)
        return self._execute(self.get_ro_connection(), lambda conn: conn.mget(keys))

    def set(self, key, value):
        log.debug("set: %s=%s", key, value)
        return self._execute(self.get_rw_connection(), lambda conn: conn.set(key, value))

    def exists(self, key):
        log.debug("exists called for key: '%s'", key)
        return self._execute(self.get_ro_connection(), lambda conn: conn.exists(key))

    def expire(self, key, time):
        log.debug("expire called for key: '%s'", key)
        return self._execute(self.get_rw_connection(), lambda conn: conn.expire(key, time))

    def delete(self, *keys):
        log.debug("delete called for keys: '%s'", keys)
        return self._execute(self.get_rw_connection(), lambda conn: conn.delete(*keys))

    def incr(self, key):
        log.debug("incr called for key '%s'", key)
        return self._execute(self.get_rw_connection(), lambda conn: conn.incr(key))

    def hincrby(self, key, field, value):
        log.debug("hincr called for key '%s.%s'", key, field)
        return self._execute(self.get_rw_connection(), lambda conn: conn.hincrby(key, field, value))

    def rpush(self, key, *values):
        log.debug("rpush called for key '%s'", key)
        return self._execute(self.get_rw_connection(), lambda conn: conn.rpush(key, *values))

    def lpush(self, key, *values):
        log.debug("lpush called for key '%s'", key)
        return self._execute(self.get_rw_connection(), lambda conn: conn.lpush(key, *values))

    def lrange(self, key, start, end):
        log.debug("lrange called for key '%s'", key)
        return self._execute(self.get_ro_connection(), lambda conn: conn.lrange(key, start, end))

    def llen(self, key):
        log.debug("llen called for key '%s'", key)
        return self._execute(self.get_ro_connection(), lambda conn: conn.llen(key))

    def ltrim(self, key, start, end):
        log.debug("ltrim called for key '%s'", key)
        return self._execute(self.get_rw_connection(), lambda conn: conn.ltrim(key, start, end))

    def ttl(self, key):
        log.debug("ttl called for key '%s'", key)
        return self._execute(self.get_ro_connection(), lambda conn: conn.ttl(key))

    def ping(self):
        return self._execute(self.get_rw_connection(), lambda conn: conn.ping())

    def pipeline(self, readonly=False):
        log.debug('pipeline created')
        redis_node = self.get_ro_connection() if readonly else self.get_rw_connection()
        return PipelineProxy(pipe=redis_node.redis.pipeline(), parent=self, redis_node=redis_node)


class PipelineMethodProxy(object):
    """Обёртка над методом объекта pipeline, логирующая выполняемую команду"""
    def __init__(self, method):
        self.method = method

    def __call__(self, *args):
        log.debug('pipeline.%s called with args: %s', self.method.__name__, args)
        return self.method(*args)


class PipelineProxy(object):
    """Обёртка над pipeline, умеющая правильно его исполнять"""
    def __init__(self, pipe, parent, redis_node):
        self.redis_node = redis_node
        self.pipe = pipe
        self.parent = parent

    def execute(self):
        log.debug('pipeline executed')
        return self.parent._execute(self.redis_node, lambda pipe: pipe.execute(), self.pipe)

    def discard(self):
        log.debug('pipeline discarded')
        return self.parent._execute(self.redis_node, lambda pipe: pipe.discard(), self.pipe)

    def watch(self, *names):
        return self.parent._execute(self.redis_node, lambda pipe: pipe.watch(*names), self.pipe)

    def __getattr__(self, item):
        method = getattr(self.pipe, item, None)
        if method is not None:
            return PipelineMethodProxy(method)


_redises = defaultdict(RedisManager)


def get_redis_mgr(name):
    return _redises[name]


def get_redis_mgrs():
    return _redises
