# -*- coding: utf-8 -*-

from contextlib import contextmanager

from passport.backend.core.differ.differ import diff
from passport.backend.core.processor import run
from six.moves import zip


@contextmanager
def UPDATE(instances, environment, events, disable_history_db=False,
           retries=None, with_low_timeout=False,
           force_history_db_external_events=False, initiator_uid=None,
           datetime_=None, allow_nested=False):
    if not isinstance(instances, list):
        instances = [instances]

    with _manage_nested_updates(instances, allow_nested) as instances:
        snapshots = [i.snapshot() for i in instances]

        yield

        for s1, instance in zip(snapshots, instances):
            difference = diff(s1, instance)
            run(
                old_snapshot=s1,
                new_snapshot=instance,
                diff=difference,
                environment=environment,
                events=events,
                disable_history_db=disable_history_db,
                retries=retries,
                with_low_timeout=with_low_timeout,
                force_history_db_external_events=force_history_db_external_events,
                initiator_uid=initiator_uid,
                datetime_=datetime_,
            )


@contextmanager
def CREATE(model, environment, events, disable_history_db=False, retries=None,
           with_low_timeout=False, datetime_=None, allow_nested=False):
    models = [model]
    with _manage_nested_updates(models, allow_nested) as models:
        yield model
        if models:
            run(
                old_snapshot=None,
                new_snapshot=models[0],
                diff=diff(None, models[0]),
                environment=environment,
                events=events,
                disable_history_db=disable_history_db,
                retries=retries,
                with_low_timeout=with_low_timeout,
                datetime_=datetime_,
            )


@contextmanager
def DELETE(model, environment, events, disable_history_db=False, retries=None,
           with_low_timeout=False, allow_nested=False):
    models = [model]
    with _manage_nested_updates(models, allow_nested) as models:
        yield model
        if models:
            run(
                old_snapshot=models[0],
                new_snapshot=None,
                diff=diff(models[0], None),
                environment=environment,
                events=events,
                disable_history_db=disable_history_db,
                retries=retries,
                with_low_timeout=with_low_timeout,
            )


@contextmanager
def _manage_nested_updates(instances, allow_nested):
    instances_to_manage = []
    try:
        for instance in instances:
            if id(instance) not in _managed_instances:
                _managed_instances.add(id(instance))
                instances_to_manage.append(instance)
            elif not allow_nested:
                raise RuntimeError('Nested transactions are prohibited')
        yield instances_to_manage
    finally:
        for instance in instances_to_manage:
            _managed_instances.remove(id(instance))


_managed_instances = set()
