# -*- coding: utf-8 -*-
from functools import wraps
import logging
import time

import boto3
from boto3.exceptions import Boto3Error
from botocore.config import Config
import botocore.exceptions
from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.s3.exceptions import (
    FileNotFoundError,
    S3PermanentError,
    UnseekableStreamError,
)


DELIMITER = '/'


log = logging.getLogger('passport.s3')


def boto_wrapper(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        start_time = time.time()
        status = 'ok'
        response_code = 'success'
        try:
            return func(*args, **kwargs)
        except (Boto3Error, botocore.exceptions.BotoCoreError, botocore.exceptions.ClientError) as e:
            status = str(e)
            response_code = 'failed'
            error_message = '%s: %s' % (e.__class__, status)
            log.error(error_message)
            raise S3PermanentError(error_message)
        finally:
            graphite_logger = GraphiteLogger(service='s3')
            graphite_logger.log(
                duration=time.time() - start_time,
                response=response_code,
                status=status,
            )
    return wrapper


@lazy_loadable()
class S3Client(object):
    """
    https://wiki.yandex-team.ru/mds/s3-api
    https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/s3.html#S3.Client
    """
    def __init__(self, bucket_name=None, endpoint_url=None, access_key_id=None, access_key=None, ca_cert=None,
                 connect_timeout=None, read_timeout=None, retries=None):
        super(S3Client, self).__init__()
        self._boto_client = boto3.client(
            service_name='s3',
            endpoint_url=endpoint_url or settings.S3_ENDPOINT,
            aws_access_key_id=access_key_id or settings.S3_SECRET_KEY_ID,
            aws_secret_access_key=access_key or settings.S3_SECRET_KEY,
            config=Config(
                connect_timeout=connect_timeout or settings.S3_CONNECT_TIMEOUT,
                read_timeout=read_timeout or settings.S3_READ_TIMEOUT,
                retries=dict(
                    max_attempts=retries or settings.S3_RETRIES,
                ),
                s3=dict(
                    addressing_style='virtual',
                ),
            ),
            verify=ca_cert or settings.SSL_CA_CERT,
        )
        self._bucket_name = bucket_name or settings.S3_TAKEOUT_BUCKET_NAME

    @property
    def _client(self):
        # промежуточный геттер нужен для тестов
        return self._boto_client

    def _assert_status(self, rv, status_code):
        actual_status_code = rv.get('ResponseMetadata', {}).get('HTTPStatusCode')
        if actual_status_code != status_code:
            raise S3PermanentError('Unexpected status=%s: %s' % (actual_status_code, rv))

    @boto_wrapper
    def file_exists(self, key):
        try:
            self._client.head_object(
                Bucket=self._bucket_name,
                Key=key,
            )
            return True
        except botocore.exceptions.ClientError as e:
            if e.response.get('Error', {}).get('Code') == '404':
                return False
            raise

    @boto_wrapper
    def read_file(self, key):
        try:
            return self._client.get_object(
                Bucket=self._bucket_name,
                Key=key,
            )['Body'].read()
        except botocore.exceptions.ClientError as e:
            if e.response.get('Error', {}).get('Code') == 'NoSuchKey':
                raise FileNotFoundError(key)
            raise

    @boto_wrapper
    def download_fileobj(self, key, fileobj):
        try:
            return self._client.download_fileobj(
                Bucket=self._bucket_name,
                Key=key,
                Fileobj=fileobj,
            )
        except botocore.exceptions.ClientError as e:
            if e.response.get('Error', {}).get('Code') == '404':
                raise FileNotFoundError(key)
            raise

    @boto_wrapper
    def upload_file(self, local_filepath, dst_folder, dst_filename):
        key = DELIMITER.join([
            dst_folder.strip(DELIMITER),
            dst_filename.strip(DELIMITER),
        ])
        with open(local_filepath, 'rb') as f:
            self._client.put_object(
                Body=f,
                Bucket=self._bucket_name,
                Key=key,
            )
        return key

    @boto_wrapper
    def upload_fileobj(
        self,
        file_object,
        dst_folder,
        dst_filename,
        skip_integrity_check=False,
    ):
        key = DELIMITER.join([
            dst_folder.strip(DELIMITER),
            dst_filename.strip(DELIMITER),
        ])
        kwargs = dict(
            Body=file_object,
            Bucket=self._bucket_name,
            Key=key,
        )
        if skip_integrity_check:
            kwargs.update(ContentMD5='')

        try:
            self._client.put_object(**kwargs)
        except botocore.exceptions.UnseekableStreamError:
            raise UnseekableStreamError()

        return key

    @boto_wrapper
    def list_files(self, folder, limit, last_seen_key=None):
        if folder:
            prefix = folder.strip(DELIMITER) + DELIMITER
        else:
            prefix = folder or ''
        rv = self._client.list_objects(
            Bucket=self._bucket_name,
            Prefix=prefix,
            MaxKeys=limit,
            Marker=last_seen_key or '',
        )
        self._assert_status(rv, 200)
        return rv.get('Contents', [])

    def iterate_files_by_chunks(self, folder, chunk_size):
        last_seen_key = None
        while True:
            files = self.list_files(folder=folder, limit=chunk_size, last_seen_key=last_seen_key)
            if not files:
                return
            yield files
            last_seen_key = files[-1]['Key']

    @boto_wrapper
    def delete_file(self, key):
        rv = self._client.delete_object(
            Bucket=self._bucket_name,
            Key=key,
        )
        self._assert_status(rv, 204)

    @boto_wrapper
    def generate_presigned_url_for_file(self, key, expires_in=10, content_type='application/x-gzip'):
        """
        Формирует короткоживущую ссылку для скачивания файла
        :param key: ключ файла в MDS S3
        :param expires_in: время жизни ссылки (в секундах)
        :param content_type: Content-Type, с которым файл будет отдаваться по ссылке
        :return: короткоживущая ссылка
        """
        return self._client.generate_presigned_url(
            ClientMethod='get_object',
            Params=dict(
                Bucket=self._bucket_name,
                Key=key,
                ResponseContentType=content_type,
            ),
            ExpiresIn=expires_in,
        )


def get_s3_client():
    return LazyLoader.get_instance('S3Client')
