# -*- coding: utf-8 -*-
import mock
from passport.backend.core.s3.client import S3Client
from passport.backend.core.test.test_utils import single_entrant_patch


class FakeNoSuchKeyError(Exception):
    pass


def s3_ok_response(status=200, bucket_name='test-bucket', **extra_data):
    return dict(
        Name=bucket_name,
        ResponseMetadata=dict(
            HTTPStatusCode=status,
        ),
        **extra_data
    )


@single_entrant_patch
class FakeS3Client(object):
    def __init__(self):
        self._boto_mock = mock.Mock()
        # В boto местами используется странный механизм работы с исключениями: исключения, которые надо ловить,
        # хранятся прямо на объекте клиента. Поэтому положим какие-то исключения и на мок клиента.
        # (Технически тут могут использоваться любые классы исключений)
        self._boto_mock.exceptions.NoSuchKey = FakeNoSuchKeyError
        self._patch = mock.patch.object(
            S3Client,
            '_client',
            new_callable=mock.PropertyMock,
            return_value=self._boto_mock,
        )

    def _method_mock(self, method):
        return getattr(self._boto_mock, method)

    def set_response_value(self, method, response_value):
        self._method_mock(method).return_value = response_value
        # Нельзя оставлять установленный side_effect: он бы имел приоритет над return_value
        self._method_mock(method).side_effect = None

    def set_response_side_effect(self, method, side_effect):
        self._method_mock(method).side_effect = side_effect
        self._method_mock(method).return_value = None

    def calls_by_method(self, method):
        return [
            call[1]  # kwargs
            for call in self._method_mock(method).call_args_list
        ]

    def start(self):
        self._patch.start()

    def stop(self):
        self._patch.stop()
