# -*- coding: utf-8 -*-

from passport.backend.core.db.query import DbTransaction
from passport.backend.core.db.runner import get_id_from_query_result
from passport.backend.core.differ.types import (
    Diff,
    EmptyDiff,
)
from passport.backend.core.differ.utils import slice_diff
from passport.backend.core.serializers.query import (
    GenericDeleteQuery,
    GenericInsertQuery,
    GenericUpdateQuery,
)
from passport.backend.core.undefined import Undefined
from six import iteritems


class Serializer(object):
    pass


class AliasSerializer(Serializer):
    def serialize(self, old, new, difference):
        queries = []
        if not old and new:
            queries.extend(self.create(new))
        elif old and not new:
            queries.extend(self.delete(old))
        elif difference != EmptyDiff:
            queries.extend(self.change(old, new, difference))
        return queries

    def create(self, new):
        raise NotImplementedError

    def change(self, old, new, difference):
        raise NotImplementedError

    def delete(self, old):
        raise NotImplementedError


def serialize(old_snapshot, new_snapshot, difference, type_mapper):

    if old_snapshot is None and new_snapshot is None:
        return

    if old_snapshot and new_snapshot:
        assert type(old_snapshot) == type(new_snapshot)

    snapshot_type = type(new_snapshot or old_snapshot)
    sz = type_mapper[snapshot_type.__name__]()
    for obj in sz.serialize(old_snapshot, new_snapshot, difference):
        try:
            obj, callback = obj
        except TypeError:
            callback = None
        yield obj, callback


def serialize_attr(old_snapshot, new_snapshot, difference, attr, serialize):
    model_diff = slice_diff(difference, attr)
    if model_diff == EmptyDiff:
        return []

    actions = serialize(getattr(old_snapshot, attr, None) if old_snapshot else None,
                        getattr(new_snapshot, attr, None) if new_snapshot else None,
                        model_diff)
    return actions


def serialize_subscriptions(old, new, difference, serialize):
    model_diff = slice_diff(difference, 'subscriptions')
    if model_diff == EmptyDiff:
        return

    new_sids = {}
    changed_sids = {}
    deleted_sids = {}
    # Следующие три for'а нужны только из-за того
    # что у нас диффер различает состояния added и changed.
    # Вполне возможно, что их надо выпилить и оставить лишь changed и deleted
    for sid, subscription in iteritems(model_diff.added):
        if (not old) or (not old.subscriptions) or (sid not in old.subscriptions):
            new_sids[sid] = Diff(added=subscription, changed={}, deleted={})
        else:
            changed_sids[sid] = Diff(added={}, changed=subscription, deleted={})
    for sid, subscription in iteritems(model_diff.changed):
        old_sub = changed_sids.get(sid, Diff(added={}, changed={}, deleted={}))
        old_sub.changed.update(subscription)
        changed_sids[sid] = old_sub
    for sid, subscription in iteritems(model_diff.deleted):
        # Удалять все поля подписки, вместо удаления -- не поддерживается
        if subscription is None:
            deleted_sids[sid] = Diff(added={}, changed={}, deleted=subscription)

    for sid, sub_diff in sorted(new_sids.items()):
        for query in serialize(None, new.subscriptions[sid], sub_diff):
            yield query
    for sid, sub_diff in sorted(changed_sids.items()):
        for query in serialize(old.subscriptions[sid], new.subscriptions[sid], sub_diff):
            yield query
    for sid, sub_diff in sorted(deleted_sids.items()):
        for query in serialize(old.subscriptions[sid], None, sub_diff):
            yield query


class DirectSerializer(Serializer):
    model = None
    table = None
    id_field_name = 'id'

    # Если установлен этот флаг и сериализатор порождает более чем 1 запрос,
    # то будут добавлены запросы на начало и завершение транзакции перед ними.
    transactional = True

    # Отображение имен полей модели в столбцы БД, если нельзя сделать
    # 1 -> 1 (e.g. 'model_id' -> 'id').
    field_mapping = None

    def __init__(self, *args, **kwargs):
        if not self.field_mapping:
            self.field_mapping = {}
        super(DirectSerializer, self).__init__(*args, **kwargs)

    def extract_data(self, old, new, fields):
        """
        Получаем данные, которые будут в дальнейшем переданы конкретному
        методу сериализации. Именно здесь нужно реализовывать свою логику
        упаковки / модификации данных перед сохранением, e.g. сохранять
        один атрибут в составе другого или перекодировать.

        В базовом варианте функция игнорирует поля со значением Undefined.
        """
        data = {}
        for k in fields:
            actual_k = self.field_mapping.get(k, k)
            v = getattr(new, k)

            if not actual_k:
                continue
            if v is not Undefined:
                data[actual_k] = v

        return data

    def _serialize(self, old, new, difference):
        if old and not new:
            queries = self.delete(old)
        elif new:
            diff_fields = difference.get_changed_fields()
            data = self.extract_data(old, new, diff_fields)

            if not old:
                queries = self.create(old, new, data)
            else:
                queries = self.change(old, new, data)

        if not queries:
            # Здесь мы расходимся со стандартным поведением сериализатора:
            # должно быть произведено хоть какое-то изменение с объектом,
            # в то время как стандартный позволяет возвращать пустой список.
            raise ValueError('No queries generated for state!')

        return queries

    def serialize(self, old, new, difference):
        # Если оба аргумента - None, то нам нечего делать.
        if old is None and new is None:
            return

        if new and difference == EmptyDiff:
            return

        if self.transactional:
            yield DbTransaction(self._serialize)(old, new, difference)
        else:
            for query in self._serialize(old, new, difference):
                yield query

    def create(self, old, new, data):
        """
        Базовый случай создания новой записи в БД с указанным набором данных.
        После создания записи будет установлен атрибут id на объекте модели.
        """
        yield (
            GenericInsertQuery(self.table, data),
            lambda result: setattr(new, self.id_field_name, get_id_from_query_result(result)),
        )

    def change(self, old, new, data):
        """
        Базовый случай обновления содержимого записи в БД.
        """
        return [
            GenericUpdateQuery(self.table, getattr(new, self.id_field_name), data),
        ]

    def delete(self, old):
        """
        Базовый случай удаления записи в БД. Удаляем только соответствующую
        объекту запись через ID.
        """
        return [
            GenericDeleteQuery(self.table, getattr(old, self.id_field_name)),
        ]
