# -*- coding: utf-8 -*-
from datetime import datetime
import json

from passport.backend.core.conf import settings
from passport.backend.core.db.runner import get_id_from_query_result
from passport.backend.core.db.schemas import (
    aliases_table,
    domains_events_table,
    pdd_domains_table,
    removed_aliases_table,
    suid2_table,
)
from passport.backend.core.db.utils import with_ignore_prefix
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE
from passport.backend.core.models.domain import (
    Domain,
    NEW_SCHEME_OPTIONS,
    OLD_SCHEME_OPTIONS,
    OLD_TO_NEW_SCHEME,
    OPTIONS_TO_DATA_MAPPING,
)
from passport.backend.core.serializers.base import DirectSerializer
from passport.backend.core.serializers.query import (
    GenericDeleteQuery,
    GenericInsertQuery,
    GenericQuery,
)
from passport.backend.core.undefined import Undefined
from passport.backend.utils.string import smart_bytes
from six import iteritems
from sqlalchemy.ext.compiler import compiles
from sqlalchemy.sql import select
from sqlalchemy.sql.expression import (
    and_,
    ClauseElement,
    Executable,
    HasPrefixes,
    or_,
    TextClause,
)


EVENT_NAME_TO_TYPE = {
    'add': 2,
    'delete': 3,
    'mx': 4,
    'ena': 5,
    'default_uid': 6,
    'options': 7,
    'swap': 8,
    'update': 9,
}


def build_insert_domain_event_query(event_type, domain_id, meta=None):
    """
    Создаем запрос на добавление записи в таблицу событий ПДД-доменов.
    Эта таблица через определенные промежутки времени опрашивается ЧЯ и
    используется им для поддержания актуальности данных у себя.
    """
    record = {
        'type': EVENT_NAME_TO_TYPE[event_type],
        'domain_id': domain_id,
        'ts': datetime.now(),
    }
    if meta is not None:
        record.update(meta=smart_bytes(meta))
    return GenericInsertQuery(
        domains_events_table,
        record,
    )


def prepare_domain_name(domain):
    try:
        return domain.lower().encode('idna').decode('utf8')
    except UnicodeError:
        raise ValueError('PDD domain cannot be IDNA encoded')


class DeleteDomainAliasesFromSuid2(Executable, ClauseElement, HasPrefixes):
    _execution_options = \
        Executable._execution_options.union({'autocommit': True})

    def __init__(self, domain_id):
        self.domain_id = domain_id


# FIXME: Ряд жутких хаков, которые необходимы для обеспечения идентичного для
# SQLite с MySQL-версией поведения при удалении домена. Это связано с тем, что в
# оригинальном коде на Перле использовался синтаксис множественного удаления,
# (https://dev.mysql.com/doc/refman/5.0/en/delete.html, multiple table syntax)
# который на SQLite можно реализовать только двумя отдельными запросами.
# Необходимость такого сложного запроса связана с сильным падением
# производительности при использовании отдельных запросов, т.к. поиск
# по таблицам производится два раза. Это не беспокоит нас при тестировании,
# но в продакшене это будет ужасно.
class DeleteDomainAliasesInSQLite(Executable, ClauseElement, HasPrefixes):
    _execution_options = \
        Executable._execution_options.union({'autocommit': True})

    def __init__(self, domain_id):
        self.domain_id = domain_id


@compiles(DeleteDomainAliasesFromSuid2)
def delete_suid2_polymorphic_query(element, compiler, **kw):
    if compiler.dialect.name == 'mysql':
        sql = 'DELETE a, s FROM aliases a LEFT JOIN ' \
              'suid2 s USING(uid) WHERE a.type IN (%d, %d) ' \
              'AND a.value LIKE "%d/%%%%"'
    elif compiler.dialect.name == 'sqlite':
        sql = 'DELETE FROM suid2 WHERE uid IN '\
              '(SELECT uid FROM aliases WHERE type IN (%d, %d) ' \
              'AND cast(value AS TEXT) LIKE "%d/%%")'
    return sql % (
        ALIAS_NAME_TO_TYPE['pdd'],
        ALIAS_NAME_TO_TYPE['pddalias'],
        element.domain_id,
    )


class DeleteDomainAliasesAndSuid2Query(GenericQuery):
    def __init__(self, domain_id):
        self.domain_id = domain_id
        super(DeleteDomainAliasesAndSuid2Query, self).__init__(suid2_table, {})

    def to_query(self):
        return DeleteDomainAliasesFromSuid2(self.domain_id)


@compiles(DeleteDomainAliasesInSQLite)
def delete_aliases_polymorphic_query(element, compiler, **kw):
    if compiler.dialect.name == 'mysql':
        sql = 'SELECT 1'
    elif compiler.dialect.name == 'sqlite':
        sql = 'DELETE FROM aliases WHERE type IN (%d, %d) ' \
              'AND cast(value AS TEXT) LIKE "%d/%%"'
        sql = sql % (
            ALIAS_NAME_TO_TYPE['pdd'],
            ALIAS_NAME_TO_TYPE['pddalias'],
            element.domain_id,
        )
    return sql


class DeleteDomainAliasesFromSQLiteQuery(GenericQuery):
    def __init__(self, domain_id):
        self.domain_id = domain_id
        super(DeleteDomainAliasesFromSQLiteQuery, self).__init__(suid2_table, {})

    def to_query(self):
        return DeleteDomainAliasesInSQLite(self.domain_id)


class InsertAllPddAliasesFromDomainIntoRemovedAliasesQuery(GenericInsertQuery):
    """
    Запрос перемещения алиасов удаляемого домена в табличку removed_aliases.
    """

    def __init__(self, domain_id, domain_name):
        super(InsertAllPddAliasesFromDomainIntoRemovedAliasesQuery, self).__init__(
            removed_aliases_table,
            {},
        )
        self.domain_id = domain_id
        self.domain_name = domain_name

    def to_query(self):
        at = aliases_table
        query = self.get_table().insert().from_select(
            self.get_table().c.keys(),
            select([
                at.c.uid,
                at.c.type,
                pdd_alias_to_removed_alias_sql_func(
                    domain_name=self.domain_name,
                    alias_column=at.c.value,
                ),
            ]).where(
                and_(
                    at.c.value.like(b'%d/%%' % self.domain_id),
                    at.c.type.in_([
                        ALIAS_NAME_TO_TYPE['pdd'],
                        ALIAS_NAME_TO_TYPE['pddalias'],
                    ]),
                ),
            ),
        )
        return with_ignore_prefix(query)


class DomainSerializer(DirectSerializer):
    model = Domain
    table = pdd_domains_table

    field_mapping = {
        'id': 'domain_id',
        'is_yandex_mx': 'mx',
        'registration_datetime': 'ts',
        'is_enabled': 'enabled',
        'domain': 'name',
        # Тип домена является "виртуальным" атрибутом без
        # соответствующей колонки в таблице
        'type': None,
    }

    def _create_alias_queries(self, master, aliases):
        for alias_domain in aliases:
            q = GenericInsertQuery(
                self.table,
                {
                    'name': prepare_domain_name(alias_domain),
                    'enabled': True,
                    'master_domain_id': master.id,
                    'options': '{"can_users_change_password": 1}',
                    'ts': datetime.now(),
                },
            )
            yield q, lambda result: master.alias_to_id_mapping.update({alias_domain: get_id_from_query_result(result)})

        for alias_domain in aliases:
            yield build_insert_domain_event_query('add', master.get_alias_id(alias_domain))

    def _delete_alias_queries(self, master, aliases):
        if aliases:
            prepared_aliases = [prepare_domain_name(alias).encode('utf8') for alias in aliases]
            yield GenericDeleteQuery(
                self.table,
                master.id,
                filter_by=(
                    self.table.c.name.in_(prepared_aliases)
                ),
            )
        for alias_domain in aliases:
            yield build_insert_domain_event_query('delete', master.get_alias_id(alias_domain))

    def extract_data(self, old, new, fields):
        data = super(DomainSerializer, self).extract_data(old, new, fields)

        # Функция hosted_domains ЧЯ полагается на хранение
        # имен доменов в нижнем регистре и в IDNA
        if 'name' in data:
            data['name'] = prepare_domain_name(data['name'])

        # Не сериализуем этот параметр, т.к. руками выставлять головный домен
        # нехорошо. Алиасы создаются и удаляются автоматически.
        data.pop('master_domain', None)

        if settings.OPTIONS_USE_NEW_SERIALIZATION_SCHEME:
            serialized_options = NEW_SCHEME_OPTIONS
        else:
            serialized_options = OLD_SCHEME_OPTIONS

        # Этот параметр должен быть сохранен в виде строки,
        # содержащей закодированный JSON с его значением.
        new_options = {}

        # Сериализуем опциональные поля в JSON
        for key in serialized_options:
            attr = OPTIONS_TO_DATA_MAPPING[key]
            value = data.pop(attr, Undefined)
            old_value = getattr(old, attr) if old else Undefined

            # Пропускаем опции, не заданные на новой модели,
            # либо изменённые с незаданного значения на удалённое
            if (
                value is Undefined or
                (old_value is Undefined and value is None)
            ):
                continue
            if isinstance(value, bool):
                value = int(value)
            new_options[key] = value

        if new_options:
            initial_options, options = dict(new._initial_options), {}
            if settings.OPTIONS_USE_NEW_SERIALIZATION_SCHEME:
                for key, value in iteritems(initial_options):
                    options[OLD_TO_NEW_SCHEME.get(key, key)] = value
            else:
                options = initial_options
            options.update(new_options)
            options_to_serialize = {
                option: value
                for option, value in iteritems(options)
                if value is not None
            }
            data['options'] = json.dumps(options_to_serialize, sort_keys=True) if options_to_serialize else ''

        return data

    def create(self, old, new, data):
        aliases = data.pop('aliases', [])
        if 'options' not in data:
            data['options'] = '{}'

        queries = super(DomainSerializer, self).create(old, new, data)
        alias_queries = self._create_alias_queries(new, aliases)

        for query in queries:
            yield query
        yield build_insert_domain_event_query('add', new.id)

        for alias_q in alias_queries:
            yield alias_q

    def change(self, old, new, data):
        aliases = data.pop('aliases', Undefined)

        if data:
            queries = super(DomainSerializer, self).change(old, new, data)
            for query in queries:
                yield query

        for field, event_type in (
            ('enabled', 'ena'),
            ('default_uid', 'default_uid'),
            ('mx', 'mx'),
            ('options', 'options'),
        ):
            if field in data:
                yield build_insert_domain_event_query(event_type, new.id)

        # Вполне может быть ситуация, что изменяется только
        # список алиасов.
        if aliases is Undefined:
            return

        # Так как связи между записями прописаны через поле с ID
        # головного домена, то здесь мы рассматриваем два случая:
        # - мы удаляем из таблицы все домены, которые теперь не являются
        # нашими алиасами
        # - мы обновляем список доменов (ставим всем ID 0), которые стали
        # нашими алиасами
        old_aliases = set(old.aliases or [])
        new_aliases = set(new.aliases or [])
        domains_to_create = new_aliases - old_aliases
        domains_to_delete = old_aliases - new_aliases

        # Если алиас более не привязан к нам, то надо удалить его домен
        for query in self._delete_alias_queries(new, domains_to_delete):
            yield query

        # Если какой-то алиас объявлен как принадлежащий нам, то
        # создаем для него новый объект домена и объявляем зависимым
        # от нас.
        for query in self._create_alias_queries(new, domains_to_create):
            yield query

    def delete(self, old):
        # Перед удалением домена заносим все его алиасы в табличку удаленных
        yield InsertAllPddAliasesFromDomainIntoRemovedAliasesQuery(old.id, old.domain)

        # Удаляем связанные с доменом алиасы
        yield DeleteDomainAliasesAndSuid2Query(old.id)
        yield DeleteDomainAliasesFromSQLiteQuery(old.id)

        # При удалении домена также удаляются и все подчиненные ему домены.
        yield GenericDeleteQuery(
            self.table,
            old.id,
            filter_by=or_(
                self.table.c.domain_id == old.id,
                self.table.c.master_domain_id == old.id,
            ),
        )

        # Запишем информацию об удалении в журнал событий
        yield build_insert_domain_event_query('delete', old.id)


def insert_alias_pdd_alias_into_removed_aliases_query(serializer,
                                                      uid,
                                                      domain_name,
                                                      login):
    return serializer.build_insert_alias_with_value_into_removed_aliases(
        uid,
        'pddalias',
        '%s/%s' % (prepare_domain_name(domain_name), login),
    )


class pdd_alias_to_removed_alias_sql_func(TextClause):
    def __init__(self, domain_name, alias_column):
        self.domain_name = prepare_domain_name(domain_name)
        self.alias_column = alias_column.table.name + '.' + alias_column.name
        super(pdd_alias_to_removed_alias_sql_func, self).__init__(self.domain_name)


@compiles(pdd_alias_to_removed_alias_sql_func)
def _sql_find_in_substring_element(element, compiler, **kw):
    if compiler.dialect.name == 'mysql':
        func = "concat('%(domain_name)s', SUBSTR(%(alias_column)s, LOCATE('/', %(alias_column)s)))"
        return func % dict(domain_name=element.domain_name, alias_column=element.alias_column)
    elif compiler.dialect.name == 'sqlite':
        return "'%s' || LTRIM(%s, '0123456789')" % (element.domain_name, element.alias_column)
