# -*- coding: utf-8 -*-

from datetime import (
    datetime,
    timedelta,
)

from passport.backend.core.conf import settings
from passport.backend.core.db.query import DbTransaction
from passport.backend.core.db.runner import get_id_from_query_result
from passport.backend.core.eav_type_mapping import ATTRIBUTE_NAME_TO_TYPE
from passport.backend.core.serializers.eav.base import (
    EavAttributeChange,
    EavAttributeMap,
    EavSerializer,
)
from passport.backend.core.serializers.eav.processors import (
    as_is_processor,
    billing_features_processor,
    boolean_processor,
    datetime_processor,
    default_processor,
    JoinProcessors,
    nullable_boolean_processor,
    str_processor,
    unary_field_processor,
)
from passport.backend.core.serializers.eav.query import (
    AccountDeletionOperationCreateQuery,
    AccountDeletionOperationDeleteQuery,
    DeleteMemberFromAllFamiliesQuery,
    EavDeleteAttributeQuery,
    EavInsertAttributeWithOnDuplicateKeyUpdateQuery,
    PassManDeleteAllRecoveryKeysQuery,
    SUID_TABLES,
)
from passport.backend.core.undefined import Undefined
from passport.backend.utils.time import datetime_to_unixtime
from six import iteritems


ACCOUNT_DISABLED_ON_DELETION = 2  # FIXME


@JoinProcessors([default_processor])
def account_registration_datetime_processor(datetime):
    timestamp = int(datetime_to_unixtime(datetime))
    if timestamp != 0:
        return timestamp


@JoinProcessors([default_processor])
def account_counter_processor(counter):
    if not counter.value:
        return
    return str(counter)


@JoinProcessors([default_processor])
def external_organization_ids_processor(external_organization_ids):
    if not external_organization_ids:
        return
    return ','.join([
        str(org_id)
        for org_id in sorted(external_organization_ids)
    ])


class AccountEavSerializer(EavSerializer):
    USER_DEFINED_LOGIN_EAV_MAPPER = EavAttributeMap(
        'account.user_defined_login',
        as_is_processor,
    )

    EAV_FIELDS_MAPPER = {
        'registration_datetime': EavAttributeMap(
            'account.registration_datetime',
            account_registration_datetime_processor,
        ),
        'global_logout_datetime': EavAttributeMap(
            'account.global_logout_datetime',
            datetime_processor,
        ),
        'tokens_revoked_at': EavAttributeMap(
            'revoker.tokens',
            datetime_processor,
        ),
        'web_sessions_revoked_at': EavAttributeMap(
            'revoker.web_sessions',
            datetime_processor,
        ),
        'app_passwords_revoked_at': EavAttributeMap(
            'revoker.app_passwords',
            datetime_processor,
        ),
        'is_employee': EavAttributeMap(
            'account.is_employee',
            boolean_processor,
        ),
        'is_maillist': EavAttributeMap(
            'account.is_maillist',
            boolean_processor,
        ),
        'disabled_status': EavAttributeMap(
            'account.is_disabled',
            default_processor,
        ),
        'default_email': EavAttributeMap(
            'account.default_email',
            default_processor,
        ),
        'enable_app_password': EavAttributeMap(
            'account.enable_app_password',
            unary_field_processor,
        ),
        'is_shared': EavAttributeMap(
            'account.is_shared',
            unary_field_processor,
        ),
        'is_easily_hacked': EavAttributeMap(
            'account.is_easily_hacked',
            unary_field_processor,
        ),
        'is_connect_admin': EavAttributeMap(
            'account.is_connect_admin',
            unary_field_processor,
        ),
        'show_2fa_promo': EavAttributeMap(
            'account.show_2fa_promo',
            unary_field_processor,
        ),
        'auth_email_datetime': EavAttributeMap(
            'account.auth_email_datetime',
            datetime_processor,
        ),
        'failed_auth_challenge_checks_counter': EavAttributeMap(
            'account.failed_auth_challenge_checks_counter',
            account_counter_processor,
        ),
        'audience_on': EavAttributeMap(
            'account.audience_on',
            unary_field_processor,
        ),
        'totp_junk_secret': EavAttributeMap(
            'account.totp.junk_secret',
            default_processor,
        ),
        'is_money_agreement_accepted': EavAttributeMap(
            'account.is_money_agreement_accepted',
            unary_field_processor,
        ),
        'additional_data_asked': EavAttributeMap(
            'account.additional_data_asked',
            default_processor,
        ),
        'additional_data_ask_next_datetime': EavAttributeMap(
            'account.additional_data_ask_next_datetime',
            datetime_processor,
        ),
        'content_rating_class': EavAttributeMap(
            'account.content_rating_class',
            default_processor,
        ),
        'creator_uid': EavAttributeMap(
            'account.creator_uid',
            default_processor,
        ),
        'external_organization_ids': EavAttributeMap(
            'account.external_organization_ids',
            external_organization_ids_processor,
        ),
        'music_content_rating_class': EavAttributeMap(
            'account.music_content_rating_class',
            default_processor,
        ),
        'phonish_namespace': EavAttributeMap(
            'account.phonish_namespace',
            default_processor,
        ),
        'video_content_rating_class': EavAttributeMap(
            'account.video_content_rating_class',
            default_processor,
        ),
        'magic_link_login_forbidden': EavAttributeMap(
            'account.magic_link_login_forbidden',
            unary_field_processor,
        ),
        'qr_code_login_forbidden': EavAttributeMap(
            'account.qr_code_login_forbidden',
            unary_field_processor,
        ),
        'sms_code_login_forbidden': EavAttributeMap(
            'account.sms_code_login_forbidden',
            unary_field_processor,
        ),
        'billing_features': EavAttributeMap(
            'account.billing_features',
            billing_features_processor,
        ),
        'force_challenge': EavAttributeMap(
            'account.force_challenge',
            unary_field_processor,
        ),
        'user_defined_public_id': EavAttributeMap(
            'account.user_defined_public_id',
            default_processor,
        ),
        'sms_2fa_on': EavAttributeMap(
            'account.sms_2fa_on',
            unary_field_processor,
        ),
        'forbid_disabling_sms_2fa': EavAttributeMap(
            'account.forbid_disabling_sms_2fa',
            unary_field_processor,
        ),
        'is_verified': EavAttributeMap(
            'account.is_verified',
            unary_field_processor,
        ),
        'hide_yandex_domains_emails': EavAttributeMap(
            'account.hide_yandex_domains_emails',
            unary_field_processor,
        ),
        'mail_status': EavAttributeMap(
            'subscription.mail.status',
            default_processor,
        ),
        'unsubscribed_from_maillists': EavAttributeMap(
            'account.unsubscribed_from_maillists',
            str_processor,
        ),
        'personal_data_third_party_processing_allowed': EavAttributeMap(
            'account.personal_data_third_party_processing_allowed',
            nullable_boolean_processor,
        ),
        'personal_data_public_access_allowed': EavAttributeMap(
            'account.personal_data_public_access_allowed',
            nullable_boolean_processor,
        ),
        'family_pay_enabled': EavAttributeMap(
            'account.family_pay.enabled',
            str_processor,
        ),
        'is_documents_agreement_accepted': EavAttributeMap(
            'account.is_documents_agreement_accepted',
            unary_field_processor,
        ),
        'is_dzen_sso_prohibited': EavAttributeMap(
            'account.is_dzen_sso_prohibited',
            unary_field_processor,
        ),
        'last_child_family': EavAttributeMap(
            'account.last_child_family',
            str_processor,
        ),
        'can_manage_children': EavAttributeMap(
            'account.can_manage_children',
            unary_field_processor,
        ),
    }

    def serialize(self, old, new, difference):
        if new is None:
            source = self.delete(old, difference)
        else:
            source = self.change(old, new, difference)

        for query in source:
            yield query

    def delete(self, account, difference):
        from passport.backend.core.serializers.eav import serialize

        if account.deletion_operation:
            del_op_age = datetime.now() - account.deletion_operation.started_at
            reservation_period = timedelta(seconds=settings.LOGIN_QUARANTINE_PERIOD) - del_op_age
        else:
            reservation_period = timedelta(seconds=settings.LOGIN_QUARANTINE_PERIOD)
        reserved_till = datetime.now() + reservation_period

        if account.is_pdd:
            # У удаляемых ПДД-алиасов особый формат при вставке в таблицу
            # removed_aliases: <domain>/<login>, при этом хранятся они в
            # aliases как <domain_id>/<login>
            yield self.build_insert_all_pdd_aliases_from_account_into_removed_aliases(
                account.uid,
                account.domain,
            )
        elif not account.is_kinopoisk:
            if reservation_period > timedelta(seconds=0):
                yield self.build_insert_into_reserved_logins(account.login, reserved_till)

        if reservation_period > timedelta(seconds=0):
            if account.public_id_alias:
                yield self.build_insert_into_reserved_logins(account.public_id_alias.alias, reserved_till)
            if hasattr(account.public_id_alias, 'old_public_ids'):
                for alias in sorted(account.public_id_alias.old_public_ids):
                    yield self.build_insert_into_reserved_logins(alias, reserved_till)

        yield self.build_insert_all_aliases_except_pdd_into_removed_aliases(account.uid)

        if account.is_kiddish:
            def tx():
                # Так как
                #
                # * Не должно быть ребёнкишей без семьи
                # * Несуществующий аккаунт не должен быть участником семьи
                #
                # поэтому удаление алиасов и удаление ребёнкиша из всех семей
                # должно случиться атомарно.
                yield self.build_delete_aliases_query(account.uid)
                yield self.build_delete_member_from_all_families_query(account.uid)

            yield DbTransaction(tx)()

        else:
            yield self.build_delete_aliases_query(account.uid)

        # Т.к. почти все подписки представлены в атрибутах, то они
        # будут удалены дальнейшими запросами. Здесь явно обрабатываем
        # только требующие удаления из дополнительных таблиц.
        if account.subscriptions is not Undefined:
            for sid, table in iteritems(SUID_TABLES):
                if sid in account.subscriptions:
                    yield self.build_delete_suid_query(account.uid, sid)

        for query in serialize.serialize_phones(account, None, difference):
            yield query

        for query in serialize.serialize_eav_attr(account, None, difference, 'emails'):
            yield query

        for query in serialize.serialize_eav_attr(account, None, difference, 'webauthn_credentials'):
            yield query

        yield self.build_delete_all_attributes_query(account.uid)
        yield self.build_delete_all_extended_attributes_query(account.uid)
        yield self.build_delete_from_password_history(account.uid)
        yield PassManDeleteAllRecoveryKeysQuery(account.uid)

        for query in serialize.serialize_eav_account_deletion_operation(account, None, difference):
            yield query

    def change(self, old, new, difference):
        from passport.backend.core.serializers.eav import serialize

        create = True if old is None else False
        # Для кинопоисковых аккаунтов и тестов надо уметь создавать пользователя с заданным uid
        if create and not new.uid:
            yield (
                # Мы ещё не знаем UID, поэтому ПДДшника определяем по наличию домена
                self.build_increment_uid_query(new.domain and new.domain.id),
                lambda result: setattr(new, 'uid', get_id_from_query_result(result)),
            )

        for query in serialize.serialize_eav_subscriptions(old, new, difference):
            yield query

        queries = self.serialize_account_fields(old, new, difference, create)

        for field in [
            'hint',
            'karma',
            'password',
            'person',
            'phone',
            'plus',
            'takeout',
            'totp_secret',
            'rfc_totp_secret',
            'browser_key',
            'emails',
            'migrated_from_phonishes',
            'passman_recovery_key',
            'webauthn_credentials',
            'scholar_password',
        ]:
            queries.extend(serialize.serialize_eav_attr(old, new, difference, field))

        queries.append(serialize.serialize_eav_phones(old, new, difference))
        queries.extend(serialize.serialize_eav_account_deletion_operation(old, new, difference))

        # Алиасы при регистрации создаём после атрибутов: чтобы не появлялось
        # пользователей с логинами, но без атрибутов.
        # Всегда просматриваем все алиасы, чтобы отлавливать попытки изменения иммутабельных
        alias_list = (
            'portal_alias',
            'pdd_alias',
            'lite_alias',
            'social_alias',
            'scholar_alias',
            'phonish_alias',
            'neophonish_alias',
            'mailish_alias',
            'kinopoisk_alias',
            'phonenumber_alias',
            'public_id_alias',
            'altdomain_alias',
            'uber_alias',
            'yambot_alias',
            'kolonkish_alias',
            'kiddish_alias',
            'yandexoid_alias',
            'bank_phonenumber_alias',
            'federal_alias',
        )

        for field in alias_list:
            queries.extend(serialize.serialize_eav_attr(old, new, difference, field))

        for query in queries:
            yield query

    def build_extra_attrs_changes(self, old, new):
        attrs_changes = []

        if (
            (not old or old.login != new.login) and
            (new.is_normal or new.is_pdd)
        ):
            # Сериализуем user_defined_login тут, чтобы склеить запрос с другими изменениями атрибутов
            if new.user_defined_login != new.normalized_login:
                value_to_serialize = new.user_defined_login
            elif old:
                # В БД мог остаться предыдущий user_defined_login (например, от неудавшейся попытки дорегистрации
                # аккаунта) - надо его удалить явно
                value_to_serialize = None
            else:
                # Аккаунт новый (в БД мусора быть не может), user_defined_login не нужен
                return attrs_changes

            attrs_changes.append(
                EavAttributeChange(
                    self.USER_DEFINED_LOGIN_EAV_MAPPER,
                    old.user_defined_login if old else Undefined,
                    value_to_serialize,
                ),
            )
        return attrs_changes

    def serialize_account_fields(self, old, new, difference, create):
        fields = difference.get_changed_fields()
        auto_serializable_fields = fields.intersection(self.EAV_FIELDS_MAPPER)
        extra_attrs_changes = self.build_extra_attrs_changes(old, new)
        return self.build_change_fields_queries(
            self.EAV_FIELDS_MAPPER,
            new.uid,
            None,
            new,
            auto_serializable_fields,
            create,
            extra_attrs_changes,
        )

    def build_delete_member_from_all_families_query(self, uid):
        return DeleteMemberFromAllFamiliesQuery(uid)


class AccountDeletionOperationSerializer(EavSerializer):
    def serialize(self, old, new, difference):
        old_op = old and old.deletion_operation
        new_op = new and new.deletion_operation
        account_undisabled = (
            old and
            old.disabled_status == ACCOUNT_DISABLED_ON_DELETION and
            new and
            # В том числе и случай, когда аккаунт становится просто
            # заблокированным.
            new.disabled_status != ACCOUNT_DISABLED_ON_DELETION
        )

        if not old_op and new_op and new.disabled_status != ACCOUNT_DISABLED_ON_DELETION:
            raise ValueError('You should disable account before deletion')

        queries = []
        if not old_op and new_op:
            queries = self._create(new_op, new.uid)
        elif old_op and not new_op or account_undisabled:
            queries = self._delete(old.uid)
        elif old_op and new_op and not account_undisabled:
            queries = self._change(old_op, new_op, new.uid)
        return queries

    def _create(self, new_op, uid):
        started_at = int(datetime_to_unixtime(new_op.started_at))
        return [
            EavInsertAttributeWithOnDuplicateKeyUpdateQuery(
                uid,
                [(ATTRIBUTE_NAME_TO_TYPE['account.deletion_operation_started_at'], started_at)],
            ),
            AccountDeletionOperationCreateQuery(uid, new_op.started_at),
        ]

    def _delete(self, uid):
        return [
            AccountDeletionOperationDeleteQuery(uid),
            EavDeleteAttributeQuery(
                uid,
                [ATTRIBUTE_NAME_TO_TYPE['account.deletion_operation_started_at']],
            ),
        ]

    def _change(self, old_op, new_op, uid):
        queries = []
        if old_op.started_at != new_op.started_at:
            queries.extend(self._create(new_op, uid))
        return queries
