# -*- coding: utf-8 -*-

from passport.backend.core.differ.differ import diff
from passport.backend.core.serializers.base import AliasSerializer as BaseAliasSerializer
from passport.backend.core.serializers.domain import insert_alias_pdd_alias_into_removed_aliases_query
from passport.backend.core.serializers.eav.base import (
    EavAttributeMap,
    EavSerializer,
)
from passport.backend.core.serializers.eav.processors import unary_field_processor


class AliasEavSerializer(EavSerializer, BaseAliasSerializer):
    pass


class ImmutableAliasEavSerializer(AliasEavSerializer):
    NAME = None

    def _build_create_alias_id_queries(self, new):
        if new.serialized_alias != new.serialized_alias.lower():
            raise ValueError('Immutable alias {} must be in lower case'.format(repr(new.serialized_alias)))
        return [
            self.build_insert_alias_query(
                new.parent.uid,
                self.NAME,
                new.serialized_alias,
            ),
        ]

    def _build_delete_alias_id_queries(self, old):
        uid = old.parent.uid
        return [
            self.build_mass_insert_alias_into_removed_aliases(uid, self.NAME),
            self.build_delete_alias_query(uid, self.NAME),
        ]

    def create(self, new):
        return self._build_create_alias_id_queries(new)

    def change(self, old, new, difference):
        raise ValueError('Alias is immutable')

    def delete(self, old):
        return self._build_delete_alias_id_queries(old)


class MutableAliasEavSerializer(ImmutableAliasEavSerializer):
    EAV_FIELDS_MAPPER = dict()
    ALIAS_ID_NAME = None

    def _is_alias_id_changed(self, old, new):
        return getattr(old, self.ALIAS_ID_NAME) != getattr(new, self.ALIAS_ID_NAME)

    def _build_change_alias_queries(self, old, new):
        uid = old.parent.uid
        return [
            self.build_mass_insert_alias_into_removed_aliases(uid, self.NAME),
            self.build_update_alias_query(uid, self.NAME, new.serialized_alias),
        ]

    def _build_change_alias_fields_queries(self, old, new):
        uid = old.parent.uid if old else new.parent.uid
        difference = diff(old, new)
        changed_fields = difference.get_changed_fields(
            fields_to_intersect_with=self.EAV_FIELDS_MAPPER.keys(),
        )
        return self.build_change_fields_queries(
            eav_mapper=self.EAV_FIELDS_MAPPER,
            uid=uid,
            old=old,
            new=new,
            changed_fields=changed_fields,
        )

    def create(self, new):
        queries = []
        queries.extend(
            self._build_change_alias_fields_queries(
                new=new,
                old=None,
            ),
        )
        queries.extend(self._build_create_alias_id_queries(new))
        return queries

    def change(self, old, new, difference):
        queries = []

        if self._is_alias_id_changed(old, new):
            queries.extend(self._build_change_alias_queries(old, new))

        queries.extend(self._build_change_alias_fields_queries(old, new))

        return queries

    def delete(self, old):
        queries = []
        queries.extend(self._build_delete_alias_id_queries(old))
        queries.extend(
            self._build_change_alias_fields_queries(
                old=old,
                new=None,
            ),
        )
        return queries


class PhonenumberAliasEavSerializer(MutableAliasEavSerializer):
    NAME = 'phonenumber'
    ALIAS_ID_NAME = 'number'

    EAV_FIELDS_MAPPER = {
        'enable_search': EavAttributeMap(
            'account.enable_search_by_phone_alias',
            unary_field_processor,
        ),
    }


class AltDomainAliasEavSerializer(ImmutableAliasEavSerializer):
    NAME = 'altdomain'


class YandexoidAliasEavSerializer(ImmutableAliasEavSerializer):
    NAME = 'yandexoid'


class UndeletableAliasEavSerializer(ImmutableAliasEavSerializer):
    def delete(self, old):
        raise ValueError('Alias cannot be deleted')


class UndeletableForNonPortalAliasEavSerializer(ImmutableAliasEavSerializer):
    def delete(self, old):
        if not old.parent.is_normal:
            raise ValueError('Alias cannot be deleted')
        return super(UndeletableForNonPortalAliasEavSerializer, self).delete(old)


class PortalAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'portal'


class PublicIdAliasEavSerializer(MutableAliasEavSerializer):
    NAME = 'public_id'
    ALIAS_ID_NAME = 'alias'

    def serialize(self, old, new, difference):
        account = old.parent if old else new.parent
        logins_have_changed = (
            'old_public_ids' in difference.added or
            'old_public_ids' in difference.changed
        )

        if logins_have_changed:
            difference.added.pop('old_public_ids', None)
            difference.changed.pop('old_public_ids', None)

        for query in super(PublicIdAliasEavSerializer, self).serialize(old, new, difference):
            yield query

        old_logins = old.old_public_ids if hasattr(old, 'old_public_ids') else set()
        new_logins = new.old_public_ids if hasattr(new, 'old_public_ids') else set()
        added_logins = new_logins - old_logins
        removed_logins = old_logins - new_logins

        for login in sorted(removed_logins):
            login = login.lower()
            yield self.build_insert_alias_with_value_into_removed_aliases(
                account.uid,
                'old_public_id',
                login,
            )
            yield self.build_delete_alias_with_value_query(
                account.uid,
                'old_public_id',
                login,
            )

        for login in sorted(added_logins):
            yield self.build_insert_alias_query(
                account.uid,
                'old_public_id',
                login.lower(),
                surrogate_type='{}-{}'.format(
                    self.alias_name_to_type('old_public_id'),
                    login.lower(),
                ),
            )


class PddAliasEavSerializer(UndeletableForNonPortalAliasEavSerializer):
    NAME = 'pdd'

    def serialize(self, old, new, difference):
        account = old.parent if old else new.parent
        logins_have_changed = (
            'additional_logins' in difference.added or
            'additional_logins' in difference.changed
        )

        if logins_have_changed:
            difference.added.pop('additional_logins', None)
            difference.changed.pop('additional_logins', None)

        for query in super(PddAliasEavSerializer, self).serialize(old, new, difference):
            yield query

        old_logins = old.additional_logins if old else set()
        new_logins = new.additional_logins if new else set()
        added_logins = new_logins - old_logins
        removed_logins = old_logins - new_logins

        for login in sorted(removed_logins):
            login = login.lower()
            yield insert_alias_pdd_alias_into_removed_aliases_query(
                self,
                account.uid,
                account.domain.domain,
                login,
            )
            yield self.build_delete_alias_with_value_query(
                account.uid,
                'pddalias',
                '%d/%s' % (account.domain.id, login),
            )

        for login in sorted(added_logins):
            serialized_alias = '%d/%s' % (account.domain.id, login.lower())
            # TODO: На данный момент 8 алиас является единственным, который использует
            # surrogate_type подобным образом. Это связано с особенностью схемы таблицы
            # aliases, где primary key объявлен через (uid, surrogate_type). Если мы
            # когда-нибудь добавим ещё один тип алиаса с множественными значениями,
            # то надо будет что-то придумать.
            yield self.build_insert_alias_query(
                account.uid,
                'pddalias',
                serialized_alias,
                surrogate_type=serialized_alias,
            )


class LiteAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'lite'


class SocialAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'social'


class PhonishAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'phonish'


class NeophonishAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'neophonish'


class MailishAliasEavSerializer(MutableAliasEavSerializer):
    NAME = 'mailish'
    ALIAS_ID_NAME = 'mailish_id'

    def delete(self, old):
        raise ValueError('Alias cannot be deleted')


class KinopoiskAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'kinopoisk'


class UberAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'uber'


class YambotAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'yambot'


class KolonkishAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'kolonkish'


class KiddishAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'kiddish'


class ScholarAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'scholar'


class BankPhoneNumberAliasEavSerializer(MutableAliasEavSerializer):
    NAME = 'bank_phonenumber'


class FederalAliasEavSerializer(UndeletableAliasEavSerializer):
    NAME = 'federal'
