# -*- coding: utf-8 -*-

from collections import namedtuple

from passport.backend.core.eav_type_mapping import (
    ALIAS_NAME_TO_TYPE,
    ALLOW_ZERO_VALUE_ATTRS,
    ATTRIBUTE_NAME_TO_TYPE,
    ATTRIBUTES_TYPES_ALLOW_APPEND,
    ATTRIBUTES_TYPES_ALLOW_INCREMENT,
    ATTRIBUTES_TYPES_NEED_INSERT_INTO_ON_CREATE,
    ATTRIBUTES_TYPES_NEED_INSERT_WITH_IF_EQUALS_ON_UPDATE,
    EXTENDED_ATTRIBUTES_ENTITY_NAME_TO_TYPE_MAPPING,
    EXTENDED_ATTRIBUTES_NAME_TO_TYPE_MAPPING,
    SID_TO_SUBSCRIPTION_ATTR,
)
from passport.backend.core.serializers.base import Serializer
from passport.backend.core.serializers.eav.exceptions import EavDeletePDDAliasWithoutValueError
from passport.backend.core.serializers.eav.query import (
    DeleteAliasWithValueQuery,
    EavDeleteAliasQuery,
    EavDeleteAllAttributesQuery,
    EavDeleteAllExtendedAttributesQuery,
    EavDeleteAttributeQuery,
    EavDeleteExtendedAttributeQuery,
    EavDeleteFromPasswordHistoryQuery,
    EavDeleteSuidQuery,
    EavEmailIdIncrementQuery,
    EavInsertAliasQuery,
    EavInsertAttributeQuery,
    EavInsertAttributeWithOnDuplicateKeyAppendQuery,
    EavInsertAttributeWithOnDuplicateKeyIfValueEqualsUpdateQuery,
    EavInsertAttributeWithOnDuplicateKeyIncrementQuery,
    EavInsertAttributeWithOnDuplicateKeyUpdateQuery,
    EavInsertExtendedAttributeWithOnDuplicateKeyQuery,
    EavInsertPasswordHistoryQuery,
    EavInsertSuidQuery,
    EavPhoneIdIncrementQuery,
    EavSuidIncrementQuery,
    EavTotpSecretIdIncrementQuery,
    EavUidIncrementQuery,
    EavUpdateAliasQuery,
    EavWebauthnCredentialIdIncrementQuery,
    InsertAliasesWithValueIntoRemovedAliasesQuery,
    InsertAllPddAliasesFromAccountIntoRemovedAliasesQuery,
    InsertLoginIntoReservedQuery,
    MassInsertAliasesIntoRemovedAliasesQuery,
)
from passport.backend.core.undefined import Undefined
import six


EavAttributeMap = namedtuple('EavAttributeMap', 'attr_name processor')
EavAttributeChange = namedtuple('EavAttributeChange', 'eav_attr_map old_value new_value')
EavExtendedAttributeMap = namedtuple('EavExtendedAttributeMap', 'entity_type attr_name processor')
EavExtendedAttributeChange = namedtuple('EavExtendedAttributeChange', 'ext_eav_attr_map entity_id old_value new_value')


class EavSerializer(Serializer):
    def is_writable_attr(self, eav_type, value):
        # NOT NULL
        if value is None or value == '':
            return False

        if value == 0 and eav_type not in ALLOW_ZERO_VALUE_ATTRS:
            return False

        return True

    def is_writable_ext_attr(self, entity_type, ext_attr_type, value):
        if value is None or value == '':
            return False

        return True

    @staticmethod
    def attr_name_to_type(attr_name):
        parts = attr_name.split('.')
        # TODO: переписать сериализатор, чтобы этого не требовалось.
        # Обработка специального случая с запросом номера атрибута для
        # имени вида "subscription.<SID>", чтобы не добавлять
        # в ATTRIBUTE_NAME_TO_TYPE записей, дублирующих номер атрибута
        # для символьного имени подписки.
        if len(parts) == 2 and parts[0] == 'subscription':
            try:
                sid = int(parts[1])
                return SID_TO_SUBSCRIPTION_ATTR[sid]
            except (ValueError, KeyError):
                # Если после "subscription" у нас идет не номер или такой SID
                # не отображен, то считаем это общим случаем и никак
                # дополнительно не обрабатываем.
                pass

        return ATTRIBUTE_NAME_TO_TYPE[attr_name]

    @staticmethod
    def ext_attr_name_to_type(entity_type, attr_name):
        if isinstance(entity_type, six.string_types):
            entity_type = EXTENDED_ATTRIBUTES_ENTITY_NAME_TO_TYPE_MAPPING[entity_type]
        return EXTENDED_ATTRIBUTES_NAME_TO_TYPE_MAPPING[entity_type][attr_name]

    @staticmethod
    def alias_name_to_type(alias_name):
        return ALIAS_NAME_TO_TYPE[alias_name]

    def set_attr_params(self, attr_name, cleaned_value):
        eav_type = self.attr_name_to_type(attr_name)
        if self.is_writable_attr(eav_type, cleaned_value):
            return (eav_type, cleaned_value)
        return None

    def chain_getattr(self, obj, attlist):
        attrs = attlist.split('.')
        cobj = obj
        for attr in attrs:
            cobj = getattr(cobj, attr)
            if cobj is Undefined:
                break
        return cobj

    def attrs_changes_to_eav_attrs(self, attrs_changes):
        """
        По списку [EavAttributeChange(...), ...] генерируется списки удаленных и измененных атрибутов
        """
        insert_types_values = []
        insert_odk_update_types_values = []
        insert_odk_update_if_equals_types_values = []
        insert_odk_append_types_values = []
        insert_odk_increment_types_values = []
        delete_types = []
        for eav_attr_change in attrs_changes:
            eav_attr_map = eav_attr_change.eav_attr_map
            if eav_attr_change.new_value is Undefined:
                continue
            eav_type = self.attr_name_to_type(eav_attr_map.attr_name)
            new_cleaned_value = eav_attr_map.processor(eav_attr_change.new_value)
            new_value_is_writable = self.is_writable_attr(eav_type, new_cleaned_value)

            # Не должны генерировать DELETE, если
            # поменялось значение
            # по-умолчанию на значение по-умолчанию
            old_cleaned_value = Undefined
            if eav_attr_change.old_value is not Undefined:
                old_cleaned_value = eav_attr_map.processor(eav_attr_change.old_value)
                if not self.is_writable_attr(eav_type, old_cleaned_value) and not new_value_is_writable:
                    continue

            is_old_value_empty = old_cleaned_value is Undefined
            # Удаляем, если пустое значение не разрешено для данного
            # типа атрибута, или если значение None.
            if not new_value_is_writable:
                delete_types.append(eav_type)
            elif is_old_value_empty and eav_type in ATTRIBUTES_TYPES_NEED_INSERT_INTO_ON_CREATE:
                insert_types_values.append((eav_type, new_cleaned_value))
            elif not is_old_value_empty and eav_type in ATTRIBUTES_TYPES_NEED_INSERT_WITH_IF_EQUALS_ON_UPDATE:
                # Передаем тип атрибута, ожидаемое значение, новое значение
                insert_odk_update_if_equals_types_values.append(
                    (eav_type, (old_cleaned_value, new_cleaned_value)),
                )
            elif eav_type in ATTRIBUTES_TYPES_ALLOW_APPEND:
                insert_odk_append_types_values.append((eav_type, new_cleaned_value))
            elif eav_type in ATTRIBUTES_TYPES_ALLOW_INCREMENT:
                insert_odk_increment_types_values.append((eav_type, new_cleaned_value))
            else:
                insert_odk_update_types_values.append((eav_type, new_cleaned_value))

        return (
            insert_types_values,
            insert_odk_update_types_values,
            insert_odk_update_if_equals_types_values,
            insert_odk_append_types_values,
            insert_odk_increment_types_values,
            delete_types,
        )

    def ext_attrs_changes_to_eav_attrs(self, attrs_changes):
        """
        По списку [EavExtendedAttributeChange(...), ...] генерируется списки удаленных и измененных атрибутов
        """
        insert_odk_types_values = []
        delete_types = []
        for ext_eav_attr_change in attrs_changes:
            ext_eav_attr_map = ext_eav_attr_change.ext_eav_attr_map
            entity_type = ext_eav_attr_map.entity_type

            ext_attribute_type = self.ext_attr_name_to_type(entity_type, ext_eav_attr_map.attr_name)
            new_cleaned_value = ext_eav_attr_map.processor(ext_eav_attr_change.new_value)
            new_value_is_writable = self.is_writable_ext_attr(
                entity_type,
                ext_attribute_type,
                new_cleaned_value,
            )

            # Не должны генерировать DELETE, если
            # поменялось значение
            # по-умолчанию на значение по-умолчанию
            if ext_eav_attr_change.old_value is not Undefined:
                old_cleaned_value = ext_eav_attr_map.processor(ext_eav_attr_change.old_value)
                old_value_is_writable = self.is_writable_ext_attr(
                    entity_type,
                    ext_attribute_type,
                    old_cleaned_value,
                )
                if not old_value_is_writable and not new_value_is_writable:
                    continue

            # Удаляем, если пустое значение не разрешено для данного
            # типа атрибута, или если значение None.
            if not new_value_is_writable:
                delete_types.append({
                    'entity_type': entity_type,
                    'type': ext_attribute_type,
                    'entity_id': ext_eav_attr_change.entity_id,
                })
            else:
                insert_odk_types_values.append({
                    'entity_type': entity_type,
                    'type': ext_attribute_type,
                    'value': new_cleaned_value,
                    'entity_id': ext_eav_attr_change.entity_id,
                })

        return insert_odk_types_values, delete_types

    def fields_to_eav_attrs(self, eav_mapper, old_model_inst, new_model_inst,
                            changed_fields, extra_attrs_changes):
        """
        По списку измененных полей генерируется списки удаленных и измененных атрибутов
        """
        attrs_changes = extra_attrs_changes or []
        for field in changed_fields:
            eav_attr_map = eav_mapper[field]
            new_value = self.chain_getattr(new_model_inst, field) if new_model_inst else None
            old_value = self.chain_getattr(old_model_inst, field) if old_model_inst else Undefined
            attrs_changes.append(EavAttributeChange(
                eav_attr_map,
                old_value,
                new_value,
            ))

        return self.attrs_changes_to_eav_attrs(attrs_changes)

    def extended_attributes_to_eav_attrs(self, eav_mapper, old_model_inst, new_model_inst,
                                         changed_fields, extra_attrs_changes):
        """
        По списку измененных полей генерируется списки удаленных и измененных расширенных атрибутов
        """

        attrs_changes = extra_attrs_changes or []
        for field in changed_fields:
            eav_attr_map = eav_mapper[field]
            new_value = self.chain_getattr(new_model_inst, field) if new_model_inst else None
            old_value = self.chain_getattr(old_model_inst, field) if old_model_inst else Undefined
            attrs_changes.append(EavExtendedAttributeChange(
                eav_attr_map,
                new_model_inst.id if new_model_inst else old_model_inst.id,
                old_value,
                new_value,
            ))

        return self.ext_attrs_changes_to_eav_attrs(attrs_changes)

    def insert_attrs_query(self, uid, types_values):
        return EavInsertAttributeQuery(uid, types_values)

    def insert_odk_update_attrs_query(self, uid, types_values):
        return EavInsertAttributeWithOnDuplicateKeyUpdateQuery(uid, types_values)

    def insert_odk_update_if_value_equals_attrs_query(self, uid, types_values):
        return EavInsertAttributeWithOnDuplicateKeyIfValueEqualsUpdateQuery(uid, types_values)

    def insert_odk_append_attrs_query(self, uid, types_values):
        return EavInsertAttributeWithOnDuplicateKeyAppendQuery(uid, types_values)

    def insert_odk_increment_attrs_query(self, uid, types_values):
        return EavInsertAttributeWithOnDuplicateKeyIncrementQuery(uid, types_values)

    def delete_attrs_query(self, uid, types):
        return EavDeleteAttributeQuery(uid, types)

    def delete_ext_attrs_query(self, uid, types):
        return EavDeleteExtendedAttributeQuery(uid, types)

    def build_set_attrs_with_cleaned_values(self, uid, attrs_values):
        """
        Вставка списка атрибутов
        """
        types_values = []
        types_values_odk = []
        for attr_name, cleaned_value in attrs_values.items():
            type_value = self.set_attr_params(attr_name, cleaned_value)
            if type_value is None:
                continue
            if type_value[0] in ATTRIBUTES_TYPES_NEED_INSERT_INTO_ON_CREATE:
                types_values.append(type_value)
            else:
                types_values_odk.append(type_value)

        queries = []
        if types_values:
            queries.append(self.insert_attrs_query(uid, types_values))
        if types_values_odk:
            queries.append(self.insert_odk_update_attrs_query(uid, types_values_odk))
        return queries

    def build_delete_attrs_by_names(self, uid, attrs_names):
        """
        Удаление списка атрибутов
        """
        types = [self.attr_name_to_type(attr_name) for attr_name in attrs_names]
        return self.delete_attrs_query(uid, types)

    def build_change_fields_queries(self, eav_mapper, uid, old, new, changed_fields,
                                    create=False, extra_attrs_changes=None, attributes_to_append=None,
                                    attributes_to_increment=None):
        """
        По списку измененных полей генерирует список EavAttributeQuery
        """
        (
            insert_types_values,
            insert_odk_update_types_values,
            insert_odk_update_with_if_types_values,
            allowed_insert_odk_append_types_values,
            allowed_insert_odk_increment_types_values,
            delete_types,
        ) = self.fields_to_eav_attrs(
            eav_mapper,
            old,
            new,
            changed_fields,
            extra_attrs_changes,
        )

        attribute_types_to_append = [
            ATTRIBUTE_NAME_TO_TYPE[attribute_name]
            for attribute_name in attributes_to_append or []
        ]
        insert_odk_append_types_values = []
        for type_, value in allowed_insert_odk_append_types_values:
            if type_ in attribute_types_to_append:
                insert_odk_append_types_values.append((type_, value))
            else:
                insert_odk_update_types_values.append((type_, value))

        attribute_types_to_increment = [
            ATTRIBUTE_NAME_TO_TYPE[attribute_name]
            for attribute_name in attributes_to_increment or []
        ]
        insert_odk_increment_types_values = []
        for type_, value in allowed_insert_odk_increment_types_values:
            if type_ in attribute_types_to_increment:
                insert_odk_increment_types_values.append((type_, value))
            else:
                insert_odk_update_types_values.append((type_, value))

        queries = []
        if insert_types_values:
            queries.append(self.insert_attrs_query(uid, insert_types_values))
        if insert_odk_update_types_values:
            queries.append(self.insert_odk_update_attrs_query(uid, insert_odk_update_types_values))
        if insert_odk_update_with_if_types_values:
            queries.append(self.insert_odk_update_if_value_equals_attrs_query(uid, insert_odk_update_with_if_types_values))
        if insert_odk_append_types_values:
            queries.append(self.insert_odk_append_attrs_query(uid, insert_odk_append_types_values))
        if insert_odk_increment_types_values:
            queries.append(self.insert_odk_increment_attrs_query(uid, insert_odk_increment_types_values))
        if delete_types and not create:
            queries.append(self.delete_attrs_query(uid, delete_types))
        return queries

    def build_extended_attribute_queries(self, eav_mapper, uid, old, new, changed_fields,
                                         create=False, extra_attrs_changes=None):
        """
        По списку измененных полей генерирует список EavExtendedAttributeQuery
        """
        insert_odk_types_values, delete_types = self.extended_attributes_to_eav_attrs(
            eav_mapper,
            old,
            new,
            changed_fields,
            extra_attrs_changes,
        )

        if insert_odk_types_values:
            yield self.insert_extended_odk_attrs_query(uid, insert_odk_types_values)
        if delete_types and not create:
            yield self.delete_ext_attrs_query(uid, delete_types)

    def build_delete_fields_query(self, eav_mapper, uid):
        delete_types = []
        for eav_attr_map in eav_mapper.values():
            eav_type = self.attr_name_to_type(eav_attr_map.attr_name)
            delete_types.append(eav_type)
        if not delete_types:
            return None
        return self.delete_attrs_query(uid, delete_types)

    def build_insert_suid_query(self, uid, sid, suid):
        return EavInsertSuidQuery(uid, sid, suid)

    def build_delete_suid_query(self, uid, sid):
        return EavDeleteSuidQuery(uid, sid)

    def build_insert_alias_query(self, uid, name, alias, surrogate_type=None):
        alias_type = self.alias_name_to_type(name)
        return EavInsertAliasQuery(
            uid,
            [(alias_type, alias, surrogate_type or alias_type)],
        )

    def build_update_alias_query(self, uid, name, alias, surrogate_type=None):
        alias_type = self.alias_name_to_type(name)
        return EavUpdateAliasQuery(
            uid,
            alias_type,
            alias,
            surrogate_type or alias_type,
        )

    def build_mass_insert_alias_into_removed_aliases(self, uid, name):
        alias_type = self.alias_name_to_type(name)
        return MassInsertAliasesIntoRemovedAliasesQuery(uid, [alias_type])

    def build_insert_alias_with_value_into_removed_aliases(self, uid, name, value):
        alias_type = self.alias_name_to_type(name)
        return InsertAliasesWithValueIntoRemovedAliasesQuery(uid, [(alias_type, value)])

    def build_insert_all_pdd_aliases_from_account_into_removed_aliases(self, uid, domain):
        return InsertAllPddAliasesFromAccountIntoRemovedAliasesQuery(uid, domain)

    def build_insert_aliases_into_removed_aliases(self, uid, aliases=None, excluded=None):
        if not aliases:
            aliases = ALIAS_NAME_TO_TYPE.keys()

        alias_types = [
            ALIAS_NAME_TO_TYPE[name]
            for name in set(aliases) - set(excluded)
        ]
        return MassInsertAliasesIntoRemovedAliasesQuery(uid, sorted(alias_types))

    def build_insert_all_aliases_except_pdd_into_removed_aliases(self, uid):
        return self.build_insert_aliases_into_removed_aliases(
            uid,
            excluded=['pdd', 'pddalias'],
        )

    def build_insert_into_reserved_logins(self, login, reserved_till):
        return InsertLoginIntoReservedQuery(login, reserved_till)

    def build_delete_alias_query(self, uid, name):
        # pddalias для удаления alias надо (uid, name, value)
        if name in ['pddalias']:
            raise EavDeletePDDAliasWithoutValueError()

        alias_type = self.alias_name_to_type(name)
        return EavDeleteAliasQuery(uid, [alias_type])

    def build_delete_aliases_query(self, uid):
        return EavDeleteAliasQuery(uid, sorted(ALIAS_NAME_TO_TYPE.values()))

    def build_delete_all_attributes_query(self, uid):
        return EavDeleteAllAttributesQuery(uid)

    def build_delete_all_extended_attributes_query(self, uid):
        return EavDeleteAllExtendedAttributesQuery(uid)

    def build_delete_alias_with_value_query(self, uid, name, value):
        alias_type = self.alias_name_to_type(name)
        return DeleteAliasWithValueQuery(uid, [(alias_type, value)])

    def build_delete_from_password_history(self, uid):
        return EavDeleteFromPasswordHistoryQuery(uid)

    def build_insert_password_history(self, uid, update_datetime, encrypted_password, reason):
        return EavInsertPasswordHistoryQuery(uid, [(update_datetime, encrypted_password, reason)])

    def build_increment_uid_query(self, is_pdd):
        return EavUidIncrementQuery(is_pdd)

    def build_increment_suid_query(self, is_pdd):
        return EavSuidIncrementQuery(is_pdd)

    def build_increment_phone_id_query(self):
        return EavPhoneIdIncrementQuery()

    def build_increment_email_id_query(self):
        return EavEmailIdIncrementQuery()

    def build_increment_totp_secret_id_query(self):
        return EavTotpSecretIdIncrementQuery()

    def build_increment_webauthn_credential_id_query(self):
        return EavWebauthnCredentialIdIncrementQuery()

    def insert_extended_odk_attrs_query(self, uid, ext_attrs_data):
        return EavInsertExtendedAttributeWithOnDuplicateKeyQuery(uid, ext_attrs_data)


class BaseListAttributeEavSerializer(EavSerializer):
    # EAV_ATTRIBUTE_NAME и EAV_MAPPER определить в подклассе.

    # EAV_ATTRIBUTE_NAME = None
    # EAV_MAPPER = {'value': EavAttributeMap(EAV_ATTRIBUTE_NAME, list_processor)}

    def serialize(self, old, new, difference):
        queries = []
        if old and not new:
            queries.extend(self.delete(old))
        elif new and not old:
            queries.extend(self.create(new))
        else:
            queries.extend(self.change(old, new, difference))
        return queries

    def create(self, new):
        if new._to_append:
            attributes_to_append = [self.EAV_ATTRIBUTE_NAME]
        else:
            attributes_to_append = []
        return self.build_change_fields_queries(
            self.EAV_MAPPER,
            new.parent.uid,
            None,
            new,
            {'value'},
            create=True,
            attributes_to_append=attributes_to_append,
        )

    def change(self, old, new, difference):
        if new._to_append:
            attributes_to_append = [self.EAV_ATTRIBUTE_NAME]
        else:
            attributes_to_append = []

        serializable_fields = difference.get_changed_fields(self.EAV_MAPPER)

        queries = self.build_change_fields_queries(
            self.EAV_MAPPER,
            new.parent.uid,
            old,
            new,
            serializable_fields,
            create=False,
            attributes_to_append=attributes_to_append,
        )
        return queries

    def delete(self, old):
        return [self.build_delete_fields_query(self.EAV_MAPPER, old.parent.uid)]
