# -*- coding: utf-8 -*-

import logging

from passport.backend.core.db.query import DbTransaction
from passport.backend.core.db.runner import run_incr_id_query
from passport.backend.core.differ.types import (
    Diff,
    EmptyDiff,
)
from passport.backend.core.differ.utils import slice_diff
from passport.backend.core.serializers.eav.base import EavSerializer
from passport.backend.core.serializers.eav.extended_attributes import EAV_EXTENDED_ATTRIBUTES_EMAIL_MAPPER
from passport.backend.core.serializers.eav.query import (
    EavDeleteAllEmailBindingsQuery,
    EavDeleteAllEmailsQuery,
    EavDeleteEmailBindingQuery,
    EavDeleteEmailQuery,
    EavInsertEmailBindingQuery,
    EavUpdateEmailBindingBound,
)
from passport.backend.core.undefined import Undefined
from passport.backend.utils.time import zero_datetime


log = logging.getLogger('passport.serializers.eav.emails')


class EmailEavSerializer(EavSerializer):
    def serialize(self, old, new, difference):
        if old and old.is_native:
            old = None
        if new and new.is_native:
            new = None

        if not old and not new:
            return

        if new and not new.address:
            raise ValueError('E-mail without address cannot be serialized!')

        emails = old.parent if old else new.parent
        uid = emails.parent.uid

        if old and not new:
            yield EavDeleteEmailBindingQuery(uid, old.id)
            yield EavDeleteEmailQuery(uid, old.id)
            return

        # Если у нас ещё нет ID, то сгенерируем его
        if new.id is Undefined:
            query = EavSerializer().build_increment_email_id_query()
            new.id = run_incr_id_query(query)

        if not old:
            yield EavInsertEmailBindingQuery(
                uid,
                new.address.encode('utf-8'),
                new.id,
                new.bound_at or zero_datetime,
            )
        elif new and old and new.bound_at != old.bound_at:
            yield EavUpdateEmailBindingBound(
                uid,
                new.id,
                new.bound_at or zero_datetime,
            )

        # Выполняем стандартную сериализацию сущностей
        for query in self.build_extended_attribute_queries(
            EAV_EXTENDED_ATTRIBUTES_EMAIL_MAPPER,
            uid, old, new,
            difference.get_changed_fields(EAV_EXTENDED_ATTRIBUTES_EMAIL_MAPPER),
            create=old is None,
        ):
            yield query


class EmailsEavSerializer(EavSerializer):
    def serialize(self, old, new, difference):
        if difference == EmptyDiff:
            return

        # FIXME: Зачинить это в функции convert(None) и пофиксить тесты/функционал
        if difference.deleted is None:
            difference = Diff(difference.added, difference.changed, {})

        if not new and old and old._emails:
            yield EavDeleteAllEmailBindingsQuery(old.parent.uid)
            yield EavDeleteAllEmailsQuery(old.parent.uid)
            return

        yield serialize_emails(old, new, difference)


@DbTransaction
def serialize_emails(old, new, difference):
    emails_diff = slice_diff(difference, '_emails')
    for address in sorted(emails_diff.get_changed_fields()):
        old_email = old.find(address) if old else None
        new_email = new.find(address) if new else None

        instance_diff = slice_diff(emails_diff, address)

        for query in EmailEavSerializer().serialize(
            old_email,
            new_email,
            instance_diff,
        ):
            yield query
