# -*- coding: utf-8 -*-

from passport.backend.core.eav_type_mapping import (
    EXTENDED_ATTRIBUTES_EMAIL_TYPE,
    EXTENDED_ATTRIBUTES_PHONE_TYPE,
    EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
)
from passport.backend.core.serializers.eav.base import EavExtendedAttributeMap
from passport.backend.core.serializers.eav.processors import (
    boolean_processor,
    default_processor,
    JoinProcessors,
)
from passport.backend.utils.time import datetime_to_integer_unixtime


@JoinProcessors([default_processor])
def datetime_processor(datetime):
    timestamp = datetime_to_integer_unixtime(datetime)
    if timestamp != 0:
        return timestamp


@JoinProcessors([default_processor])
def phone_number_processor(phone_number):
    return phone_number.digital


@JoinProcessors([default_processor])
def webauthn_public_key_processor(public_key):
    if public_key:
        return '1:%s' % public_key


EAV_EXTENDED_ATTRIBUTES_PHONE_MAPPER = {
    'number': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_PHONE_TYPE,
        'number',
        phone_number_processor,
    ),
    'created': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_PHONE_TYPE,
        'created',
        datetime_processor,
    ),
    '_bound': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_PHONE_TYPE,
        'bound',
        datetime_processor,
    ),
    'confirmed': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_PHONE_TYPE,
        'confirmed',
        datetime_processor,
    ),
    'admitted': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_PHONE_TYPE,
        'admitted',
        datetime_processor,
    ),
    'secured': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_PHONE_TYPE,
        'secured',
        datetime_processor,
    ),
}

EAV_EXTENDED_ATTRIBUTES_EMAIL_MAPPER = {
    'address': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        'address',
        default_processor,
    ),
    'bound_at': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        'bound',
        datetime_processor,
    ),
    'created_at': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        'created',
        datetime_processor,
    ),
    'confirmed_at': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        'confirmed',
        datetime_processor,
    ),
    'is_rpop': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        'is_rpop',
        boolean_processor,
    ),
    'is_unsafe': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        'is_unsafe',
        boolean_processor,
    ),
    'is_silent': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_EMAIL_TYPE,
        'is_silent',
        boolean_processor,
    ),
}

EAV_EXTENDED_ATTRIBUTES_WEBAUTHN_CREDENTIAL_MAPPER = {
    'external_id': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'external_id',
        default_processor,
    ),
    'public_key': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'public_key',
        webauthn_public_key_processor,
    ),
    'sign_count': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'sign_count',
        default_processor,
    ),
    'device_name': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'device_name',
        default_processor,
    ),
    'created_at': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'created',
        datetime_processor,
    ),
    'relying_party_id': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'relying_party_id',
        default_processor,
    ),
    'os_family_id': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'os_family_id',
        default_processor,
    ),
    'browser_id': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'browser_id',
        default_processor,
    ),
    'is_device_mobile': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'is_device_mobile',
        boolean_processor,
    ),
    'is_device_tablet': EavExtendedAttributeMap(
        EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
        'is_device_tablet',
        boolean_processor,
    ),
}
