# -*- coding: utf-8 -*-
from passport.backend.core.differ.types import EmptyDiff
from passport.backend.core.serializers.eav.base import (
    EavAttributeMap,
    EavSerializer,
)
from passport.backend.core.serializers.eav.processors import (
    default_processor,
    JoinProcessors,
)
from passport.backend.utils.time import datetime_to_unixtime


@JoinProcessors([default_processor])
def karma_value_processor(value):
    if value <= 0:
        return None
    return value


@JoinProcessors([default_processor])
def karma_activation_datetime_processor(activation_datetime):
    timestamp = int(datetime_to_unixtime(activation_datetime))
    if timestamp in [0, 2]:
        return None
    return timestamp


class KarmaEavSerializer(EavSerializer):
    EAV_FIELDS_MAPPER = {
        'activation_datetime': EavAttributeMap('karma.activation_datetime', karma_activation_datetime_processor),
        'value': EavAttributeMap('karma.value', karma_value_processor)
    }

    def serialize(self, old, new, difference):
        if old is None:
            return self.create(new)
        return self.change(old, new, difference)

    def create(self, new):
        return self.build_change_fields_queries(
            self.EAV_FIELDS_MAPPER,
            new.parent.uid,
            None,
            new,
            self.EAV_FIELDS_MAPPER.keys(),
            True)

    def change(self, old, new, difference):
        if difference == EmptyDiff:
            return []

        fields = dict(difference.changed)
        fields.update(difference.added)

        changed_fields = []
        if 'prefix' in fields or 'suffix' in fields:
            changed_fields.append('value')
        if 'activation_datetime' in fields:
            changed_fields.append('activation_datetime')

        return self.build_change_fields_queries(
            self.EAV_FIELDS_MAPPER,
            new.parent.uid,
            old,
            new,
            changed_fields)
