# -*- coding: utf-8 -*-
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_PWDHISTORY_REASON_COMPROMISED,
    BLACKBOX_PWDHISTORY_REASON_STRONG_POLICY,
)
from passport.backend.core.differ.differ import diff
from passport.backend.core.serializers.eav.base import (
    EavAttributeChange,
    EavAttributeMap,
    EavSerializer,
)
from passport.backend.core.serializers.eav.processors import (
    datetime_processor,
    default_processor,
)
from passport.backend.core.undefined import Undefined
from passport.backend.utils.time import datetime_to_unixtime


class PasswordEavSerializer(EavSerializer):
    EAV_FIELDS_MAPPER = {
        'forced_changing_reason': EavAttributeMap(
            'password.forced_changing_reason',
            default_processor,
        ),
        'forced_changing_time': EavAttributeMap(
            'password.forced_changing_time',
            datetime_processor,
        ),
        'pwn_forced_changing_suspended_at': EavAttributeMap(
            'password.pwn_forced_changing_suspended_at',
            datetime_processor,
        ),
    }

    PASSWORD_ENCRYPTED_EAV_MAPPER = EavAttributeMap('password.encrypted', default_processor)

    UPDATE_DATETIME_EAV_MAPPER = EavAttributeMap('password.update_datetime',
                                                 default_processor)
    QUALITY_EAV_MAPPER = EavAttributeMap('password.quality',
                                         default_processor)

    def delete(self, password):
        mapper = dict(self.EAV_FIELDS_MAPPER)
        mapper['password.encrypted'] = self.PASSWORD_ENCRYPTED_EAV_MAPPER
        mapper['password.update_datetime'] = self.UPDATE_DATETIME_EAV_MAPPER
        mapper['password.quality'] = self.QUALITY_EAV_MAPPER

        queries = [
            self.build_delete_fields_query(
                mapper,
                password.parent.uid,
            ),
        ]

        # Если содержимое поля пароля удаляется, то нам все равно необходимо
        # записать в историю пароль для пользователя с усиленной
        # парольной политикой.
        if password.parent.is_strong_password_required:
            password_history_query = self.build_insert_password_history(
                password.parent.uid,
                password.update_datetime,
                password.serialized_encrypted_password,
                reason=BLACKBOX_PWDHISTORY_REASON_STRONG_POLICY,
            )
            queries.insert(0, password_history_query)

        return queries

    def serialize(self, old, new, difference):
        if old and new is None:
            return self.delete(old)

        create = False if old else True

        if create and new.password_hash_provider:
            # Убедиться в получении хеша нужно только при создании аккаунта, т.к. только в этом случае
            # хеш мог быть не сгенерирован из-за отсутствия UID
            result = new.password_hash_provider.try_create_hash(uid=new.parent.uid)
            assert result.is_created
            new.encoding_version, new.encrypted_password = result.version, result.encrypted_password
            updated_difference = diff(old, new)
            difference.added.update(updated_difference.added)

        changed_fields = difference.get_changed_fields()
        extra_attrs_changes = self.build_extra_attrs_changes(old, new, changed_fields)

        queries = self.build_change_fields_queries(
            self.EAV_FIELDS_MAPPER,
            new.parent.uid,
            old,
            new,
            difference.get_changed_fields(self.EAV_FIELDS_MAPPER),
            create,
            extra_attrs_changes,
        )

        if queries and 'encrypted_password' in changed_fields and new and old and old.encrypted_password:
            if new.parent.is_strong_password_required or old.is_changing_required:
                reason = (
                    # Ситуации могут пересекаться, приоритет отдаем принудительной смене пароля, так как
                    # при поиске пароля в истории случай строгой политики включает в себя другие случаи
                    BLACKBOX_PWDHISTORY_REASON_COMPROMISED if old.is_changing_required
                    else BLACKBOX_PWDHISTORY_REASON_STRONG_POLICY
                )
                password_history_query = self.build_insert_password_history(
                    old.parent.uid,
                    new.update_datetime,
                    old.serialized_encrypted_password,
                    reason=reason,
                )
                queries.append(password_history_query)
        return queries

    def build_quality_value(self, password):
        return '%s:%s' % (password.quality_version, password.quality)

    def build_extra_attrs_changes(self, old, new, fields):
        attrs_changes = []

        if 'encrypted_password' in fields:
            attrs_changes.append(
                EavAttributeChange(
                    self.PASSWORD_ENCRYPTED_EAV_MAPPER,
                    old.serialized_encrypted_password if old else Undefined,
                    new.serialized_encrypted_password,
                ),
            )

        if 'update_datetime' in fields:
            timestamp = int(datetime_to_unixtime(new.update_datetime))
            attrs_changes.append(
                EavAttributeChange(
                    self.UPDATE_DATETIME_EAV_MAPPER,
                    Undefined,
                    timestamp,
                ),
            )

        if 'quality' in fields or 'quality_version' in fields:
            if not isinstance(new.quality_version, int) or not isinstance(new.quality, int):
                raise ValueError('Both password.quality_version and password.quality should be set to integer!')

            attrs_changes.append(
                EavAttributeChange(
                    self.QUALITY_EAV_MAPPER,
                    self.build_quality_value(old) if old else Undefined,
                    self.build_quality_value(new),
                ),
            )

        return attrs_changes


class ScholarPasswordEavSerializer(EavSerializer):
    EAV_FIELDS_MAPPER = {
        'serialized_encrypted_password': EavAttributeMap('account.scholar_password', default_processor),
    }

    def serialize(self, old, new, difference):
        if new:
            return self.build_change_scholar_password_queries(old, new, difference)
        elif old:
            return self.build_delete_scholar_password_queries(old, new, difference)
        raise NotImplementedError()

    def build_change_scholar_password_queries(self, old, new, difference):
        self.finish_password_set_if_required(old, new, difference)

        return self.build_change_fields_queries(
            changed_fields=difference.get_changed_fields(self.EAV_FIELDS_MAPPER),
            create=self.is_create(old, new),
            eav_mapper=self.EAV_FIELDS_MAPPER,
            new=new,
            old=old,
            uid=new.parent.uid,
        )

    def build_delete_scholar_password_queries(self, old, new, difference):
        return [self.build_delete_fields_query(self.EAV_FIELDS_MAPPER, old.parent.uid)]

    def finish_password_set_if_required(self, old, new, difference):
        # У создаваемого аккаунта нет UID'а пока его не сериализовали в БД.
        # А в момент сериализации пароля в БД UID известен, и мы точно можем
        # вычислить хеш пароля, если ещё этого не сделали.
        changed_fields = difference.get_changed_fields(self.EAV_FIELDS_MAPPER)

        if (
            self.is_create(old, new) and
            'serialized_encrypted_password' not in changed_fields and
            new.password_hash_provider
        ):
            hash_ = new.password_hash_provider.try_create_hash(uid=new.parent.uid)
            assert hash_.is_created
            new.encoding_version, new.encrypted_password = hash_.version, hash_.encrypted_password

            updated_difference = diff(old, new)
            difference.added.update(updated_difference.added)

            changed_fields = difference.get_changed_fields(self.EAV_FIELDS_MAPPER)
            assert 'serialized_encrypted_password' in changed_fields

    def is_create(self, old, new):
        return not old and new
