# -*- coding: utf-8 -*-
# from passport.backend.core.models.person import MOSCOW_TIMEZONE
from passport.backend.core.conf import settings
from passport.backend.core.differ.types import EmptyDiff
from passport.backend.core.serializers.eav.base import (
    EavAttributeMap,
    EavSerializer,
)
from passport.backend.core.serializers.eav.processors import (
    default_processor,
    JoinProcessors,
    unary_field_processor,
)
from passport.backend.core.types.gender import Gender
import pytz


MOSCOW_TIMEZONE = pytz.timezone('Europe/Moscow')  # FIXME


@JoinProcessors([default_processor])
def person_gender_processor(gender):
    return Gender.to_char(gender)


@JoinProcessors([default_processor])
def person_country_processor(code):
    code = code.lower()
    if code == settings.DEFAULT_COUNTRY:
        return None
    return code


@JoinProcessors([default_processor])
def person_birthday_processor(birthday):
    s = str(birthday)
    if s == '0000-00-00':
        return None
    return s


@JoinProcessors([default_processor])
def person_language_processor(language):
    if language == 'ru':
        return None
    return language


@JoinProcessors([default_processor])
def person_timezone_processor(timezone):
    # Вообще timezone должна быть pytz.tzinfo.BaseTzInfo,
    # но чтобы обезопасить себя проверяем и str
    if isinstance(timezone, pytz.tzinfo.BaseTzInfo) and timezone == MOSCOW_TIMEZONE:
        return
    elif timezone == str(MOSCOW_TIMEZONE):
        return
    return str(timezone)


class PersonEavSerializer(EavSerializer):
    EAV_FIELDS_MAPPER = {
        'firstname': EavAttributeMap('person.firstname', default_processor),
        'lastname': EavAttributeMap('person.lastname', default_processor),
        'gender': EavAttributeMap('person.gender', person_gender_processor),
        'birthday': EavAttributeMap('person.birthday', person_birthday_processor),
        'country': EavAttributeMap('person.country', person_country_processor),
        'city': EavAttributeMap('person.city', default_processor),
        'timezone': EavAttributeMap('person.timezone', person_timezone_processor),
        'language': EavAttributeMap('person.language', person_language_processor),

        'default_avatar': EavAttributeMap('avatar.default', default_processor),
        'display_name': EavAttributeMap('account.display_name', default_processor),
        'dont_use_displayname_as_public_name': EavAttributeMap(
            'person.dont_use_displayname_as_public_name',
            unary_field_processor,
        ),
        'show_fio_in_public_name': EavAttributeMap('person.show_fio_in_public_name', unary_field_processor),

        'firstname_global': EavAttributeMap('person.firstname_global', default_processor),
        'lastname_global': EavAttributeMap('person.lastname_global', default_processor),
    }

    def serialize(self, old, new, difference):
        if old is None:
            return self.create(new)
        return self.change(old, new, difference)

    def create(self, new):
        return self.build_change_fields_queries(
            self.EAV_FIELDS_MAPPER,
            new.parent.uid,
            None,
            new,
            self.EAV_FIELDS_MAPPER.keys(),
            True)

    def change(self, old, new, difference):
        if difference == EmptyDiff:
            return []

        changed_fields = difference.get_changed_fields(self.EAV_FIELDS_MAPPER)
        # PASSP-23375 при изменении display_name пытаемся удалить атрибут dont_use_displayname_as_public_name
        # чтобы ЧЯ опирался при вычислении public_display_name в том числе и на display_name, а не только на ФИО
        if (
            'display_name' in changed_fields and
            'dont_use_displayname_as_public_name' not in changed_fields and
            new.dont_use_displayname_as_public_name
        ):
            new.dont_use_displayname_as_public_name = False
            changed_fields |= {'dont_use_displayname_as_public_name'}

        return self.build_change_fields_queries(
            self.EAV_FIELDS_MAPPER,
            new.parent.uid,
            old,
            new,
            changed_fields,
        )
