# -*- coding: utf-8 -*-

from functools import partial
import logging

from passport.backend.core.builders.blackbox.parsers import (
    PHONE_OP_DEFAULT_VALUES,
    PHONE_OP_TYPE_IDX,
)
from passport.backend.core.db.query import split_query_and_callback
from passport.backend.core.db.runner import get_id_from_query_result
from passport.backend.core.db.schemas import phone_operations_table
from passport.backend.core.differ.types import (
    Diff,
    EmptyDiff,
)
from passport.backend.core.differ.utils import slice_diff
from passport.backend.core.serializers.eav.base import (
    EavAttributeMap,
    EavSerializer,
    Serializer,
)
from passport.backend.core.serializers.eav.exceptions import (
    EavDeletedObjectNotFound,
    EavUpdatedObjectNotFound,
)
from passport.backend.core.serializers.eav.extended_attributes import EAV_EXTENDED_ATTRIBUTES_PHONE_MAPPER
from passport.backend.core.serializers.eav.processors import default_processor
from passport.backend.core.serializers.eav.query import (
    EavDeletePhoneBindingCreatedQuery,
    EavDeletePhoneOperationCreatedQuery,
    EavInsertPhoneBindingCreatedQuery,
    EavInsertPhoneBindingHistoryCreatedQuery,
    EavInsertPhoneOperationCreatedQuery,
    EavUpdatePhoneBindingCreatedQuery,
    EavUpdatePhoneBindingHistoryQuery,
    EavUpdatePhoneOperationCreatedQuery,
)
from passport.backend.core.types.bit_vector.bit_vector import PhoneBindingsFlags
from passport.backend.core.undefined import Undefined
from passport.backend.utils.time import zero_datetime
from six import iteritems


log = logging.getLogger('passport.serializers.eav.phones')


class PhoneOperationSerializer(Serializer):

    # Значения этих полей будут передаваться не абсолютным значением, а
    # относительно текущего значения.
    # Например, code_checks_count = operations_table.code_checks_count + 1
    increment_fields = ['code_checks_count']

    @staticmethod
    def _get_operation_data(operation):
        operation_data = dict(operation)
        operation_data['phone_id'] = operation.phone_id

        # Заменим все Undefined на None.
        operation_data = {k: None if v == Undefined else v
                          for k, v in iteritems(operation_data)}

        operation_data['type'] = PHONE_OP_TYPE_IDX[operation_data['type']]

        # Проставим значения по умолчанию.
        for k, v in iteritems(PHONE_OP_DEFAULT_VALUES):
            if operation_data[k] is None:
                operation_data[k] = v() if callable(v) else v

        flags = operation_data['flags']
        if flags is not None:
            operation_data['flags'] = int(flags)

        return operation_data

    def serialize(self, old, new, difference):
        if difference == EmptyDiff or (not old and not new):
            return

        phone = old.parent if old else new.parent
        uid = phone.uid

        if new:
            if old and new.id == old.id:
                if not old.id:
                    raise ValueError('Operation identifier has to be set for update operation.')
                # Обновление существующей операции.
                new_operation_data = self._get_operation_data(new)
                old_operation_data = self._get_operation_data(old)

                # Отсеем поля, которые не изменялись.
                update_operation_data = dict(filter(
                    lambda el: old_operation_data[el[0]] != new_operation_data[el[0]],
                    iteritems(new_operation_data),
                ))

                if not update_operation_data:
                    # Ничего не изменено.
                    # Сюда попадем, если только меняли значение по умолчанию на None или наоборот.
                    return

                for key in self.increment_fields:
                    if key not in update_operation_data:
                        continue
                    value_diff = update_operation_data[key] - getattr(old, key)
                    update_operation_data[key] = getattr(phone_operations_table.c, key) + value_diff

                yield EavUpdatePhoneOperationCreatedQuery(uid, new_operation_data['id'], update_operation_data), None
            else:
                if old:
                    yield EavDeletePhoneOperationCreatedQuery(uid, old.id), None
                # Создание перации
                new_operation_data = self._get_operation_data(new)
                yield (
                    EavInsertPhoneOperationCreatedQuery(uid, new_operation_data),
                    lambda result: setattr(new, 'id', get_id_from_query_result(result)),
                )
        else:
            yield (
                EavDeletePhoneOperationCreatedQuery(uid, old.id),
                partial(
                    PhoneOperationSerializer._on_phone_operation_deleted,
                    operation_id=old.id,
                ),
            )

    @classmethod
    def _on_phone_operation_deleted(cls, result, operation_id):
        if result.rowcount != 1:
            log.error('Delete non-existent phone operation: operation_id=%d', operation_id)
            raise EavDeletedObjectNotFound()


class PhoneBindingsSerializer(Serializer):
    def serialize(self, old, new):
        old_binding = old and old.binding
        new_binding = new and new.binding

        queries = []

        if new_binding:
            if not old_binding and new_binding.time:
                queries = self._create_bound_phone(new)
            elif not old_binding and not new_binding.time:
                queries = self._create_unbound_phone(new)
            elif (old_binding.time != new_binding.time and
                  new_binding.time):
                queries = self._rebind_phone(new, old)
            elif old_binding.time and not new_binding.time:
                queries = self._unbind_phone(new, old)
            elif old_binding.should_ignore_binding_limit != new_binding.should_ignore_binding_limit:
                queries = self._change_flags(new)
        elif old_binding:
            uid = old.parent.parent.uid
            queries = [EavDeletePhoneBindingCreatedQuery(uid, old.id)]

        if new_binding and old_binding and new.number != old.number:
            queries += self._change_phone_number(new, old)

        for query in queries:
            yield query

    def _create_bound_phone(self, new):
        uid = new.parent.parent.uid

        flags = PhoneBindingsFlags()
        flags.should_ignore_binding_limit = new.binding.should_ignore_binding_limit

        yield EavInsertPhoneBindingCreatedQuery(
            uid=uid,
            number=int(new.number),
            phone_id=new.id,
            bound=self._none_to_zero_datetime(new.binding.time),
            flags=int(flags),
        )
        yield EavInsertPhoneBindingHistoryCreatedQuery(
            uid=uid,
            number=int(new.number),
            bound=self._none_to_zero_datetime(new.binding.time),
        )

    def _create_unbound_phone(self, new):
        uid = new.parent.parent.uid

        flags = PhoneBindingsFlags()
        flags.should_ignore_binding_limit = new.binding.should_ignore_binding_limit

        yield EavInsertPhoneBindingCreatedQuery(
            uid=uid,
            number=int(new.number),
            phone_id=new.id,
            bound=zero_datetime,
            flags=int(flags),
        )

    def _rebind_phone(self, new, old):
        uid = new.parent.parent.uid

        flags = PhoneBindingsFlags()
        flags.should_ignore_binding_limit = new.binding.should_ignore_binding_limit

        yield (
            EavUpdatePhoneBindingCreatedQuery(
                uid=uid,
                phone_id=new.id,
                old_bound=self._none_to_zero_datetime(old.binding.time),
                bound=self._none_to_zero_datetime(new.binding.time),
                flags=int(flags),
            ),
            partial(
                self._on_phone_binding_updated,
                phone_id=new.id,
                bound=old.binding.time,
            ),
        )
        yield EavInsertPhoneBindingHistoryCreatedQuery(
            uid=uid,
            number=int(new.number),
            bound=self._none_to_zero_datetime(new.binding.time),
        )

    def _unbind_phone(self, new, old):
        uid = new.parent.parent.uid

        yield EavUpdatePhoneBindingCreatedQuery(
            uid=uid,
            phone_id=new.id,
            old_bound=self._none_to_zero_datetime(old.binding.time),
            bound=zero_datetime,
            flags=int(PhoneBindingsFlags()),
        )

    def _change_flags(self, new):
        uid = new.parent.parent.uid

        flags = PhoneBindingsFlags()
        flags.should_ignore_binding_limit = new.binding.should_ignore_binding_limit

        yield EavUpdatePhoneBindingCreatedQuery(
            uid=uid,
            phone_id=new.id,
            flags=int(flags),
        )

    def _none_to_zero_datetime(self, time):
        if time is None:
            time = zero_datetime
        return time

    @classmethod
    def _on_phone_binding_updated(cls, result, phone_id, bound):
        if result.rowcount != 1:
            log.error(
                'Update non-existent phone binding: phone_id=%d, bound=%s',
                phone_id,
                bound,
            )
            raise EavUpdatedObjectNotFound()

    def _change_phone_number(self, new, old):
        uid = new.parent.parent.uid

        flags = PhoneBindingsFlags()
        flags.should_ignore_binding_limit = new.binding.should_ignore_binding_limit

        yield EavUpdatePhoneBindingCreatedQuery(uid=uid, phone_id=new.id, number=int(new.number), old_number=int(old.number))

        if old.binding and old.binding.time:
            # Номер раньше был привязан, а значит об этом есть запись в
            # истории, поэтому нужно в ней тоже поменять значение номера.
            yield EavUpdatePhoneBindingHistoryQuery(
                uid=uid,
                number=int(new.number),
                old_number=int(old.number),
                old_bound=old.binding.time,
            )


class PhoneEavSerializer(EavSerializer):
    def serialize(self, old, new, difference):
        changed_fields = difference.get_changed_fields(EAV_EXTENDED_ATTRIBUTES_PHONE_MAPPER)

        phones = old.parent if old else new.parent

        for query in self.build_extended_attribute_queries(
            EAV_EXTENDED_ATTRIBUTES_PHONE_MAPPER,
            phones.parent.uid,
            old,
            new,
            changed_fields,
            create=old is None,
        ):
            yield query

        for query in PhoneBindingsSerializer().serialize(old, new):
            yield query

        for query in PhoneOperationSerializer().serialize(
            old.operation if old else None,
            new.operation if new else None,
            slice_diff(difference, 'operation'),
        ):
            yield query


class PhonesEavSerializer(EavSerializer):
    EAV_FIELDS_MAPPER = {
        'default_id': EavAttributeMap('phones.default', default_processor),
        'secure_id': EavAttributeMap('phones.secure', default_processor),
    }

    def serialize(self, old, new, difference):
        if difference == EmptyDiff:
            return

        phones_diff = slice_diff(difference, '_phones')

        # пройдемся по всем измененным телефонам
        queries = []
        for phone_id in phones_diff.get_changed_fields():
            old_phone = old.by_id(phone_id, assert_exists=False) if old else None
            new_phone = new.by_id(phone_id, assert_exists=False) if new else None

            if new_phone is None:
                # Если телефон полностью удален - удалим все его атрибуты и операцию.
                deleted = {k: v for k, v in iteritems(dict(old_phone)) if v not in [None, Undefined]}
                phone_diff = Diff({}, {}, deleted)
            else:
                phone_diff = slice_diff(phones_diff, phone_id)

            queries.extend(list(PhoneEavSerializer().serialize(old_phone, new_phone, phone_diff)))

        for query in self._sort(queries):
            yield query

        # Сериализация атрибутов аккаунта.
        for query in self.build_change_fields_queries(
            self.EAV_FIELDS_MAPPER,
            new.parent.uid if new else old.parent.uid,
            old,
            new,
            difference.get_changed_fields(self.EAV_FIELDS_MAPPER),
        ):
            yield query

    def _sort(self, queries):
        # Перед тем как создавать одни операции, нужно удалить другие,
        # чтобы не возникло конфликтов.
        for index in range(len(queries)):
            query, callback = split_query_and_callback(queries[index])
            if isinstance(query, (EavDeletePhoneOperationCreatedQuery, EavDeletePhoneBindingCreatedQuery)):
                del queries[index]
                queries.insert(0, (query, callback))
        return queries


def serialize_phones(old, new, difference):
    for query in PhonesEavSerializer().serialize(
        old.phones if old else None,
        new.phones if new else None,
        slice_diff(difference, 'phones'),
    ):
        yield query
