# -*- coding: utf-8 -*-

from google.protobuf import json_format
from passport.backend.utils.string import snake_case_to_camel_case
from passport.backend.utils.time import datetime_to_unixtime
from passport.protobuf.billing_features.billing_features_pb2 import (
    BillingFeatures,
    FeatureAttributes,
)
from passport.protobuf.plus_subscriber_state.plus_subscriber_state_pb2 import PlusSubscriberState


class JoinProcessors(object):
    def __init__(self, processors, stop_on=None):
        self.processors = processors
        self.stop_on = stop_on

    def __call__(self, fn):
        def wrapped(value):
            for processor in self.processors:
                value = processor(value)
                if value == self.stop_on:
                    return value
            return fn(value)
        return wrapped


def default_processor(value):
    if value in ['', None]:
        return
    return value


def as_is_processor(value):
    return value


@JoinProcessors([default_processor])
def boolean_processor(value):
    if int(value):
        return '1'


def nullable_boolean_processor(value):
    return {True: '1', False: '0'}.get(value)


def str_processor(value):
    return str(value)


@JoinProcessors([default_processor])
def unary_field_processor(value):
    if value:
        return '1'


@JoinProcessors([default_processor])
def datetime_processor(datetime):
    timestamp = int(datetime_to_unixtime(datetime))
    if timestamp > 1:
        return timestamp


def list_processor(value):
    if value:
        return ';'.join(value)


def billing_features_processor(value):
    if value:
        bf = BillingFeatures()

        for feature_name, attributes in value.items():
            fa = FeatureAttributes()
            for option_name, option_value in attributes.items():
                setattr(fa, snake_case_to_camel_case(option_name), option_value)
            bf.Features[feature_name].CopyFrom(fa)
        return bf.SerializeToString()


def plus_subscriber_state_processor(value):
    if value:
        pss = PlusSubscriberState()
        json_format.Parse(value, pss)
        return pss.SerializeToString()
