# -*- coding: utf-8 -*-
from passport.backend.core.db.query import (
    DbQuery,
    DbQueryForUID,
)
from passport.backend.core.db.schemas import (
    account_deletion_operations_table,
    aliases_table,
    attributes_table,
    device_public_key_table,
    email_bindings_table,
    email_id_table,
    extended_attributes_table,
    family_members_table,
    passman_recovery_keys_table,
    password_history_eav_table,
    pddsuid_table,
    pdduid_table,
    phone_bindings_history_table,
    phone_bindings_table,
    phone_id_table,
    phone_operations_table,
    removed_aliases_table,
    reserved_logins_table,
    suid2_table,
    suid_table,
    totp_secret_id_table,
    uid_table,
    webauthn_credential_id_table,
    webauthn_credentials_table,
    yakey_backups_table,
)
from passport.backend.core.db.utils import (
    insert_with_on_duplicate_key_append,
    insert_with_on_duplicate_key_increment,
    insert_with_on_duplicate_key_update,
    insert_with_on_duplicate_key_update_if_equals,
    with_ignore_prefix,
)
from passport.backend.core.eav_type_mapping import (
    ALIAS_NAME_TO_TYPE,
    EXTENDED_ATTRIBUTES_EMAIL_TYPE,
    EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
)
from passport.backend.core.serializers.domain import pdd_alias_to_removed_alias_sql_func
from passport.backend.utils.common import remove_none_values
from passport.backend.utils.string import smart_bytes
from sqlalchemy.sql import select
from sqlalchemy.sql.expression import (
    and_,
    or_,
)


SUID_TABLES = {
    2: suid2_table,
}


class BaseEavIncrementQuery(DbQuery):
    TABLE = None

    def get_table(self):
        return self.TABLE

    def to_query(self):
        t = self.get_table()
        return t.insert()

    def __repr__(self):
        return '<%s: table: %s>' % (self.__class__.__name__, self.get_table())


class EavIncrementQuery(BaseEavIncrementQuery):
    PDD_TABLE = None

    def __init__(self, is_pdd=False):
        self.is_pdd = is_pdd

    def get_table(self):
        return self.PDD_TABLE if self.is_pdd else self.TABLE

    def __repr__(self):
        return '<%s: is_pdd: %s | table: %s>' % (self.__class__.__name__, self.is_pdd, self.get_table())


class EavUidIncrementQuery(EavIncrementQuery):
    TABLE = uid_table
    PDD_TABLE = pdduid_table


class EavPhoneIdIncrementQuery(EavIncrementQuery):
    TABLE = phone_id_table


class EavEmailIdIncrementQuery(EavIncrementQuery):
    TABLE = email_id_table


class EavSuidIncrementQuery(EavIncrementQuery):
    TABLE = suid_table
    PDD_TABLE = pddsuid_table


class EavTotpSecretIdIncrementQuery(BaseEavIncrementQuery):
    TABLE = totp_secret_id_table


class EavWebauthnCredentialIdIncrementQuery(EavIncrementQuery):
    TABLE = webauthn_credential_id_table


class EavAdditiveQueryForUID(DbQueryForUID):

    can_be_joined = True

    def __add__(self, other):
        cls = self.__class__
        if type(other) != cls:
            raise TypeError('unsupported operand type(s) for +: \'%s\' and \'%s\'' % (cls.__name__,
                                                                                      other.__class__.__name__))
        if self.uid != other.uid:
            raise ValueError('operands could not have different uids')
        return cls(self.uid, self.query_params() + other.query_params())


class EavSuidQuery(DbQueryForUID):
    def __init__(self, uid, sid):
        super(EavSuidQuery, self).__init__(uid)
        self.sid = sid

    def get_table(self):
        return SUID_TABLES[self.sid]


class EavInsertSuidQuery(EavSuidQuery):
    def __init__(self, uid, sid, suid):
        super(EavInsertSuidQuery, self).__init__(uid, sid)
        self.suid = suid

    def query_params(self):
        return {'suid': self.suid, 'sid': self.sid}

    def to_query(self):
        t = self.get_table()
        return t.insert().values({'suid': self.suid, 'uid': self.uid})

    def __repr__(self):
        return '<%s: uid: %s | sid: %s | suid: %s>' % (self.__class__.__name__, self.uid, self.sid, self.suid)


class EavDeleteSuidQuery(EavSuidQuery):
    def __init__(self, uid, sid):
        super(EavDeleteSuidQuery, self).__init__(uid, sid)

    def query_params(self):
        return {'sid': self.sid}

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)

    def __repr__(self):
        return '<%s: uid: %s | sid: %s>' % (self.__class__.__name__, self.uid, self.sid)


class EavShardedQuery(DbQueryForUID):
    _SHARDED = True


class EavShardedAdditiveQueryForUID(EavAdditiveQueryForUID):
    _SHARDED = True


class EavPhoneBindingCreatedQuery(EavShardedQuery):
    def get_table(self):
        return phone_bindings_table


class EavWebauthnCredentialToUidQuery(DbQueryForUID):
    def get_table(self):
        return webauthn_credentials_table


class EavInsertWebauthnCredentialToUidQuery(EavWebauthnCredentialToUidQuery):
    def __init__(self, uid, credential_external_id):
        super(EavInsertWebauthnCredentialToUidQuery, self).__init__(uid)
        self.credential_external_id = credential_external_id

    def query_params(self):
        return [self.credential_external_id]

    def to_query(self):
        t = self.get_table()
        return t.insert().values([
            {
                'uid': self.uid,
                'credential_id': smart_bytes(self.credential_external_id),
            },
        ])


class EavDeleteWebauthnCredentialToUidQuery(EavWebauthnCredentialToUidQuery):
    def __init__(self, uid, credential_external_id):
        super(EavDeleteWebauthnCredentialToUidQuery, self).__init__(uid)
        self.credential_external_id = credential_external_id

    def query_params(self):
        return [self.credential_external_id]

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(
                t.c.uid == self.uid,
                t.c.credential_id == smart_bytes(self.credential_external_id),
            ),
        )


class EavDeleteAllWebauthnCredentialToUidQuery(EavWebauthnCredentialToUidQuery):
    def query_params(self):
        return []

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)


class EavEmailBindingQuery(EavShardedQuery):
    def get_table(self):
        return email_bindings_table


class EavInsertEmailBindingQuery(EavEmailBindingQuery):
    def __init__(self, uid, address, email_id, bound_at):
        super(EavInsertEmailBindingQuery, self).__init__(uid)
        self.email_id = email_id
        self.address = address.lower()
        self.bound_at = bound_at

    def query_params(self):
        return [self.address, self.email_id, self.bound_at]

    def to_query(self):
        t = self.get_table()
        return t.insert().values([
            {
                'uid': self.uid,
                'email_id': self.email_id,
                'address': self.address,
                'bound': self.bound_at,
            },
        ])


class EavDeleteEmailBindingQuery(EavEmailBindingQuery):
    def __init__(self, uid, email_id):
        super(EavDeleteEmailBindingQuery, self).__init__(uid)
        self.email_id = email_id

    def query_params(self):
        return [self.email_id]

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(
                t.c.uid == self.uid,
                t.c.email_id == self.email_id,
            ),
        )


class EavDeleteAllEmailBindingsQuery(EavEmailBindingQuery):

    def query_params(self):
        return []

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)


class EavUpdateEmailBindingBound(EavEmailBindingQuery):
    def __init__(self, uid, email_id, bound_at):
        super(EavUpdateEmailBindingBound, self).__init__(uid)
        self.email_id = email_id
        self.bound_at = bound_at

    def query_params(self):
        return [self.email_id, self.bound_at]

    def to_query(self):
        t = self.get_table()
        return t.update().values(
            bound=self.bound_at,
        ).where(
            and_(t.c.email_id == self.email_id),
        )


class EavInsertPhoneBindingCreatedQuery(EavPhoneBindingCreatedQuery):
    def __init__(self, uid, **data):
        super(EavInsertPhoneBindingCreatedQuery, self).__init__(uid)
        self.data = data

    def to_query(self):
        return self.get_table().insert().values(uid=self.uid, **self.data)


class EavDeletePhoneBindingCreatedQuery(EavPhoneBindingCreatedQuery):
    def __init__(self, uid, phone_id):
        super(EavDeletePhoneBindingCreatedQuery, self).__init__(uid)
        self.phone_id = phone_id

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(t.c.uid == self.uid, t.c.phone_id == self.phone_id),
        )


class EavUpdatePhoneBindingCreatedQuery(EavPhoneBindingCreatedQuery):
    def __init__(self, uid, phone_id, old_bound=None, old_number=None, **kwargs):
        super(EavUpdatePhoneBindingCreatedQuery, self).__init__(uid)
        self._phone_id = phone_id
        self._kwargs = kwargs
        self._old_bound = old_bound
        self._old_number = old_number

    def to_query(self):
        t = self.get_table()
        conds = [
            t.c.uid == self.uid,
            t.c.phone_id == self._phone_id,
        ]
        if self._old_bound is not None:
            conds.append(t.c.bound == self._old_bound)
        if self._old_number is not None:
            conds.append(t.c.number == self._old_number)
        return t.update().values(**self._kwargs).where(and_(*conds))


class EavPhoneBindingHistoryCreatedQuery(EavShardedQuery):
    def get_table(self):
        return phone_bindings_history_table


class EavInsertPhoneBindingHistoryCreatedQuery(EavPhoneBindingHistoryCreatedQuery):
    def __init__(self, uid, **data):
        super(EavInsertPhoneBindingHistoryCreatedQuery, self).__init__(uid)
        self.data = data

    def to_query(self):
        t = self.get_table()
        return t.insert().values(uid=self.uid, **self.data)


class EavUpdatePhoneBindingHistoryQuery(EavPhoneBindingHistoryCreatedQuery):
    def __init__(self, uid, old_number, old_bound, **kwargs):
        super(EavUpdatePhoneBindingHistoryQuery, self).__init__(uid)
        self._kwargs = kwargs
        self._old_number = old_number
        self._old_bound = old_bound

    def to_query(self):
        t = self.get_table()
        conds = [
            t.c.uid == self.uid,
        ]
        if self._old_bound is not None:
            conds.append(t.c.bound == self._old_bound)
        if self._old_number is not None:
            conds.append(t.c.number == self._old_number)
        return (
            t.update()
            .values(**self._kwargs)
            .where(and_(*conds))
        )


class EavPhoneOperationCreatedQuery(EavShardedQuery):
    """
    Отличительная особенность - запрос составляется для одной строчки таблицы за раз.
    """
    def get_table(self):
        return phone_operations_table

    def __repr__(self):
        return 'Instance of %s' % self.__class__.__name__  # pragma: no cover


class EavInsertPhoneOperationCreatedQuery(EavPhoneOperationCreatedQuery):
    def __init__(self, uid, operation_data):
        super(EavInsertPhoneOperationCreatedQuery, self).__init__(uid)
        self.operation_data = operation_data

    def to_query(self):
        t = self.get_table()

        data = remove_none_values(self.operation_data)

        return t.insert().values(
            uid=self.uid,
            **data
        )


class EavUpdatePhoneOperationCreatedQuery(EavPhoneOperationCreatedQuery):
    def __init__(self, uid, operationid, operation_data):
        super(EavUpdatePhoneOperationCreatedQuery, self).__init__(uid)
        self.operation_data = operation_data
        self.operationid = operationid

    def to_query(self):
        t = self.get_table()

        return t.update().values(
            uid=self.uid,
            **self.operation_data
        ).where(and_(t.c.uid == self.uid, t.c.id == self.operationid))


class EavDeletePhoneOperationCreatedQuery(EavPhoneOperationCreatedQuery):
    def __init__(self, uid, operationid):
        super(EavDeletePhoneOperationCreatedQuery, self).__init__(uid)
        self.operationid = operationid

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(t.c.uid == self.uid, t.c.id == self.operationid),
        )


class EavPasswordHistoryQuery(EavShardedAdditiveQueryForUID):
    def get_table(self):
        return password_history_eav_table


class EavDeleteFromPasswordHistoryQuery(EavPasswordHistoryQuery):
    def query_params(self):
        return []

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)


class EavInsertPasswordHistoryQuery(EavPasswordHistoryQuery):
    def __init__(self, uid, password_history):
        """
        Принимает список обновлений паролей: [(update_datetime, encrypted_password, reason)]
        """
        super(EavInsertPasswordHistoryQuery, self).__init__(uid)
        self.password_history = password_history

    def query_params(self):
        return self.password_history

    def to_query(self):
        t = self.get_table()
        return t.insert().values([
            {'uid': self.uid,
             'ts': update_datetime,
             'encrypted_password': smart_bytes(encrypted_password),
             'reason': reason}
            for update_datetime, encrypted_password, reason in self.password_history])


class EavAttributeQuery(EavShardedAdditiveQueryForUID):
    def get_table(self):
        return attributes_table


class EavInsertAttributeQuery(EavAttributeQuery):
    def __init__(self, uid, types_values):
        super(EavInsertAttributeQuery, self).__init__(uid)
        self.types_values = types_values

    def query_params(self):
        return self.types_values

    def to_query(self):
        t = self.get_table()
        return t.insert().values([
            {
                'uid': self.uid,
                'type': eav_type,
                'value': smart_bytes(value),
            } for eav_type, value in self.types_values
        ])


class EavInsertAttributeWithOnDuplicateKeyUpdateQuery(EavInsertAttributeQuery):
    def to_query(self):
        t = self.get_table()
        return insert_with_on_duplicate_key_update(t, ['value']).values([
            {
                'uid': self.uid,
                'type': eav_type,
                'value': smart_bytes(value),
            } for eav_type, value in self.types_values
        ])


class EavInsertAttributeWithOnDuplicateKeyIfValueEqualsUpdateQuery(EavInsertAttributeQuery):
    can_be_joined = False

    def to_query(self):
        t = self.get_table()
        if len(self.types_values) != 1:
            raise ValueError('Only one row can be inserted with if_equals')
        eav_type, (exp_val, new_value) = self.types_values[0]
        return insert_with_on_duplicate_key_update_if_equals(t, ['value'], 'value', exp_val).values([
            {
                'uid': self.uid,
                'type': eav_type,
                'value': smart_bytes(new_value),
            },
        ])


class EavInsertAttributeWithOnDuplicateKeyAppendQuery(EavInsertAttributeQuery):
    def to_query(self):
        t = self.get_table()
        return insert_with_on_duplicate_key_append(t, ['value']).values([
            {
                'uid': self.uid,
                'type': eav_type,
                'value': smart_bytes(value),
            } for eav_type, value in self.types_values
        ])


class EavInsertAttributeWithOnDuplicateKeyIncrementQuery(EavInsertAttributeQuery):
    def to_query(self):
        t = self.get_table()
        return insert_with_on_duplicate_key_increment(t, 'value').values([
            {
                'uid': self.uid,
                'type': eav_type,
                'value': smart_bytes(value),
            } for eav_type, value in self.types_values
        ])


class EavDeleteAttributeQuery(EavAttributeQuery):
    def __init__(self, uid, types):
        super(EavDeleteAttributeQuery, self).__init__(uid)
        self.types = types

    def query_params(self):
        return self.types

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(
                t.c.uid == self.uid,
                t.c.type.in_(sorted(self.types)),
            ),
        )


class EavDeleteAllAttributesQuery(EavAttributeQuery):
    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)

    def query_params(self):
        return []


class EavExtendedAttributeQuery(EavShardedAdditiveQueryForUID):
    def get_table(self):
        return extended_attributes_table


class EavDeleteEmailQuery(EavExtendedAttributeQuery):
    def __init__(self, uid, email_id):
        self.uid = uid
        self.email_id = email_id
        super(EavDeleteEmailQuery, self).__init__(uid)

    def query_params(self):
        return [self.email_id]

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(
                t.c.uid == self.uid,
                t.c.entity_id == self.email_id,
                t.c.entity_type == EXTENDED_ATTRIBUTES_EMAIL_TYPE,
            ),
        )


class EavDeleteAllEmailsQuery(EavExtendedAttributeQuery):

    def query_params(self):
        return []

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(
                t.c.uid == self.uid,
                t.c.entity_type == EXTENDED_ATTRIBUTES_EMAIL_TYPE,
            ),
        )


class EavDeleteWebauthnCredentialQuery(EavExtendedAttributeQuery):
    def __init__(self, uid, cred_id):
        self.uid = uid
        self.cred_id = cred_id
        super(EavDeleteWebauthnCredentialQuery, self).__init__(uid)

    def query_params(self):
        return [self.cred_id]

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(
                t.c.uid == self.uid,
                t.c.entity_id == self.cred_id,
                t.c.entity_type == EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
            ),
        )


class EavDeleteAllWebauthnCredentialsQuery(EavExtendedAttributeQuery):
    def query_params(self):
        return []

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(
                t.c.uid == self.uid,
                t.c.entity_type == EXTENDED_ATTRIBUTES_WEBAUTHN_TYPE,
            ),
        )


class EavInsertExtendedAttributeWithOnDuplicateKeyQuery(EavExtendedAttributeQuery):
    def __init__(self, uid, ext_attributes_data):
        super(EavExtendedAttributeQuery, self).__init__(uid)
        self.ext_attributes_data = ext_attributes_data

    def query_params(self):
        return self.ext_attributes_data

    def to_query(self):
        t = self.get_table()
        for item in self.ext_attributes_data:
            item['value'] = smart_bytes(item['value'])
        return insert_with_on_duplicate_key_update(t, ['value']).values([
            dict(uid=self.uid, **item) for item in self.ext_attributes_data
        ])


class EavDeleteAllExtendedAttributesQuery(EavExtendedAttributeQuery):
    def query_params(self):
        return [self.uid]

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)


class EavDeleteExtendedAttributeQuery(EavExtendedAttributeQuery):
    def __init__(self, uid, ext_attrs_data):
        super(EavDeleteExtendedAttributeQuery, self).__init__(uid)
        self.ext_attrs_data = ext_attrs_data

    def query_params(self):
        return self.ext_attrs_data

    def to_query(self):
        t = self.get_table()
        del_conditions = (and_(
            t.c.entity_type == item['entity_type'],
            t.c.entity_id == item['entity_id'],
            t.c.type == item['type'],
        ) for item in sorted(self.ext_attrs_data, key=lambda d: d['type']))

        return t.delete().where(
            and_(t.c.uid == self.uid, or_(*del_conditions)),
        )


class EavAliasQuery(EavAdditiveQueryForUID):
    def get_table(self):
        return aliases_table


class EavInsertAliasQuery(EavAliasQuery):
    def __init__(self, uid, types_values):
        super(EavInsertAliasQuery, self).__init__(uid)
        self.types_values = types_values

    def query_params(self):
        return self.types_values

    def to_query(self):
        t = self.get_table()
        return t.insert().values([
            {
                'uid': self.uid,
                'type': alias_type,
                'value': smart_bytes(value),
                'surrogate_type': smart_bytes(surrogate),
            } for alias_type, value, surrogate in self.types_values])

    def __repr__(self):
        return '<%s: uid: %s | types_values: %s>' % (self.__class__.__name__, self.uid, self.types_values)


class EavUpdateAliasQuery(EavAliasQuery):
    def __init__(self, uid, alias_type, alias, surrogate_type):
        super(EavUpdateAliasQuery, self).__init__(uid)
        self.alias_type = alias_type
        self.alias = alias
        self.surrogate_type = surrogate_type

    def to_query(self):
        t = self.get_table()
        return t.update().values(
            value=smart_bytes(self.alias),
        ).where(and_(
            t.c.uid == self.uid,
            t.c.type == self.alias_type,
            t.c.surrogate_type == smart_bytes(self.surrogate_type),
        ))

    def __repr__(self):
        return '<%s: uid: %s | types_values: %s>' % (
            self.__class__.__name__,
            self.uid,
            (self.alias_type, self.alias, self.surrogate_type),
        )


class EavDeleteAliasQuery(EavAliasQuery):
    def __init__(self, uid, types):
        super(EavDeleteAliasQuery, self).__init__(uid)
        self.types = types

    def query_params(self):
        return self.types

    def to_query(self):
        t = self.get_table()
        return t.delete().where(
            and_(
                t.c.uid == self.uid,
                t.c.type.in_(self.types),
            ),
        )

    def __repr__(self):
        return '<%s: uid: %s | types: %s>' % (self.__class__.__name__, self.uid, self.types)


class MassInsertAliasesIntoRemovedAliasesQuery(EavAdditiveQueryForUID):
    def __init__(self, uid, types):
        super(MassInsertAliasesIntoRemovedAliasesQuery, self).__init__(uid)
        self.types = types

    def query_params(self):
        return self.types

    def get_table(self):
        return removed_aliases_table

    def to_query(self):
        at = aliases_table
        query = self.get_table().insert().from_select(
            self.get_table().c.keys(),
            select([at.c.uid, at.c.type, at.c.value]).where(
                and_(
                    at.c.uid == self.uid,
                    at.c.type.in_(self.types),
                ),
            ),
        )
        return with_ignore_prefix(query)

    def __repr__(self):
        return '<%s: uid: %s | types: %s>' % (self.__class__.__name__, self.uid, self.types)


class InsertAliasesWithValueIntoRemovedAliasesQuery(EavAdditiveQueryForUID):
    def __init__(self, uid, types_values):
        super(InsertAliasesWithValueIntoRemovedAliasesQuery, self).__init__(uid)
        self.types_values = types_values

    def query_params(self):
        return self.types_values

    def get_table(self):
        return removed_aliases_table

    def to_query(self):
        table = self.get_table()
        query = table.insert().values([
            (self.uid, alias_type, smart_bytes(value))
            for alias_type, value
            in self.types_values
        ])

        return with_ignore_prefix(query)

    def __repr__(self):
        return '<%s: uid: %s | types_values: %s>' % (
            self.__class__.__name__,
            self.uid,
            self.types_values,
        )


class InsertAllPddAliasesFromAccountIntoRemovedAliasesQuery(EavAdditiveQueryForUID):
    def __init__(self, uid, domain):
        super(InsertAllPddAliasesFromAccountIntoRemovedAliasesQuery, self).__init__(uid)
        self.domain = domain
        self.types = [
            ALIAS_NAME_TO_TYPE['pdd'],
            ALIAS_NAME_TO_TYPE['pddalias'],
        ]

    def query_params(self):
        return [self.domain.domain]

    def get_table(self):
        return removed_aliases_table

    def to_query(self):
        at = aliases_table
        query = self.get_table().insert().from_select(
            self.get_table().c.keys(),
            select([
                at.c.uid,
                at.c.type,
                pdd_alias_to_removed_alias_sql_func(
                    domain_name=self.domain.domain,
                    alias_column=at.c.value,
                ),
            ]).where(
                and_(
                    at.c.uid == self.uid,
                    at.c.type.in_(self.types),
                ),
            ),
        )
        return with_ignore_prefix(query)

    def __repr__(self):
        return '<%s: uid: %s | types: %s>' % (self.__class__.__name__, self.uid, self.types)


class InsertLoginIntoReservedQuery(DbQuery):
    def __init__(self, login, reserved_till):
        super(InsertLoginIntoReservedQuery, self).__init__()
        self.login = login
        self.reserved_till = reserved_till

    def get_table(self):
        return reserved_logins_table

    def to_query(self):
        t = self.get_table()
        query = insert_with_on_duplicate_key_update(t, ['free_ts'])
        return query.values({'login': smart_bytes(self.login), 'free_ts': self.reserved_till})

    def __repr__(self):
        return '<%s: login: %s, timestamp: %s>' % (self.__class__.__name__, self.login, self.timestamp)


class DeleteAliasWithValueQuery(EavAliasQuery):
    def __init__(self, uid, types_values):
        super(DeleteAliasWithValueQuery, self).__init__(uid)
        self.types_values = types_values

    def query_params(self):
        return self.types_values

    def to_query(self):
        t = self.get_table()
        filters = [
            and_(
                t.c.type == alias_type,
                t.c.value == smart_bytes(value),
            ) for alias_type, value in self.types_values
        ]
        return t.delete().where(and_(t.c.uid == self.uid, or_(*filters)))

    def __repr__(self):
        return '<%s: uid: %s | types_values: %s>' % (
            self.__class__.__name__,
            self.uid,
            self.types_values,
        )


class YaKeyBackupCreateQuery(DbQuery):
    def __init__(self, phone_number, backup, updated, device_name):
        super(YaKeyBackupCreateQuery, self).__init__()
        self.phone_number = phone_number
        self.backup = backup
        self.updated = updated
        self.device_name = device_name

    def get_table(self):
        return yakey_backups_table

    def query_params(self):
        return {
            'phone_number': self.phone_number,
            'backup': smart_bytes(self.backup),
            'updated': self.updated,
            'device_name': self.device_name,
        }

    def to_query(self):
        return self.get_table().insert().values(self.query_params())


class YaKeyBackupUpdateQuery(DbQuery):
    def __init__(self, phone_number, backup, updated, previous_update, device_name):
        super(YaKeyBackupUpdateQuery, self).__init__()
        self.phone_number = phone_number
        self.backup = backup
        self.updated = updated
        self.previous_update = previous_update
        self.device_name = device_name

    def get_table(self):
        return yakey_backups_table

    def query_params(self):
        return {
            'phone_number': self.phone_number,
            'backup': smart_bytes(self.backup),
            'updated': self.updated,
            'device_name': self.device_name,
        }

    def to_query(self):
        return insert_with_on_duplicate_key_update_if_equals(
            table=self.get_table(),
            args=['backup', 'device_name', 'updated'],
            key_name='updated',
            key_value=self.previous_update,
        ).values(self.query_params())


class YaKeyBackupDeleteQuery(DbQuery):
    def __init__(self, phone_number):
        super(YaKeyBackupDeleteQuery, self).__init__()
        self.phone_number = phone_number

    def get_table(self):
        return yakey_backups_table

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.phone_number == self.phone_number)


class AccountDeletionOperationCreateQuery(EavShardedQuery):
    def __init__(self, uid, started_at):
        super(AccountDeletionOperationCreateQuery, self).__init__(uid)
        self.started_at = started_at

    def get_table(self):
        return account_deletion_operations_table

    def to_query(self):
        return insert_with_on_duplicate_key_update(
            self.get_table(),
            ['started_at'],
        ).values([{
            'uid': self.uid,
            'started_at': self.started_at,
        }])


class AccountDeletionOperationDeleteQuery(EavShardedQuery):
    def get_table(self):
        return account_deletion_operations_table

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)


class PassManInsertRecoveryKeyQuery(EavShardedQuery):
    def get_table(self):
        return passman_recovery_keys_table

    def __init__(self, uid, key_id, recovery_key):
        super(PassManInsertRecoveryKeyQuery, self).__init__(uid=uid)
        self.key_id = key_id
        self.recovery_key = recovery_key

    def to_query(self):
        return insert_with_on_duplicate_key_update(
            table=self.get_table(),
            keys=['recovery_key'],
            binary_value_ids=[1, 2],
        ).values([
            {
                'uid': self.uid,
                'key_id': self.key_id,
                'recovery_key': self.recovery_key,
            },
        ])


class PassManDeleteAllRecoveryKeysQuery(EavShardedQuery):
    def get_table(self):
        return passman_recovery_keys_table

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)


class InsertDevicePublicKeyQuery(DbQuery):
    def __init__(self, device_id, owner_id, public_key, version):
        self.device_id = device_id
        self.owner_id = owner_id
        self.public_key = public_key
        self.version = version

    def get_table(self):
        return device_public_key_table

    def query_params(self):
        return dict(
            device_id=smart_bytes(self.device_id),
            owner_id=self.owner_id,
            public_key=smart_bytes(self.public_key),
            version=self.version,
        )

    def to_query(self):
        return self.get_table().insert().values(self.query_params())


class UpdateDevicePublicKeyQuery(DbQuery):
    def __init__(self, device_id, owner_id, public_key, version):
        self.device_id = device_id
        self.owner_id = owner_id
        self.public_key = public_key
        self.version = version

    def get_table(self):
        return device_public_key_table

    def query_params(self):
        return dict(
            device_id=smart_bytes(self.device_id),
            owner_id=self.owner_id,
            public_key=smart_bytes(self.public_key),
            version=self.version,
        )

    def to_query(self):
        table = self.get_table()
        return (
            table.update()
            .where(
                and_(
                    table.c.device_id == smart_bytes(self.device_id),
                    table.c.owner_id == self.owner_id,
                ),
            )
            .values(
                public_key=smart_bytes(self.public_key),
                version=self.version,
            )
        )


class DeleteDevicePublicKeyQuery(DbQuery):
    def __init__(self, device_id, owner_id):
        self.device_id = device_id
        self.owner_id = owner_id

    def get_table(self):
        return device_public_key_table

    def query_params(self):
        return dict(
            device_id=smart_bytes(self.device_id),
            owner_id=self.owner_id,
        )

    def to_query(self):
        table = self.get_table()
        return (
            table.delete()
            .where(
                and_(
                    table.c.device_id == smart_bytes(self.device_id),
                    table.c.owner_id == self.owner_id,
                ),
            )
        )


class SelectDevicePublicKeyQuery(DbQuery):
    def __init__(self, device_id):
        self.device_id = device_id

    def get_table(self):
        return device_public_key_table

    def query_params(self):
        return dict(
            device_id=smart_bytes(self.device_id),
        )

    def to_query(self):
        table = self.get_table()
        return (
            table.select()
            .where(
                table.c.device_id == smart_bytes(self.device_id),
            )
        )

    def parse_query_result(self, query_result):
        return query_result.fetchone()


class DeleteMemberFromAllFamiliesQuery(DbQueryForUID):
    def get_table(self):
        return family_members_table

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.uid == self.uid)
