# -*- coding: utf-8 -*-
from itertools import chain

from passport.backend.core.db.runner import get_id_from_query_result
from passport.backend.core.eav_type_mapping import (
    attr_name_exists,
    SID_TO_SUBSCRIPTION_ATTR,
)
from passport.backend.core.serializers.eav.base import (
    EavAttributeMap,
    EavSerializer,
)
from passport.backend.core.serializers.eav.processors import as_is_processor
from passport.backend.core.serializers.eav.query import SUID_TABLES
from passport.backend.core.undefined import Undefined
from six import iteritems


def subscription_2_login_rule_processor(login_rule):
    if login_rule == 1:
        return
    return login_rule


def password_is_creating_required_processor(login_rule):
    if login_rule == 1:
        return str(login_rule)


SUBSCRIPTIONS_EAV_MAPPERS = {
    2: {
        'login_rule': EavAttributeMap('subscription.mail.login_rule', subscription_2_login_rule_processor),
    },
    27: {'login_rule': EavAttributeMap('subscription.jabber.login_rule', as_is_processor)},
    42: {'host.id': EavAttributeMap('subscription.wwwdgt.mode', as_is_processor)},
    44: {'login_rule': EavAttributeMap('subscription.disk.login_rule', as_is_processor)},
    89: {'login': EavAttributeMap('person.contact_phone_number', as_is_processor)},
    100: {'login_rule': EavAttributeMap('password.is_creating_required', password_is_creating_required_processor)},
}


SID_TO_ALIAS_NAME = {
    2: 'mail',
    16: 'narodmail',
}


NOT_SERIALIZABLE_SIDS = [61, 65, 669]


class SubscriptionEavSerializer(EavSerializer):
    '''
    Подписка:
        - записи в таблицу attributes
        - запись в suid* для 2

    Отписка:
        - чистим attributes
        - чистим suid*
        - удаляем aliases (TODO: вынести на account)
    Сейчас удаляются алиасы тут для 2 (не pdd) и 16

    INFO:
        В новой схеме многие сиды удалены или превращены в алиасы:
        - 8 сида больше не существует. Поэтому всё что связано с ним
    сериализуется в Account;
        - алиасы отдельная сущность от подписки и создаются в другом месте.
    '''

    def subscription_attribute_name(self, sid):
        return 'subscription.%s' % sid

    def serialize(self, old, new, difference):

        if (new is not None and new.sid in NOT_SERIALIZABLE_SIDS or
                old is not None and old.sid in NOT_SERIALIZABLE_SIDS):
            return

        queries = []
        create = True if (old is None and new) else False
        if create:
            if new.sid == 2 and not new.suid:
                yield (
                    self.build_increment_suid_query(new.parent.is_pdd),
                    lambda result: setattr(new, 'suid', get_id_from_query_result(result)),
                )
            queries.extend(self.create(new))
        elif old and new is None:  # удаляем подписку
            queries.extend(self.delete(old))
        else:  # изменяем подписку
            queries.extend(self.change(old, new, difference))
        for query in queries:
            yield query

    def create(self, new):
        sid = new.sid
        uid = new.parent.uid
        queries = []
        sub_attr_name = self.subscription_attribute_name(sid)
        if new.login_rule in (None, Undefined):
            new.login_rule = 1

        if sid in SUBSCRIPTIONS_EAV_MAPPERS:
            eav_mapper = SUBSCRIPTIONS_EAV_MAPPERS[sid]
            queries = self.build_change_fields_queries(
                eav_mapper,
                uid,
                None,
                new,
                eav_mapper.keys(),
                True,
            )
        # У этих подписок ничего не записывается в attributes
        elif attr_name_exists(sub_attr_name) or sid in SID_TO_SUBSCRIPTION_ATTR:
            queries.extend(self.build_set_attrs_with_cleaned_values(uid, {sub_attr_name: '1'}))

        if sid in SUID_TABLES:
            query = self.build_insert_suid_query(uid, sid, new.suid)
            queries.append(query)

        return queries

    def delete(self, subscription):
        sid = subscription.sid
        uid = subscription.parent.uid
        queries = []
        sub_attr_name = self.subscription_attribute_name(sid)

        # Удаляем алиасы только в случае подписок:
        # 1) narodmail
        # 2) mail и пользователь не pdd
        # TODO: Перенести логику по alias-ам в Account
        if sid == 16 or (sid == 2 and not subscription.parent.is_pdd):
            alias_name = SID_TO_ALIAS_NAME[sid]
            # Алиасы соответствующие подпискам 2, 4, 16 заполнены только тогда,
            # когда в них хранится "ненатуральный" алиас. Поэтому в
            # removed_aliases попадают только "ненатуральные" алиасы (там
            # внутри insert from select).
            removed_aliases_query = self.build_mass_insert_alias_into_removed_aliases(uid, alias_name)
            delete_alias_query = self.build_delete_alias_query(uid, alias_name)
            queries.extend([removed_aliases_query, delete_alias_query])

        if sid in SUBSCRIPTIONS_EAV_MAPPERS:
            delete_attrs_query = self.build_delete_fields_query(SUBSCRIPTIONS_EAV_MAPPERS[sid], uid)
            if delete_attrs_query:
                queries.append(delete_attrs_query)
        elif attr_name_exists(sub_attr_name) or sid in SID_TO_SUBSCRIPTION_ATTR:
            query = self.build_delete_attrs_by_names(uid, [sub_attr_name])
            queries.append(query)

        if sid in SUID_TABLES:
            query = self.build_delete_suid_query(uid, sid)
            queries.append(query)

        return queries

    def change(self, old, new, difference):
        sid = new.sid
        uid = new.parent.uid
        if sid not in SUBSCRIPTIONS_EAV_MAPPERS:
            return []

        eav_mapper = SUBSCRIPTIONS_EAV_MAPPERS[sid]
        changed_fields = []
        for k, v in chain(iteritems(difference.added),
                          iteritems(difference.changed),
                          iteritems(difference.deleted)):
            if k == 'host':
                if 'id' in v and 'host.id' in eav_mapper:
                    changed_fields.append('host.id')
            elif k in eav_mapper:
                changed_fields.append(k)

        return self.build_change_fields_queries(
            eav_mapper,
            uid,
            old,
            new,
            changed_fields,
        )
