# -*- coding: utf-8 -*-
from passport.backend.core.db.runner import get_id_from_query_result
from passport.backend.core.serializers.eav.base import (
    EavAttributeMap,
    EavSerializer,
)
from passport.backend.core.serializers.eav.processors import (
    datetime_processor,
    default_processor,
    JoinProcessors,
)


@JoinProcessors([default_processor])
def totp_secret_processor(totp_secret):
    return totp_secret.encrypted_pin_and_secret


@JoinProcessors([default_processor])
def yakey_device_ids_processor(yakey_device_ids):
    return ','.join(yakey_device_ids or [])


class TotpSecretEavSerializer(EavSerializer):
    EAV_FIELDS_MAPPER = {
        'failed_pin_checks_count': EavAttributeMap(
            'account.totp.failed_pin_checks_count',
            default_processor,
        ),
        'check_time': EavAttributeMap(
            'account.totp.check_time',
            default_processor,
        ),
        'totp_secret': EavAttributeMap(
            'account.totp.secret',
            totp_secret_processor,
        ),
        'update_datetime': EavAttributeMap(
            'account.totp.update_datetime',
            datetime_processor,
        ),
        'yakey_device_ids': EavAttributeMap(
            'account.totp.yakey_device_ids',
            yakey_device_ids_processor,
        ),
    }

    def serialize(self, old, new, difference):
        queries = []
        if old and new is None:
            queries.extend(self.delete(old))
        else:
            queries.extend(self.change(old, new, difference))
        return queries

    def change(self, old, new, difference):
        serializable_fields = difference.get_changed_fields(self.EAV_FIELDS_MAPPER)
        attributes_to_increment = None
        if (
            old and old.failed_pin_checks_count and
            new and new.failed_pin_checks_count and
            new.failed_pin_checks_count - old.failed_pin_checks_count == 1
        ):
            attributes_to_increment = [self.EAV_FIELDS_MAPPER['failed_pin_checks_count'].attr_name]
        return self.build_change_fields_queries(
            self.EAV_FIELDS_MAPPER,
            new.parent.uid,
            old,
            new,
            serializable_fields,
            old is None,
            attributes_to_increment=attributes_to_increment,
        )

    def delete(self, totp_secret):
        return [
            self.build_delete_fields_query(
                self.EAV_FIELDS_MAPPER,
                totp_secret.parent.uid,
            ),
        ]


class TotpSecretAutoIdEavSerializer(EavSerializer):
    def serialize(self, old, new, difference):
        if not old and new:
            return self.create(new)
        else:
            raise ValueError('AutoId can only be created')

    def create(self, new):
        return [
            (
                self.build_increment_totp_secret_id_query(),
                lambda result: setattr(new, 'id', get_id_from_query_result(result)),
            ),
        ]
